// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.DoublePair;
import de.caff.generics.Primitives;
import de.caff.generics.Types;

/**
 * Predicate which expects two primitive double arguments.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since October 05, 2020
 */
@FunctionalInterface
public interface DoublePredicate2
        extends Predicate2<Double, Double>
{
  /**
   * Test the given value.
   * @param value1 first value to test
   * @param value2 second value to test
   * @return {@code true} if the values match the predicate, {@code false} otherwise
   */
  boolean testDoubles(double value1, double value2);

  /**
   * Default implementation.
   * This forwards to {@link #testDoubles(double, double)}.
   * {@link Double#NaN} will be used instead of {@code null}.
   * @param value1 first value
   * @param value2 second value
   * @return {@code true} if the values match the predicate, {@code false} otherwise
   */
  @Override
  default boolean test(Double value1, Double value2)
  {
    return testDoubles(Types.notNull(value1, Double.NaN),
                       Types.notNull(value2, Double.NaN));
  }

  /**
   * Test a double pair by expanding its elements.
   * @param pair pair to test
   * @return result of the test
   * @throws NullPointerException if pair is {@code null}
   */
  default boolean test(@NotNull DoublePair pair)
  {
    return pair.test(this);
  }

  /**
   * Predicate which checks whether two double values are equal
   * which handles NaN values in a way that {@code Double.NaN == Double.NaN}
   * is {@code true}. It also handles {@code -0.0 == 0.0} in a better way
   * than standard Java using {@link Double#compare(double, double)} which considers
   * them different, although above equality is {@code true}.
   * @see #EQUAL_STANDARD
   * @see Primitives#areEqual(double, double)
   */
  @NotNull
  DoublePredicate2 EQUAL_EVEN_NAN = Primitives::areEqual;

  /**
   * Predicate which checks whether two float values are equal
   * which handles NaN as defined by IEEE 745 so that {@code Float.NaN == Float.NaN}
   * is {@code false}, i.e. NaN is different from any other value including
   * itself.
   * @see #EQUAL_EVEN_NAN
   */
  @NotNull
  DoublePredicate2 EQUAL_STANDARD = (v1, v2) -> v1 == v2;
}
