// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright ©2023-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.Order;
import de.caff.generics.Primitives;

import java.io.Serializable;
import java.util.Comparator;

/**
 * Define an order for two primitive double values.
 * <p>
 * Compared to a {@code Comparator<Double>} or {@code Ordering<Double>} this
 * interface allows to avoid boxing and unboxing in various situations.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 11, 2023
 */
@FunctionalInterface
public interface DoubleOrdering
        extends FloatOrdering
{
  /**
   * Natural order of ascending {@code double} values.
   * Besides collapsing all NaN values and ordering them between
   * everything else this ordering is nearer to IEEE 784 than
   * standard {@link Double#compare(double, double)}.
   *
   * @see Primitives#order(double, double)
   * @see #STANDARD_ASCENDING
   */
  DoubleOrdering NATURAL_ASCENDING = new Serial()
  {
    private static final long serialVersionUID = 3025885466446256167L;

    @NotNull
    @Override
    public Order checkDouble(double v1, double v2)
    {
      return Primitives.order(v1, v2);
    }

    @NotNull
    @Override
    public Comparator<Double> asDoubleComparator()
    {
      return (Comparator<Double> & Serializable)Primitives::compare;
    }

    @NotNull
    @Override
    public DoubleOrdering inverse()
    {
      return NATURAL_DESCENDING;
    }
  };

  /**
   * Natural order of descending {@code double} values.
   * Besides collapsing all NaN values and ordering them before
   * everything else this ordering is nearer to IEEE 784 than
   * standard {@link Double#compare(double, double)}.
   * @see Primitives#order(double, double)
   * @see #STANDARD_DESCENDING
   */
  DoubleOrdering NATURAL_DESCENDING = new Serial()
  {
    private static final long serialVersionUID = 3025885466446256167L;

    @NotNull
    @Override
    public Order checkDouble(double v1, double v2)
    {
      return Primitives.order(v2, v1);
    }

    @NotNull
    @Override
    public Comparator<Double> asDoubleComparator()
    {
      return (Comparator<Double> & Serializable)(v1, v2) -> Primitives.compare(v2, v1);
    }

    @NotNull
    @Override
    public DoubleOrdering inverse()
    {
      return NATURAL_ASCENDING;
    }
  };

  /**
   * Standard Java order of ascending {@code double} values.
   * Please note that this order is broken as it considers
   * {@code -0.0 < 0.0} which is not correct, for primitive
   * {@code double} values {@code -0.0 == 0.0} is valid.
   * @see #NATURAL_ASCENDING
   */
  DoubleOrdering STANDARD_ASCENDING = new DoubleOrdering.Serial()
  {
    private static final long serialVersionUID = 3025885466446256167L;

    @NotNull
    @Override
    public Order checkDouble(double v1, double v2)
    {
      if (v1 < v2) {
        return Order.Ascending;
      }
      if (v1 > v2) {
        return Order.Descending;
      }

      final long bits1 = Double.doubleToLongBits(v1);
      final long bits2 = Double.doubleToLongBits(v2);

      return bits1 == bits2
              ? Order.Same
              : (bits1 < bits2
                         ? Order.Ascending
                         : Order.Descending);

    }

    @NotNull
    @Override
    public DoubleOrdering inverse()
    {
      return DoubleOrdering.STANDARD_DESCENDING;
    }

    @NotNull
    @Override
    public Comparator<Double> asDoubleComparator()
    {
      return Double::compare;
    }
  };

  /**
   * Inverse standard Java order of {@code double} values.
   * Please note that this order is broken as it considers
   * {@code -0.0 < 0.0} which is not correct, for primitive
   * {@code double} values {@code -0.0 == 0.0} is valid.
   */
  DoubleOrdering STANDARD_DESCENDING = new DoubleOrdering.Serial()
  {
    private static final long serialVersionUID = 8490254886442973329L;

    @NotNull
    @Override
    public Order checkDouble(double v2, double v1)
    {  // code copied for efficiency
      if (v1 < v2) {
        return Order.Ascending;
      }
      if (v1 > v2) {
        return Order.Descending;
      }

      final long bits1 = Double.doubleToLongBits(v1);
      final long bits2 = Double.doubleToLongBits(v2);

      return bits1 == bits2
              ? Order.Same
              : (bits1 < bits2
                         ? Order.Ascending
                         : Order.Descending);
    }

    @NotNull
    @Override
    public DoubleOrdering inverse()
    {
      return DoubleOrdering.STANDARD_ASCENDING;
    }

    @NotNull
    @Override
    public Comparator<Double> asDoubleComparator()
    {
      return (v1, v2) -> Double.compare(v2, v1);
    }
  };

  /**
   * Get the order of the two values.
   * @param v1 first value
   * @param v2 second value
   * @return order of the two values
   */
  @NotNull
  Order checkDouble(double v1, double v2);

  @NotNull
  @Override
  default Order checkFloat(float v1, float v2)
  {
    return checkDouble(v1, v2);
  }

  /**
   * See this ordering as one working on general numbers.
   * @return this ordering but working on any number
   */
  @NotNull
  default Ordering<Number> asNumberOrdering()
  {
    return (Ordering.Serial<Number>) (v1, v2) -> DoubleOrdering.this.checkDouble(v1.doubleValue(),
                                                                                 v2.doubleValue());
  }

  /**
   * Are the two values represented in strictly ascending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 < v2} according to this ordering<br>
   *         {@code false} if {@code v1 >= v2} according to this ordering
   * @see #descending(double, double)
   * @see #ascendingOrSame(double, double)
   * @see #descendingOrSame(double, double)
   * @see #same(double, double)
   * @see #different(double, double)
   */
  default boolean ascending(double v1, double v2)
  {
    return checkDouble(v1, v2).ascending;
  }

  /**
   * Are the two values represented in ascending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 <= v2} according to this ordering<br>
   *         {@code false} if {@code v1 > v2} according to this ordering
   * @see #ascending(double, double)
   * @see #descending(double, double)
   * @see #descendingOrSame(double, double)
   * @see #same(double, double)
   * @see #different(double, double)
   */
  default boolean ascendingOrSame(double v1, double v2)
  {
    return checkDouble(v1, v2).ascendingOrSame;
  }

  /**
   * Are the two values represented in strictly descending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 > v2} according to this ordering<br>
   *         {@code false} if {@code v1 <= v2} according to this ordering
   * @see #ascending(double, double)
   * @see #ascendingOrSame(double, double)
   * @see #descendingOrSame(double, double)
   * @see #same(double, double)
   * @see #different(double, double)
   */
  default boolean descending(double v1, double v2)
  {
    return checkDouble(v1, v2).descending;
  }

  /**
   * Are the two values represented in descending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 >= v2} according to this ordering<br>
   *         {@code false} if {@code v1 < v2} according to this ordering
   * @see #ascending(double, double)
   * @see #descending(double, double)
   * @see #ascendingOrSame(double, double)
   * @see #same(double, double)
   * @see #different(double, double)
   */
  default boolean descendingOrSame(double v1, double v2)
  {
    return checkDouble(v1, v2).descendingOrSame;
  }

  /**
   * Are the two values considered equal by this order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 == v2} according to this order<br>
   *         {@code false} if {@code v1 != v2} according to this ordering
   * @see #ascending(double, double)
   * @see #descending(double, double)
   * @see #ascendingOrSame(double, double)
   * @see #descendingOrSame(double, double)
   * @see #different(double, double)
   */
  default boolean same(double v1, double v2)
  {
    return checkDouble(v1, v2).same;
  }

  /**
   * Are the two values considered different by this order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 != v2} according to this order<br>
   *         {@code false} if {@code v1 == v2} according to this ordering
   * @see #ascending(double, double)
   * @see #descending(double, double)
   * @see #ascendingOrSame(double, double)
   * @see #descendingOrSame(double, double)
   * @see #same(double, double)
   */
  default boolean different(double v1, double v2)
  {
    return checkDouble(v1, v2).different;
  }

  /**
   * Invert this order.
   * @return inverse order
   */
  @NotNull
  default DoubleOrdering inverse()
  {
    return new DoubleOrdering()
    {
      @Override
      @NotNull
      public Order checkDouble(double v1, double v2)
      {
        return DoubleOrdering.this.checkDouble(v2, v1);
      }

      @Override
      @NotNull
      public DoubleOrdering inverse()
      {
        return DoubleOrdering.this;
      }
    };
  }

  /**
   * Use this ordering as a comparator.
   * @return comparator of {@code Double}
   */
  @NotNull
  default Comparator<Double> asDoubleComparator()
  {
    return (Comparator<Double> & Serializable)(v1, v2) -> checkDouble(v1, v2).comparison;
  }

  /**
   * View this ordering as a number comparator.
   * @return comparator of {@link Number}
   */
  @NotNull
  default Comparator<Number> asNumberComparator()
  {
    return (Comparator<Number> & Serializable)(v1, v2) -> checkDouble(v1.doubleValue(), v2.doubleValue()).comparison;
  }

  /**
   * Get this ordering as a generic double ordering. 
   * @return generic ordering for doubles
   */
  @NotNull
  default Ordering<Double> asDoubleOrdering()
  {
    return (Ordering<Double> & Serializable)this::checkDouble;
  }

  /**
   * Convert a standard comparator of {@code Double} into a double ordering.
   * @param comparator comparator of {@code Double}
   * @return ordering
   */
  @NotNull
  static DoubleOrdering fromComparator(@NotNull Comparator<Double> comparator)
  {
    return new DoubleOrdering.Serial()
    {
      private static final long serialVersionUID = -7834494744673376888L;

      @NotNull
      @Override
      public Order checkDouble(double v1, double v2)
      {
        return Order.fromCompare(comparator.compare(v1, v2));
      }

      @NotNull
      @Override
      public Comparator<Double> asDoubleComparator()
      {
        return comparator;
      }
    };
  }

  /**
   * A double ordering which is serializable.
   * It is a recommendation that comparators implement {@code java.io.Serializable}
   * to allow serializing {@code TreeMap}s and similar. If an Ordering is expected to be used
   * as a comparator in such areas implement either this interface, or for lambdas
   * use a cast to force the compiler to make the order serializable:
   * <pre>{@code
   *   return (ByteOrdering & Serializable) (v1, v2) -> v1.compareTo(v2);
   * }</pre>
   * You can also use a {@code (ByteOrdering.Serial)} cast to save a bit of typing,
   * but the above is a nice trick to have in your toolbox.
   */
  interface Serial extends DoubleOrdering,
                           java.io.Serializable
  {
  }
}
