// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2019-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;

import java.util.function.DoubleBinaryOperator;

/**
 * Binary operator with two double arguments and a double result.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since October 31, 2019
 */
@FunctionalInterface
public interface DoubleOperator2
        extends DoubleBinaryOperator
{
  /**
   * Double operator which returns the sum of two arguments:
   * {@code a + b}.
   */
  DoubleOperator2 PLUS = (a, b) -> a + b;
  /**
   * Double operator which returns the difference of two arguments:
   * {@code a - b}
   */
  DoubleOperator2 MINUS = (a, b) -> a - b;
  /**
   * Double operator which returns the product of two arguments:
   * {@code a * b}
   */
  DoubleOperator2 MULT = (a, b) -> a * b;
  /**
   * Double operator which returns the division result of two arguments:
   * {@code a / b}
   */
  DoubleOperator2 DIV = (a, b) -> a / b;
  /**
   * Double operator which returns the modulo result of two arguments:
   * {@code a % b}.
   */
  DoubleOperator2 MOD = (a, b) -> a % b;
  /**
   * Double operator which returns the first argument to the power given by the second argument:
   * {@code a ^ b}
   */
  DoubleOperator2 POWER = Math::pow;

  /**
   * Get a partially applied operator.
   * @param argument value applied for first argument
   * @return partially evaluated operator
   */
  @NotNull
  default DoubleOperator1 partialLeft(double argument)
  {
    return arg -> applyAsDouble(argument, arg);
  }

  /**
   * Get a partially applied operator.
   * @param argument value applied for second argument
   * @return partially evaluated operator
   */
  @NotNull
  default DoubleOperator1 partialRight(double argument)
  {
    return arg -> applyAsDouble(arg, argument);
  }

  /**
   * Create a operator which applies this operator and feeds the result into the next operator.
   * @param after operator called with the result of this operator
   * @return chained operator
   */
  @NotNull
  default DoubleOperator2 andThen(@NotNull DoubleOperator1 after)
  {
    return (arg1, arg2) -> after.applyAsDouble(applyAsDouble(arg1, arg2));
  }

  /**
   * Use this operator on the results of two other operators.
   * This will use the first argument handler on the first argument,
   * the second argument handler on the second argument, and
   * this operator on their results.
   * <p>
   * Example:
   * {@code combine(DoubleFunction1.SQUARE, DoubleFunction11.SQUARE).andThen(DoubleFunction1.SQUARE_ROOT)}
   * creates a operator which calculates the length of a 2D vector.
   * @param firstArgHandler  handler for first argument
   * @param secondArgHandler handler for second argument
   * @return resulting operator
   */
  @NotNull
  default DoubleOperator2 combine(@NotNull DoubleOperator1 firstArgHandler,
                                  @NotNull DoubleOperator1 secondArgHandler)
  {
    return (arg1, arg2) -> applyAsDouble(firstArgHandler.applyAsDouble(arg1),
                                         secondArgHandler.applyAsDouble(arg2)) ;
  }

  /**
   * Make a standard unary double operator usable as {@code DoubleOperator2}.
   * This is useful if you want to make use of the extended functionality of this implementation.
   * @param operator operator to be used as a {@code DoubleOperator2}
   * @return {@code DoubleOperator2} view of the incoming operator, or possibly just a cast if
   *         {@code operator} already has the correct type
   */
  @NotNull
  static DoubleOperator2 from(@NotNull DoubleBinaryOperator operator)
  {
    return operator instanceof DoubleOperator2
            ? (DoubleOperator2) operator
            : operator::applyAsDouble;
  }
}
