// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2019-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;

import java.util.function.DoubleUnaryOperator;

/**
 * Unary operator with a double argument and result.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since October 31, 2019
 */
@FunctionalInterface
public interface DoubleOperator1
        extends DoubleUnaryOperator
{
  /** Standard NEGATE operator ({@code -}). */
  DoubleOperator1 NEGATE = x -> -x;

  /**
   * Identity function.
   * This returns the argument.
   */
  DoubleOperator1 IDENT = a -> a;

  /**
   * Square function.
   * This returns the square of the argument.
   */
  DoubleOperator1 SQUARE = a -> a * a;
  /**
   * Cubic function.
   * This returns the third power of the argument.
   */
  DoubleOperator1 CUBIC = a -> a * a * a;

  /**
   * Square root function.
   * This returns the square root of the argument.
   */
  DoubleOperator1 SQUARE_ROOT = Math::sqrt;
  /**
   * Cubic root function.
   * This returns the cubic root of the argument.
   */
  DoubleOperator1 CUBIC_ROOT = Math::cbrt;

  /**
   * Create a zero argument function from this one by applying a constant argument.
   * Although useless for pure functions it can be useful in cases where the zero
   * argument function is a factory (and therefore not pure).
   * @param argument constant argument
   * @return zero argument function
   */
  @NotNull
  default DoubleOperator0 partial(double argument)
  {
    return () -> applyAsDouble(argument);
  }

  /**
   * Create a function which applies this function and feeds the result into the next function.
   * @param after function called with the result of this function
   * @return chained function
   */
  @NotNull
  default DoubleOperator1 andThen(@NotNull DoubleOperator1 after)
  {
    return (arg) -> after.applyAsDouble(applyAsDouble(arg));
  }

  /**
   * Make a standard unary double operator usable as {@code DoubleOperator1}.
   * This is useful if you want to make use of the extended functionality of this implementation.
   * @param operator operator to be used as a {@code DoubleOperator1}
   * @return {@code DoubleOperator1} view of the incoming operator, or possibly just a cast if
   *         {@code operator} already has the correct type
   */
  @NotNull
  static DoubleOperator1 from(@NotNull DoubleUnaryOperator operator)
  {
    return operator instanceof DoubleOperator1
            ? (DoubleOperator1) operator
            : operator::applyAsDouble;
  }
}
