// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2021-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;

import java.util.function.Function;

/**
 * Function with 3 primitive double arguments creation a general result.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since April 26, 2021
 * @param <T> result type
 */
@FunctionalInterface
public interface DoubleFunction3<T>
{
  /**
   * Apply this function to three double arguments and return the result.
   * @param v1 first argument value
   * @param v2 second argument value
   * @param v3 third argument value
   * @return result value
   */
  T apply(double v1, double v2, double v3);

  /**
   * Get a partially applied function where the first argument is fix.
   * @param argument value applied for first argument
   * @return partially evaluated function
   */
  @NotNull
  default DoubleFunction2<T> partial1(double argument)
  {
    return (arg2, arg3) -> apply(argument, arg2, arg3);
  }

  /**
   * Get a partially applied function where the second argument is fix.
   * @param argument value applied for second argument
   * @return partially evaluated function
   */
  @NotNull
  default DoubleFunction2<T> partial2(double argument)
  {
    return (arg1, arg3) -> apply(arg1, argument, arg3);
  }

  /**
   * Get a partially applied function where the third argument is fix.
   * @param argument value applied for third argument
   * @return partially evaluated function
   */
  @NotNull
  default DoubleFunction2<T> partial3(double argument)
  {
    return (arg1, arg2) -> apply(arg1, arg2, argument);
  }

  /**
   * Get a chained function which first applies this function
   * and then the given function to the returned value.
   * @param after function to apply after this one
   * @param <R> result type of returned function
   * @return function chain
   */
  @NotNull
  default <R> DoubleFunction3<R> andThen(@NotNull Function<? super T, ? extends R> after)
  {
    return (v1, v2, v3) -> after.apply(apply(v1, v2, v3));
  }
}
