// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.CharPair;

/**
 * Predicate which expects two primitive char arguments.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since October 05, 2020
 */
@FunctionalInterface
public interface CharPredicate2
        extends Predicate2<Character, Character>
{
  /**
   * Test the given value.
   * @param value1 first value to test
   * @param value2 second value to test
   * @return {@code true} if the values match the predicate, {@code false} otherwise
   */
  boolean testChars(char value1, char value2);

  /**
   * Default implementation.
   * This forward toe {@link #testChars(char, char)}
   * @param value1 first value
   * @param value2 second value
   * @return {@code true} if the values match the predicate, {@code false} otherwise
   * @throws NullPointerException if either value is {@code null}
   */
  @Override
  default boolean test(Character value1, Character value2)
  {
    return testChars(value1, value2);
  }

  /**
   * Test a char pair by expanding its elements.
   * @param pair pair to test
   * @return result of the test
   * @throws NullPointerException if pair is {@code null}
   */
  default boolean test(@NotNull CharPair pair)
  {
    return pair.test(this);
  }
}
