// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright ©2023-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.Order;

import java.io.Serializable;
import java.util.Comparator;

/**
 * Define an order for two primitive char values.
 * <p>
 * Compared to a  {@code Comparator<Char>} this can
 * avoid boxing and unboxing in various situations.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 23, 2023
 */
@FunctionalInterface
public interface CharOrdering
{
  /**
   * Natural order of ascending {@code char} values.
   */
  CharOrdering ASCENDING = new CharOrdering.Serial()
  {
    private static final long serialVersionUID = -7646031177018732313L;

    @NotNull
    @Override
    public Order checkChar(char v1, char v2)
    {
      return (v1 < v2)
              ? Order.Ascending
              : (v1 > v2
                         ? Order.Descending
                         : Order.Same);
    }

    @NotNull
    @Override
    public CharOrdering inverse()
    {
      return CharOrdering.DESCENDING;
    }

    @NotNull
    @Override
    public Comparator<Character> asCharComparator()
    {
      return Character::compare;
    }
  };
  /** Inverse natural order of {@code char} values. */
  CharOrdering DESCENDING = new CharOrdering.Serial()
  {
    private static final long serialVersionUID = 3026851066329505442L;

    @NotNull
    @Override
    public Order checkChar(char v2, char v1)
    {
      return (v1 < v2)
              ? Order.Ascending
              : (v1 > v2
                         ? Order.Descending
                         : Order.Same);
    }

    @NotNull
    @Override
    public CharOrdering inverse()
    {
      return CharOrdering.ASCENDING;
    }

    @NotNull
    @Override
    public Comparator<Character> asCharComparator()
    {
      return (Comparator<Character> & Serializable) (v1, v2) -> Character.compare(v2, v1);
    }
  };

  /**
   * Get the order of the two values.
   * @param v1 first value
   * @param v2 second value
   * @return order of the two values
   */
  @NotNull
  Order checkChar(char v1, char v2);

  /**
   * Are the two values represented in strictly ascending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 < v2} according to this ordering<br>
   *         {@code false} if {@code v1 >= v2} according to this ordering
   * @see #descending(char, char)
   * @see #ascendingOrSame(char, char)
   * @see #descendingOrSame(char, char)
   * @see #same(char, char)
   * @see #different(char, char)
   */
  default boolean ascending(char v1, char v2)
  {
    return checkChar(v1, v2).ascending;
  }

  /**
   * Are the two values represented in ascending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 <= v2} according to this ordering<br>
   *         {@code false} if {@code v1 > v2} according to this ordering
   * @see #ascending(char, char)
   * @see #descending(char, char)
   * @see #descendingOrSame(char, char)
   * @see #same(char, char)
   * @see #different(char, char)
   */
  default boolean ascendingOrSame(char v1, char v2)
  {
    return checkChar(v1, v2).ascendingOrSame;
  }

  /**
   * Are the two values represented in strictly descending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 > v2} according to this ordering<br>
   *         {@code false} if {@code v1 <= v2} according to this ordering
   * @see #ascending(char, char)
   * @see #ascendingOrSame(char, char)
   * @see #descendingOrSame(char, char)
   * @see #same(char, char)
   * @see #different(char, char)
   */
  default boolean descending(char v1, char v2)
  {
    return checkChar(v1, v2).descending;
  }

  /**
   * Are the two values represented in descending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 >= v2} according to this ordering<br>
   *         {@code false} if {@code v1 < v2} according to this ordering
   * @see #ascending(char, char)
   * @see #descending(char, char)
   * @see #ascendingOrSame(char, char)
   * @see #same(char, char)
   * @see #different(char, char)
   */
  default boolean descendingOrSame(char v1, char v2)
  {
    return checkChar(v1, v2).descendingOrSame;
  }

  /**
   * Are the two values considered equal by this order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 == v2} according to this order<br>
   *         {@code false} if {@code v1 != v2} according to this ordering
   * @see #ascending(char, char)
   * @see #descending(char, char)
   * @see #ascendingOrSame(char, char)
   * @see #descendingOrSame(char, char)
   * @see #different(char, char)
   */
  default boolean same(char v1, char v2)
  {
    return checkChar(v1, v2).same;
  }

  /**
   * Are the two values considered different by this order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 != v2} according to this order<br>
   *         {@code false} if {@code v1 == v2} according to this ordering
   * @see #ascending(char, char)
   * @see #descending(char, char)
   * @see #ascendingOrSame(char, char)
   * @see #descendingOrSame(char, char)
   * @see #same(char, char)
   */
  default boolean different(char v1, char v2)
  {
    return checkChar(v1, v2).different;
  }

  /**
   * Invert this order.
   * @return inverse order
   */
  @NotNull
  default CharOrdering inverse()
  {
    return new CharOrdering.Serial()
    {
      private static final long serialVersionUID = 3270504639153727362L;

      @Override
      @NotNull
      public Order checkChar(char v1, char v2)
      {
        return CharOrdering.this.checkChar(v2, v1);
      }

      @NotNull
      @Override
      public CharOrdering inverse()
      {
        return CharOrdering.this;
      }
    };
  }

  /**
   * Use this ordering as a comparator.
   * @return comparator of {@code Char}
   */
  @NotNull
  default Comparator<Character> asCharComparator()
  {
    return (Comparator<Character> & Serializable) (v1, v2) -> checkChar(v1, v2).comparison;
  }

  /**
   * Get this ordering as a generic char ordering. 
   * @return generic ordering for chars
   */
  @NotNull
  default Ordering<Character> asCharOrdering()
  {
    return (Ordering<Character> & Serializable)this::checkChar;
  }

  /**
   * Convert a standard comparator of {@code Double} into a double ordering.
   * @param comparator comparator of {@code Double}
   * @return ordering
   */
  @NotNull
  static CharOrdering fromComparator(@NotNull Comparator<Character> comparator)
  {
    return new CharOrdering.Serial()
    {
      private static final long serialVersionUID = 393576352771473074L;

      @NotNull
      @Override
      public Order checkChar(char v1, char v2)
      {
        return Order.fromCompare(comparator.compare(v1, v2));
      }

      @NotNull
      @Override
      public Comparator<Character> asCharComparator()
      {
        return comparator;
      }
    };
  }

  /**
   * A char ordering which is serializable.
   * It is a recommendation that comparators implement {@code java.io.Serializable}
   * to allow serializing {@code TreeMap}s and similar. If an Ordering is expected to be used
   * as a comparator in such areas implement either this interface, or for lambdas
   * use a cast to force the compiler to make the order serializable:
   * <pre>{@code
   *   return (CharOrdering & Serializable) (v1, v2) -> v1.compareTo(v2);
   * }</pre>
   * You can also use a {@code (CharOrdering.Serial)} cast to save a bit of typing,
   * but the above is a nice trick to have in your toolbox.
   */
  interface Serial extends CharOrdering,
                           java.io.Serializable
  {
  }
}
