// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright ©2023-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.Order;

import java.io.Serializable;
import java.util.Comparator;

/**
 * Define an order for two primitive byte values.
 * <p>
 * Compared to a  {@code Comparator<Byte>} this can
 * avoid boxing and unboxing in various situations.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 23, 2023
 */
@FunctionalInterface
public interface ByteOrdering
{
  /**
   * Natural order of ascending {@code byte} values.
   */
  ByteOrdering ASCENDING = new ByteOrdering.Serial()
  {
    private static final long serialVersionUID = -7646031177018732313L;

    @NotNull
    @Override
    public Order checkByte(byte v1, byte v2)
    {
      return (v1 < v2)
              ? Order.Ascending
              : (v1 > v2
                         ? Order.Descending
                         : Order.Same);
    }

    @NotNull
    @Override
    public ByteOrdering inverse()
    {
      return ByteOrdering.DESCENDING;
    }

    @NotNull
    @Override
    public Comparator<Byte> asByteComparator()
    {
      return Byte::compare;
    }
  };
  /** Inverse natural order of {@code byte} values. */
  ByteOrdering DESCENDING = new ByteOrdering.Serial()
  {
    private static final long serialVersionUID = 3026851066329505442L;

    @NotNull
    @Override
    public Order checkByte(byte v2, byte v1)
    {
      return (v1 < v2)
              ? Order.Ascending
              : (v1 > v2
                         ? Order.Descending
                         : Order.Same);
    }

    @NotNull
    @Override
    public ByteOrdering inverse()
    {
      return ByteOrdering.ASCENDING;
    }

    @NotNull
    @Override
    public Comparator<Byte> asByteComparator()
    {
      return (Comparator<Byte> & Serializable) (v1, v2) -> Byte.compare(v2, v1);
    }
  };

  /**
   * Natural order of ascending unsigned {@code byte} values.
   * This handles byte values as if they are unsigned and not signed.
   */
  ByteOrdering UNSIGNED_ASCENDING = new ByteOrdering.Serial()
  {
    private static final long serialVersionUID = -3495520630473840505L;

    @NotNull
    @Override
    public Order checkByte(byte v1, byte v2)
    {
      return IntOrdering.ASCENDING.checkInt(v1 & 0xFF, v2 & 0xFF);
    }

    @NotNull
    @Override
    public ByteOrdering inverse()
    {
      return ByteOrdering.UNSIGNED_DESCENDING;
    }

    @NotNull
    @Override
    public Comparator<Byte> asByteComparator()
    {
      return Comparator.comparingInt(v -> v & 0xFF);
    }
  };

  /**
   * Natural order of ascending unsigned {@code byte} values.
   * This handles byte values as if they are unsigned and not signed.
   */
  ByteOrdering UNSIGNED_DESCENDING = new ByteOrdering.Serial()
  {
    private static final long serialVersionUID = 8393916740777317119L;

    @NotNull
    @Override
    public Order checkByte(byte v1, byte v2)
    {
      return IntOrdering.DESCENDING.checkInt(v1 & 0xFF, v2 & 0xFF);
    }

    @NotNull
    @Override
    public ByteOrdering inverse()
    {
      return ByteOrdering.UNSIGNED_ASCENDING;
    }

    @NotNull
    @Override
    public Comparator<Byte> asByteComparator()
    {
      return (Comparator<Byte> & Serializable) (v1, v2) -> Integer.compareUnsigned(v2, v1);
    }
  };

  /**
   * Get the order of the two values.
   * @param v1 first value
   * @param v2 second value
   * @return order of the two values
   */
  @NotNull
  Order checkByte(byte v1, byte v2);

  /**
   * Are the two values represented in strictly ascending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 < v2} according to this ordering<br>
   *         {@code false} if {@code v1 >= v2} according to this ordering
   * @see #descending(byte, byte)
   * @see #ascendingOrSame(byte, byte)
   * @see #descendingOrSame(byte, byte)
   * @see #same(byte, byte)
   * @see #different(byte, byte)
   */
  default boolean ascending(byte v1, byte v2)
  {
    return checkByte(v1, v2).ascending;
  }

  /**
   * Are the two values represented in ascending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 <= v2} according to this ordering<br>
   *         {@code false} if {@code v1 > v2} according to this ordering
   * @see #ascending(byte, byte)
   * @see #descending(byte, byte)
   * @see #descendingOrSame(byte, byte)
   * @see #same(byte, byte)
   * @see #different(byte, byte)
   */
  default boolean ascendingOrSame(byte v1, byte v2)
  {
    return checkByte(v1, v2).ascendingOrSame;
  }

  /**
   * Are the two values represented in strictly descending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 > v2} according to this ordering<br>
   *         {@code false} if {@code v1 <= v2} according to this ordering
   * @see #ascending(byte, byte)
   * @see #ascendingOrSame(byte, byte)
   * @see #descendingOrSame(byte, byte)
   * @see #same(byte, byte)
   * @see #different(byte, byte)
   */
  default boolean descending(byte v1, byte v2)
  {
    return checkByte(v1, v2).descending;
  }

  /**
   * Are the two values represented in descending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 >= v2} according to this ordering<br>
   *         {@code false} if {@code v1 < v2} according to this ordering
   * @see #ascending(byte, byte)
   * @see #descending(byte, byte)
   * @see #ascendingOrSame(byte, byte)
   * @see #same(byte, byte)
   * @see #different(byte, byte)
   */
  default boolean descendingOrSame(byte v1, byte v2)
  {
    return checkByte(v1, v2).descendingOrSame;
  }

  /**
   * Are the two values considered equal by this order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 == v2} according to this order<br>
   *         {@code false} if {@code v1 != v2} according to this ordering
   * @see #ascending(byte, byte)
   * @see #descending(byte, byte)
   * @see #ascendingOrSame(byte, byte)
   * @see #descendingOrSame(byte, byte)
   * @see #different(byte, byte)
   */
  default boolean same(byte v1, byte v2)
  {
    return checkByte(v1, v2).same;
  }

  /**
   * Are the two values considered different by this order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 != v2} according to this order<br>
   *         {@code false} if {@code v1 == v2} according to this ordering
   * @see #ascending(byte, byte)
   * @see #descending(byte, byte)
   * @see #ascendingOrSame(byte, byte)
   * @see #descendingOrSame(byte, byte)
   * @see #same(byte, byte)
   */
  default boolean different(byte v1, byte v2)
  {
    return checkByte(v1, v2).different;
  }

  /**
   * Invert this order.
   * @return inverse order
   */
  @NotNull
  default ByteOrdering inverse()
  {
    return new ByteOrdering.Serial()
    {
      private static final long serialVersionUID = 3270504639153727362L;

      @Override
      @NotNull
      public Order checkByte(byte v1, byte v2)
      {
        return ByteOrdering.this.checkByte(v2, v1);
      }

      @NotNull
      @Override
      public ByteOrdering inverse()
      {
        return ByteOrdering.this;
      }
    };
  }

  /**
   * Use this ordering as a comparator.
   * @return comparator of {@code Byte}
   */
  @NotNull
  default Comparator<Byte> asByteComparator()
  {
    return (Comparator<Byte> & Serializable) (v1, v2) -> checkByte(v1, v2).comparison;
  }

  /**
   * Get this ordering as a number comparator.
   * @return comparator of {@link Number}
   */
  @NotNull
  default Comparator<Number> asNumberComparator()
  {
    return (Comparator<Number> & Serializable)(v1, v2) -> checkByte(v1.byteValue(), v2.byteValue()).comparison;
  }

  /**
   * Get this ordering as a generic byte ordering.
   * @return generic ordering for bytes
   */
  @NotNull
  default Ordering<Byte> asByteOrdering()
  {
    return (Ordering<Byte> & Serializable)this::checkByte;
  }

  /**
   * Convert a standard comparator of {@code Double} into a double ordering.
   * @param comparator comparator of {@code Double}
   * @return ordering
   */
  @NotNull
  static ByteOrdering fromComparator(@NotNull Comparator<Byte> comparator)
  {
    return new ByteOrdering.Serial()
    {
      private static final long serialVersionUID = 393576352771473074L;

      @NotNull
      @Override
      public Order checkByte(byte v1, byte v2)
      {
        return Order.fromCompare(comparator.compare(v1, v2));
      }

      @NotNull
      @Override
      public Comparator<Byte> asByteComparator()
      {
        return comparator;
      }
    };
  }

  /**
   * A byte ordering which is serializable.
   * It is a recommendation that comparators implement {@code java.io.Serializable}
   * to allow serializing {@code TreeMap}s and similar. If an Ordering is expected to be used
   * as a comparator in such areas implement either this interface, or for lambdas
   * use a cast to force the compiler to make the order serializable:
   * <pre>{@code
   *   return (ByteOrdering & Serializable) (v1, v2) -> v1.compareTo(v2);
   * }</pre>
   * You can also use a {@code (ByteOrdering.Serial)} cast to save a bit of typing,
   * but the above is a nice trick to have in your toolbox.
   */
  interface Serial extends ByteOrdering,
                           java.io.Serializable
  {
  }
}
