// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2019-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;

/**
 * Binary operator with two byte arguments and a byte result.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since October 31, 2019
 */
@FunctionalInterface
public interface ByteOperator2
{
  /**
   * PLUS ({@code +}) operator.
   * In difference to the standard PLUS this operator returns a {@code byte} result.
   */
  ByteOperator2 PLUS = (x1, x2) -> (byte)(x1 + x2);
  /**
   * MINUS ({@code -}) operator.
   * In difference to the standard MINUS this operator returns a {@code byte} result.
   */
  ByteOperator2 MINUS = (x1, x2) -> (byte)(x1 - x2);
  /**
   * MULTIPLICATION ({@code *}) operator.
   * In difference to the standard MULTIPLICATION this operator returns a {@code byte} result.
   */
  ByteOperator2 MULT = (x1, x2) -> (byte)(x1 * x2);
  /**
   * DIVISION ({@code /}) operator.
   * In difference to the standard DIVISION this operator returns a {@code byte} result.
   */
  ByteOperator2 DIV = (x1, x2) -> (byte)(x1 / x2);
  /**
   * MODULO ({@code %}) operator.
   * In difference to the standard MOULO this operator returns a {@code byte} result.
   */
  ByteOperator2 MOD = (x1, x2) -> (byte)(x1 % x2);

  /**
   * Apply the function.
   * @param arg1 first argument
   * @param arg2 second argument
   * @return result of the function
   */
  byte applyAsByte(byte arg1, byte arg2);

  /**
   * Get a partially applied function.
   * @param argument value applied for first argument
   * @return partially evaluated function
   */
  @NotNull
  default ByteOperator1 partialLeft(byte argument)
  {
    return arg -> applyAsByte(argument, arg);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for second argument
   * @return partially evaluated function
   */
  @NotNull
  default ByteOperator1 partialRight(byte argument)
  {
    return arg -> applyAsByte(arg, argument);
  }

  /**
   * Create a function which applies this function and feeds the result into the next function.
   * @param after function called with the result of this function
   * @return chained function
   */
  @NotNull
  default ByteOperator2 andThen(@NotNull ByteOperator1 after)
  {
    return (arg1, arg2) -> after.applyAsByte(applyAsByte(arg1, arg2));
  }
}
