// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2019-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;

/**
 * Ternary operator with three boolean arguments and a boolean result.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since October 31, 2019
 */
@FunctionalInterface
public interface BooleanOperator3
{
  /**
   * Apply the function.
   * @param arg1 first argument
   * @param arg2 second argument
   * @param arg3 third argument
   * @return result of the function
   */
  boolean applyAsBoolean(boolean arg1, boolean arg2, boolean arg3);

  /**
   * Get a partially applied function.
   * @param argument value applied for first argument
   * @return partially evaluated function
   */
  @NotNull
  default BooleanOperator2 partialLeft(boolean argument)
  {
    return (arg2, arg3) -> applyAsBoolean(argument, arg2, arg3);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for second argument
   * @return partially evaluated function
   */
  @NotNull
  default BooleanOperator2 partialMid(boolean argument)
  {
    return (arg1, arg3) -> applyAsBoolean(arg1, argument, arg3);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for third argument
   * @return partially evaluated function
   */
  @NotNull
  default BooleanOperator2 partialRight(boolean argument)
  {
    return (arg1, arg2) -> applyAsBoolean(arg1, arg2, argument);
  }

  /**
   * Create a function which applies this function and feeds the result into the next function.
   * @param after function called with the result of this function
   * @return chained function
   */
  @NotNull
  default BooleanOperator3 andThen(@NotNull BooleanOperator1 after)
  {
    return (arg1, arg2, arg3) -> after.applyAsBoolean(applyAsBoolean(arg1, arg2, arg3));
  }
}
