// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.converter;

import de.caff.annotation.NotNull;
import de.caff.generics.function.Function1;

import java.util.function.Function;

/**
 * A two step type converter.
 * This converter does conversion via an intermediate step.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <T> target type
 * @param <I> intermediate type
 * @param <S> soucre type
 */
public class TransitiveTypeConverter<T, I, S>
        implements Function1<T,S>
{
  private final Function<I, T> secondStep;
  private final Function<S, ? extends I> firstStep;

  /**
   * Constructor.
   * @param secondStep second conversion step
   * @param firstStep  first conversion step
   */
  public TransitiveTypeConverter(@NotNull Function<I, T> secondStep,
                                 @NotNull Function<S, ? extends I> firstStep)
  {
    this.secondStep  = secondStep;
    this.firstStep = firstStep;
  }

  /**
   * Convert an object to another type.
   *
   * @param object object to convert
   * @return target object
   */
  @Override
  public T apply(S object)
  {
    return secondStep.apply(firstStep.apply(object));
  }
}
