// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.converter;

import de.caff.annotation.NotNull;
import de.caff.generics.FragileTypeConverter;
import de.caff.generics.TypeConverterException;
import de.caff.generics.Types;

/**
 * A two step fragile type converter.
 * This converter does conversion via an intermediate step.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <T> target type
 * @param <I> intermediate type
 * @param <S> source type
 */
public class TransitiveFragileTypeConverter<T, I, S>
        implements FragileTypeConverter<T,S>
{
  private final FragileTypeConverter<T, I> secondStep;
  private final FragileTypeConverter<I, S> firstStep;

  /**
   * Constructor.
   * @param secondStep type converter which does the second step of the conversion
   * @param firstStep  type converter which does the first step of the conversion
   */
  public TransitiveFragileTypeConverter(@NotNull FragileTypeConverter<T, I> secondStep,
                                        @NotNull FragileTypeConverter<? extends I, S> firstStep)
  {
    this.secondStep  = secondStep;
    this.firstStep = Types.downCastT(firstStep);
  }

  /**
   * Convert an object to another type.
   *
   * @param object object to convert
   * @return target object
   */
  @Override
  public T convert(S object) throws TypeConverterException
  {
    return secondStep.convert(firstStep.convert(object));
  }
}
