// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.converter;

import de.caff.generics.FragileTypeConverter;
import de.caff.generics.TypeConverterException;

/**
 * Type converter which converts strings to short integers.
 * <p>
 * It may throw a {@link de.caff.generics.TypeConverterException} wrapping
 * a {@link NumberFormatException}.
 * Null values will be passed through.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class StringToShortTypeConverter
        implements FragileTypeConverter<Short, String>
{
  /** Radix */
  private final int radix;

  /**
   * Constructor.
   * @param radix radix to use for conversion
   */
  public StringToShortTypeConverter(int radix)
  {
    this.radix = radix;
  }

  /**
   * Constructor.
   * This creates a base10 string to integer converter.
   */
  public StringToShortTypeConverter()
  {
    this(10);
  }

  /**
   * Convert an object to another type.
   *
   * @param object object to convert
   * @return target object
   */
  @Override
  public Short convert(String object) throws TypeConverterException
  {
    if (object == null) {
      return null;
    }
    try {
      return Short.parseShort(object, radix);
    } catch (NumberFormatException e) {
      throw new TypeConverterException(e);
    }
  }
}
