// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.converter;

import de.caff.annotation.NotNull;
import de.caff.generics.OrderedPair;
import de.caff.generics.function.Function1;

import java.util.Map;
import java.util.function.Function;

/**
 * A map converter which converts the key and the value in a map simultaneously.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class MapKeyValueConverter<KT, VT, KS, VS>
  implements Function1<Map.Entry<KT, VT>, Map.Entry<KS, VS>>
{
  @NotNull
  private final Function<KS,KT> keyConverter;
  @NotNull
  private final Function<VS,VT> valueConverter;

  /**
   * Constructor.
   * @param keyConverter the key converter, used for changing the key
   * @param valueConverter the value converter, used for changing the value
   */
  public MapKeyValueConverter(@NotNull Function<KS, KT> keyConverter,
                              @NotNull Function<VS, VT> valueConverter)
  {
    this.keyConverter = keyConverter;
    this.valueConverter = valueConverter;
  }

  /**
   * Convert an object to another type.
   *
   * @param object object to convert
   * @return target object
   */
  @Override
  public Map.Entry<KT, VT> apply(Map.Entry<KS, VS> object)
  {
    return OrderedPair.create(keyConverter.apply(object.getKey()),
                              valueConverter.apply(object.getValue()));
  }
}
