// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.converter;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.FragileTypeConverter;
import de.caff.generics.TypeConverterException;
import de.caff.generics.function.Function1;

/**
 * A standard type converter which wraps a fragile type converter.
 * This allows to use fragile type converters as standard type converters.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class CarelessTypeConverter<T, S>
        implements Function1<T, S>
{
  @NotNull
  private final FragileTypeConverter<T, S> wrapped;
  private final T defaultValue;
  private TypeConverterException latestException;

  /**
   * Constructor.
   * @param wrapped converter to wrap
   * @param defaultValue default value used as return value if an exception occurred
   */
  public CarelessTypeConverter(@NotNull FragileTypeConverter<T, S> wrapped,
                               @Nullable T defaultValue)
  {
    this.wrapped = wrapped;
    this.defaultValue = defaultValue;
  }

  /**
   * Get the latest exception caught during conversion.
   * @return latest exception caught
   */
  @Nullable
  public TypeConverterException getLatestException()
  {
    return latestException;
  }

  /**
   * Clear the latest exception.
   */
  public void clear()
  {
    latestException = null;
  }

  /**
   * Convert an object to another type.
   *
   * @param object object to convert
   * @return target object
   */
  @Override
  public T apply(S object)
  {
    try {
      return wrapped.convert(object);
    } catch (TypeConverterException e) {
      latestException = e;
      return defaultValue;
    }
  }
}
