// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.checker;

import de.caff.annotation.NotNull;
import de.caff.generics.function.Predicate1;

import java.util.Comparator;

/**
 * This value checker compares the checked value to its element,
 * and returns {@code true} if the given value is smaller.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @see Check#lessThan(Comparable)
 * @see Check#lessThan(Object, java.util.Comparator)
 */
public class LessThanChecker<T>
        implements Predicate1<T>
{
  private final T element;
  private final Comparator<T> comparator;

  /**
   * Constructor.
   * @param element element to compare against
   * @param comparator element comparator
   */
  public LessThanChecker(@NotNull T element,
                         @NotNull Comparator<T> comparator)
  {
    this.element = element;
    this.comparator = comparator;
  }

  /**
   * Check a value.
   *
   * @param value the value
   * @return <code>true</code>: the value is accepted<br>
   *         <code>false</code>: the value is not accepted
   */
  @Override
  public boolean test(T value)
  {
    return comparator.compare(element, value) > 0;
  }
}
