// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.checker;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.function.Predicate1;

import java.util.Collection;
import java.util.Comparator;
import java.util.function.Predicate;

/**
 * Helper methods for using {@link java.util.function.Predicate}s as checkers,,
 * useful for combination of checkers.
 * <p>
 * The following example only keeps all numbers between 17 and 42 in the outgoing list:
 * <blockquote><pre>
 *   Collection&lt;Integer&gt; foo(Collection&lt;Integer&gt; incoming) {
 *     return Types.filter(incoming,
 *                         Check.and(Check.greaterEquals(17), Check.lessEquals(42)));
 *   }
 * </pre></blockquote>
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public final class Check
{
  /**
   * Not to be created.
   */
  private Check()
  {
  }

  /**
   * Get a value checker with checks for a given value using the equals method.
   * @param value value to look for
   * @param <T> value type
   * @return equals checker
   */
  @NotNull
  public static <T> Predicate1<T> isEqual(@NotNull T value)
  {
    return new EqualsChecker<>(value);
  }

  /**
   * Get a value checker with checks for a given value using the equals method.
   * @param value value to look for
   * @param <T> value type
   * @return not equals checker
   */
  @NotNull
  public static <T> Predicate1<T> notEquals(@NotNull T value)
  {
    return new NotEqualsChecker<>(value);
  }

  /**
   * Get a value checker which checks for a given value using reference equality.
   * @param element value to look for
   * @param <T> value type
   * @return reference equals checker
   */
  @NotNull
  public static <T> Predicate1<T> refEquals(@Nullable final T element)
  {
    return value -> element == value;
  }

  /**
   * Get a value checker which checks for being not a given value using reference equality.
   * @param element value to look for
   * @param <T> value type
   * @return reference not equals checker
   */
  @NotNull
  public static <T> Predicate1<T> notRefEquals(@Nullable final T element)
  {
    return value -> element != value;
  }

  /**
   * Get a value checker which returns the boolean and of two checkers.
   * @param checker1 first checker
   * @param checker2 second checker
   * @param <T> checked value type
   * @return and checker
   */
  @NotNull
  public static <T> Predicate1<T> and(@NotNull Predicate<? super T> checker1,
                                      @NotNull Predicate<? super T> checker2)
  {
    return new AndChecker<>(checker1, checker2);
  }

  /**
   * Get a value checker which returns the boolean and of some checkers.
   * @param checker1 first checker
   * @param checker2 second checker
   * @param moreCheckers more checkers
   * @param <T> checked value type
   * @return and checker
   */
  @NotNull
  @SafeVarargs
  @SuppressWarnings("varargs")
  public static <T> Predicate1<T> and(@NotNull Predicate<? super T> checker1,
                                      @NotNull Predicate<? super T> checker2,
                                      @NotNull Predicate<? super T> ... moreCheckers)
  {
    return new AndChecker<>(checker1, checker2, moreCheckers);
  }

  /**
   * Get a value checker which returns the boolean and of some checkers
   * @param checkers  checkers collection (must not be empty)
   * @param <T> checked value type
   * @return combined and checker
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T> Predicate1<T> and(@NotNull Collection<? extends Predicate<T>> checkers)
  {
    if (checkers.isEmpty()) {
      throw new IllegalArgumentException("checkers must not be enpty!");
    }
    if (checkers.size() == 1) {
      return Predicate1.from(checkers.iterator().next());
    }
    return new AndChecker<T>(checkers);
  }

  /**
   * Get a value checker which returns the boolean or of two checkers.
   * @param checker1 first checker
   * @param checker2 second checker
   * @param <T> checked value type
   * @return or checker
   */
  @NotNull
  public static <T> Predicate1<T> or(@NotNull Predicate<? super T> checker1,
                                     @NotNull Predicate<? super T> checker2)
  {
    return new OrChecker<>(checker1, checker2);
  }

  /**
   * Get a value checker which returns the boolean or of some checkers.
   * @param checker1 first checker
   * @param checker2 second checker
   * @param <T> checked value type
   * @param moreCheckers more checkers
   * @return or checker
   */
  @NotNull
  @SafeVarargs
  @SuppressWarnings("varargs")
  public static <T> Predicate1<T> or(@NotNull Predicate<? super T> checker1,
                                     @NotNull Predicate<? super T> checker2,
                                     @NotNull Predicate<? super T> ... moreCheckers)
  {
    return new OrChecker<>(checker1, checker2, moreCheckers);
  }

  /**
   * Get a value checker which returns the boolean or of some checkers
   * @param checkers  checkers collection (must not be empty)
   * @param <T> checked value type
   * @return combined or checker
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T> Predicate1<T> or(@NotNull Collection<? extends Predicate<T>> checkers)
  {
    if (checkers.isEmpty()) {
      throw new IllegalArgumentException("checkers must not be enpty!");
    }
    if (checkers.size() == 1) {
      return Predicate1.from(checkers.iterator().next());
    }
    return new OrChecker<T>(checkers);
  }

  /**
   * Get a value checker which returns the boolean xor of two checkers.
   * @param checker1 first checker
   * @param checker2 second checker
   * @param <T> checked value type
   * @return xor checker
   */
  @NotNull
  public static <T> Predicate1<T> xor(@NotNull final Predicate<? super T> checker1,
                                      @NotNull final Predicate<? super T> checker2)
  {
    return value -> checker1.test(value) ^ checker2.test(value);
  }

  /**
   * Get a value checker which negates the result of a checker.
   * @param checker checker
   * @param <T> checked value type
   * @return negated checker
   */
  @NotNull
  public static <T> Predicate1<T> not(@NotNull final Predicate<T> checker)
  {
    return value -> !checker.test(value);
  }

  /**
   * Get a checker which compares to a given element,
   * and returns {@code true} if the checked value is less than the element.
   * @param element element to check against
   * @param <T> element/value type
   * @return less than checker
   */
  @NotNull
  public static <T> Predicate1<T> lessThan(@NotNull final Comparable<T> element)
  {
    return value -> element.compareTo(value) > 0;
  }

  /**
   * Get a checker which compares to a given element,
   * and returns {@code true} if the checked value is less than the element.
   * @param element element to check against
   * @param comparator comparator to do the comparison
   * @param <T> element/value type
   * @return less than checker
   */
  @NotNull
  public static <T> Predicate1<T> lessThan(@NotNull T element,
                                             @NotNull Comparator<T> comparator)
  {
    return new LessThanChecker<>(element, comparator);
  }

  /**
   * Get a checker which compares to a given element,
   * and returns {@code true} if the checked value is less than the element,
   * or equal.
   * @param element element to check against
   * @param <T> element/value type
   * @return less equals checker
   */
  @NotNull
  public static <T> Predicate1<T> lessEquals(@NotNull final Comparable<T> element)
  {
    return value -> element.compareTo(value) >= 0;
  }

  /**
   * Get a checker which compares to a given element,
   * and returns {@code true} if the checked value is less than the element,
   * or equal.
   * @param element element to check against
   * @param comparator comparator to do the comparison
   * @param <T> element/value type
   * @return less than checker
   */
  @NotNull
  public static <T> Predicate1<T> lessEquals(@NotNull T element,
                                             @NotNull Comparator<T> comparator)
  {
    return new LessEqualsChecker<>(element, comparator);
  }

  /**
   * Get a checker which compares to a given element,
   * and returns {@code true} if the checked value is greater than the element.
   * @param element element to check against
   * @param <T> element/value type
   * @return less than checker
   */
  @NotNull
  public static <T> Predicate1<T> greaterThan(@NotNull final Comparable<T> element)
  {
    return value -> element.compareTo(value) < 0;
  }

  /**
   * Get a checker which compares to a given element,
   * and returns {@code true} if the checked value is greater than the element.
   * @param element element to check against
   * @param comparator comparator to do the comparison
   * @param <T> element/value type
   * @return less than checker
   */
  @NotNull
  public static <T> Predicate1<T> greaterThan(@NotNull T element,
                                              @NotNull Comparator<T> comparator)
  {
    return new GreaterThanChecker<>(element, comparator);
  }

  /**
   * Get a checker which compares to a given element,
   * and returns {@code true} if the checked value is greater than the element,
   * or equal.
   * @param element element to check against
   * @param <T> element/value type
   * @return less equals checker
   */
  @NotNull
  public static <T> Predicate1<T> greaterEquals(@NotNull final Comparable<T> element)
  {
    return value -> element.compareTo(value) <= 0;
  }

  /**
   * Get a checker which compares to a given element,
   * and returns {@code true} if the checked value is greater than the element,
   * or equal.
   * @param element element to check against
   * @param comparator comparator to do the comparison
   * @param <T> element/value type
   * @return less than checker
   */
  @NotNull
  public static <T> Predicate1<T> greaterEquals(@NotNull T element,
                                                @NotNull Comparator<T> comparator)
  {
    return new GreaterEqualsChecker<>(element, comparator);
  }

  /**
   * Get a checker which checks whether an object is an instance of a given class
   * or interface.
   * <p>
   * The check is using reflection, namely the {@link Class#isInstance(Object)} method,
   * which is not considered fast. If you need this check often, rethink your design.
   *
   * @param type  class or interface type
   * @param <T>   checker parameter
   * @return value checker which returns {@code true} for objects of the given type
   */
  @NotNull
  public static <T> Predicate1<T> isType(@NotNull final Class<?> type)
  {
    return type::isInstance;
  }

  /**
   * Get a checker which checks whether an object is not an instance of a given class
   * or interface.
   * <p>
   * The check is using reflection, namely the {@link Class#isInstance(Object)} method,
   * which is not considered fast. If you need this check often, rethink your design.
   *
   * @param type  class or interface type
   * @param <T>   checker parameter
   * @return value checker which returns {@code true} for objects not of the given type
   */
  @NotNull
  public static <T> Predicate1<T> isNotType(@NotNull final Class<?> type)
  {
    return value -> !type.isInstance(value);
  }
}
