// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.checker;

import de.caff.annotation.NotNull;
import de.caff.generics.function.Predicate1;

import java.util.ArrayList;
import java.util.Collection;
import java.util.function.Predicate;

/**
 * A value checker which returns {@code true} when all wrapped
 * checkers return {@code true}.
 * It is doing shortcut checking similar to the Java
 * {@code &amp;&amp;} operator.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @see Check#and(java.util.function.Predicate, java.util.function.Predicate)
 */
public class AndChecker<T>
        implements Predicate1<T>
{
  private final Collection<Predicate<T>> checker;

  /**
   * Constructor.
   * @param checker1 first checker
   * @param checker2 second checker
   */
  @SuppressWarnings("unchecked")
  public AndChecker(@NotNull Predicate<? super T> checker1,
                    @NotNull Predicate<? super T> checker2)
  {
    checker = new ArrayList<>(2);
    checker.add((Predicate1<T>)checker1);
    checker.add((Predicate1<T>)checker2);
  }

  /**
   * Constructor.
   * @param checker1 first checker
   * @param checker2 second checker
   * @param more     more checkers
   */
  @SafeVarargs
  @SuppressWarnings({"unchecked", "varargs"})
  public AndChecker(@NotNull Predicate<? super T> checker1,
                    @NotNull Predicate<? super T> checker2,
                    @NotNull Predicate<? super T> ... more)
  {
    checker = new ArrayList<>(2 + more.length);
    checker.add((Predicate1<T>)checker1);
    checker.add((Predicate1<T>)checker2);
    for (Predicate<? super T> p : more) {
      checker.add((Predicate1<T>)p);
    }
  }

  /**
   * Constructor.
   * @param checkers  checkers
   */
  public AndChecker(@NotNull Collection<? extends Predicate<T>> checkers)
  {
    checker = new ArrayList<>(checkers);
  }

  /**
   * Check a value.
   *
   * @param value the value
   * @return {@code true}: the value is accepted<br>
   *         {@code false}: the value is not accepted
   */
  @Override
  public boolean test(T value)
  {
    for (Predicate<? super T> c : checker) {
      if (!c.test(value)) {
        return false;
      }
    }
    return true;
  }
}
