// ============================================================================
// File:               FastSortByte
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2023-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            1/25/23 2:59 PM
//=============================================================================
package de.caff.generics.algorithm;

import de.caff.annotation.NotNull;
import de.caff.generics.MutableCharIndexable;
import de.caff.generics.Order;
import de.caff.generics.function.CharOrdering;

import java.util.Arrays;

/**
 * A sort algorithm for byte arrays which is much faster compared to TimSort.
 * This algorithm is making advantage that there are just 256 distinct byte values,
 * so for larger arrays it is simpler to count the bytes.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 25, 2023
 */
public class FastSortChar
{
  /**
   * Birthday paradox says that for 256 possible values there is a 50% chance
   * that 2 random elements are the same for a set of only 20. But creating
   * the array and counting takes some time, too.
   */
  private static final int COUNT_SORT_BOUNDARY = 333;
  /**
   * Number of different char values.
   */
  public static final int NUM_CHARS = 0x10000;

  /**
   * Sort the given byte array in natural order.
   * This will forward to {@link Arrays#sort(byte[])}.
   * @param bytes byte array
   */
  public static void sort(@NotNull byte[] bytes)
  {
    Arrays.sort(bytes);
  }

  /**
   * Sort the given char array in user-defined order.
   * This will forward to {@link Arrays#sort(char[])}.
   * @param chars char array
   * @param order sort order
   */
  public static void sort(@NotNull char[] chars,
                          @NotNull CharOrdering order)
  {
    sort(MutableCharIndexable.viewArray(chars), order);
  }

  /**
   * Sort a part of the given char array in natural order.
   * This will forward to {@link Arrays#sort(char[],int,int)}.
   * @param chars char array
   * @param startIndex start index into the char array where the part to be sorted begins
   * @param length     length of the part to be sorted
   */
  public static void sort(@NotNull char[] chars, int startIndex, int length)
  {
    Arrays.sort(chars, startIndex, length);
  }

  /**
   * Sort a part of the given char array in user-defined order.
   * This will forward to {@link Arrays#sort(char[],int,int)}.
   * @param chars char array
   * @param startIndex start index into the char array where the part to be sorted begins
   * @param length     length of the part to be sorted
   * @param order      sort order   */
  public static void sort(@NotNull char[] chars, int startIndex, int length,
                          @NotNull CharOrdering order)
  {
    sort(MutableCharIndexable.viewArray(chars, startIndex, length), order);
  }

  /**
   * Sort the given mutable char indexable in natural order.
   * @param chars char indexable
   */
  public static void sort(@NotNull MutableCharIndexable chars)
  {
    sort(chars, CharOrdering.ASCENDING);
  }

  /**
   * Sort the given mutable char indexable by the defined order.
   * @param chars char indexable
   * @param order sort order
   */
  public static void sort(@NotNull MutableCharIndexable chars,
                          @NotNull CharOrdering order)
  {
    final int size = chars.size();

    if (size < COUNT_SORT_BOUNDARY) {  // heuristic value where qsort is faster than counting sort
      // do a quick sort directly
      quickSort(chars, order);
      return;
    }

    final int[] counter = new int[NUM_CHARS];
    chars.forEachChar(ch -> ++counter[ch]);
    final char[] testSet = new char[Math.min(NUM_CHARS, size)];
    int testSetSize = 0;
    for (int i = 0;  i < counter.length;  ++i) {
      final int count = counter[i];
      if (count > 0) {
        testSet[testSetSize++] = (char)(i);
      }
    }
    if (testSetSize == size) {
      quickSort(chars, order);
      return;
    }

    quickSort(MutableCharIndexable.viewArray(testSet, 0, testSetSize), order);
    int idx = 0;
    for (int i = 0;  i < testSetSize;  ++i) {
      final char ch = testSet[i];
      final int count = counter[ch];
      assert count > 0;
      chars.setRange(idx, count, ch);
      idx += count;
    }
  }

  /**
   * Simple quick sort.
   * @param elements mutable char indexable being sorted
   * @param order    sort order
   */
  private static void quickSort(@NotNull MutableCharIndexable elements,
                                @NotNull CharOrdering order)
  {
    final int size = elements.size();
    if (size <= 1) {
      return;
    }
    if (size == 2) {
      if (order.checkChar(elements.get(0), elements.get(1)) == Order.Descending) {
        elements.swap(0, 1);
      }
      return;
    }

    final int pIndex = partition(elements, order);

    quickSort(elements.headSet(pIndex), order);
    quickSort(elements.tailSet(pIndex + 1), order);
  }

  /**
   * Part of simple quick sort: partition around pivot element.
   * @param elements mutable char indexable being sorted
   * @param order    sort order
   * @return index of the partition element, all elements left will
   *         be less than it according to the order, all elements
   *         to the right will be greater
   */
  private static int partition(@NotNull MutableCharIndexable elements,
                               @NotNull CharOrdering order)
  {
    final int len = elements.size() - 1;
    final char pivot = elements.get(len);
    int i = 0;

    for (int j = 0; j < len; j++) {
      if (order.checkChar(elements.get(j), pivot) == Order.Ascending) {
        elements.swap(i++, j);
      }
    }

    elements.swyp(i, -1);

    return i;
  }
}
