// ============================================================================
// File:               FastSortByte
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2023-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            1/25/23 2:59 PM
//=============================================================================
package de.caff.generics.algorithm;

import de.caff.annotation.NotNull;
import de.caff.generics.MutableByteIndexable;
import de.caff.generics.Order;
import de.caff.generics.function.ByteOrdering;

import java.util.Arrays;

/**
 * A non-stable sort algorithm for byte arrays which is much faster compared to TimSort.
 * This algorithm is making advantage that there are just 256 distinct byte values,
 * so for larger arrays it is simpler to count the bytes.
 * <p>
 * It also uses a best-of-3 pivot selection for partitioning during quicksort,
 * which is slower than DualPivotQuicksort but much simpler.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 25, 2023
 */
class FastSortByte
{
  /** Number of possible byte values. */
  private static final int NUM_BYTES = 0x100;
  /** Offset to move bytes in order into the positive range. */
  private static final int OFFSET = NUM_BYTES / 2;
  /**
   * Birthday paradox says that for 256 possible values there is a 50% chance
   * that 2 random elements are the same for a set of only 20. But creating
   * the array and counting takes some time, too.
   */
  private static final int COUNT_SORT_BOUNDARY = 33;

  /**
   * Sort the given byte array in natural order.
   * This will forward to {@link Arrays#sort(byte[])}.
   * @param bytes byte array
   */
  public static void sort(@NotNull byte[] bytes)
  {
    Arrays.sort(bytes);
  }

  /**
   * Sort the given byte array in user-defined order.
   * This will forward to {@link Arrays#sort(byte[])}.
   * @param bytes byte array
   * @param order sort order
   */
  public static void sort(@NotNull byte[] bytes,
                          @NotNull ByteOrdering order)
  {
    sort(MutableByteIndexable.viewArray(bytes), order);
  }

  /**
   * Sort a part of the given byte array in natural order.
   * This will forward to {@link Arrays#sort(byte[],int,int)}.
   * @param bytes byte array
   * @param startIndex start index into the byte array where the part to be sorted begins
   * @param length     length of the part to be sorted
   */
  public static void sort(@NotNull byte[] bytes, int startIndex, int length)
  {
    Arrays.sort(bytes, startIndex, length);
  }

  /**
   * Sort a part of the given byte array in user-defined order.
   * This will forward to {@link Arrays#sort(byte[],int,int)}.
   * @param bytes byte array
   * @param startIndex start index into the byte array where the part to be sorted begins
   * @param length     length of the part to be sorted
   * @param order      sort order   */
  public static void sort(@NotNull byte[] bytes, int startIndex, int length,
                          @NotNull ByteOrdering order)
  {
    sort(MutableByteIndexable.viewArray(bytes, startIndex, length), order);
  }

  /**
   * Sort the given mutable byte indexable in natural order.
   * @param bytes byte indexable
   */
  public static void sort(@NotNull MutableByteIndexable bytes)
  {
    sort(bytes, ByteOrdering.ASCENDING);
  }

  /**
   * Sort the given mutable byte indexable using the given ordering.
   * @param bytes byte indexable
   * @param order sort order
   */
  public static void sort(@NotNull MutableByteIndexable bytes,
                          @NotNull ByteOrdering order)
  {
    final int size = bytes.size();

    if (size < COUNT_SORT_BOUNDARY) {  // heuristic value where qsort is faster than counting sort
      // do a quick sort directly
      quickSort(bytes, order);
      return;
    }

    final int[] counter = new int[NUM_BYTES];
    bytes.forEachByte(b -> ++counter[b + OFFSET]);
    final byte[] testSet = new byte[Math.min(NUM_BYTES, size)];
    int testSetSize = 0;
    for (int i = 0;  i < counter.length;  ++i) {
      final int count = counter[i];
      if (count > 0) {
        testSet[testSetSize++] = (byte)(i - OFFSET);
      }
    }
    if (testSetSize == size) {
      quickSort(bytes, order);
      return;
    }

    quickSort(MutableByteIndexable.viewArray(testSet, 0, testSetSize), order);
    int idx = 0;
    for (int i = 0;  i < testSetSize;  ++i) {
      final byte b = testSet[i];
      final int count = counter[b + OFFSET];
      assert count > 0;
      bytes.setRange(idx, count, b);
      idx += count;
    }
  }

  static void quickSort(@NotNull MutableByteIndexable elements,
                        @NotNull ByteOrdering order)
  {
    final int size = elements.size();
    if (size <= 1) {
      return;
    }
    if (size == 2) {
      if (order.checkByte(elements.get(0), elements.get(1)) == Order.Descending) {
        elements.swap(0, 1);
      }
      return;
    }

    if (size < COUNT_SORT_BOUNDARY) {
      insertionSort(elements, order);
      return;
    }

    final int pIndex = partition(elements, order);

    quickSort(elements.headSet(pIndex), order);
    quickSort(elements.tailSet(pIndex + 1), order);
  }

  private static int partition(@NotNull MutableByteIndexable elements,
                               @NotNull ByteOrdering order)
  {
    final int len = elements.size() - 1;
    final byte pivot;
    // select middle of 3 test items as pivot
    final int m = len >> 1;
    final byte start = elements.get(0);
    final byte mid   = elements.get(m);
    final byte end   = elements.get(len);

    if (order.ascending(start, mid)) {
      if (order.ascending(mid, end)) { // ABC
        pivot = mid;
        elements.swap(m, len);
      }
      else { // ACB or BCA
        pivot = end;
      }
    }
    else {
      if (order.ascending(mid, end)) { // BAC or CAB
        pivot = start;
        elements.swap(0, len);
      }
      else { // CBA
        pivot = mid;
        elements.swap(m, len);
      }
    }

    int i = 0;

    for (int j = 0; j < len; j++) {
      if (order.checkByte(elements.get(j), pivot) == Order.Ascending) {
        elements.swap(i++, j);
      }
    }

    elements.swyp(i, -1);

    return i;
  }

  private static void insertionSort(@NotNull MutableByteIndexable a, @NotNull ByteOrdering order)
  {
    final int right = a.size() - 1;
    for (int i = 0, j = i; i < right; j = ++i) {
      final byte ai = a.get(i + 1);
      while (order.ascending(ai, a.get(j))) {
        a.set(j + 1, a.get(j));
        if (j-- == 0) {
          break;
        }
      }
      a.set(j + 1, ai);
    }
  }

}
