/*
 * Copyright (c) 2009, 2011, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package de.caff.generics.algorithm;

import de.caff.annotation.NotNull;
import de.caff.generics.MutableByteIndexable;
import de.caff.generics.MutableIndexable;
import de.caff.generics.MutableIntIndexable;
import de.caff.generics.function.*;

/**
 * This class implements the Dual-Pivot Quicksort algorithm by
 * Vladimir Yaroslavskiy, Jon Bentley, and Josh Bloch. The algorithm
 * offers O(n log(n)) performance on many data sets that cause other
 * quicksorts to degrade to quadratic performance, and is typically
 * faster than traditional (one-pivot) Quicksort implementations.
 * <p>
 * Changes by Rammi: make this class public and allow user-defined order.
 * Because standard order tests for primitive types are fast operations
 * having a user-defined order makes sorting some 50% slower for these types.
 * For test purposes this also allows {@link #sort(MutableIndexable,Ordering)
 * sorting generic indexables} like {@link TimSort}, but tests show that
 * TimSort is faster in most cases.
 * <p>
 * Like standard quicksort this sorting algorithm is not stable.
 *
 * @author Vladimir Yaroslavskiy
 * @author Jon Bentley
 * @author Josh Bloch
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 *
 * @version 2011.02.11 m765.827.12i:5\7pm
 * @since 1.7
 */
public final class DualPivotQuicksort {
  /**
   * Prevents instantiation.
   */
  private DualPivotQuicksort() {}

  /*
   * Tuning parameters.
   */

  /**
   * The maximum number of runs in merge sort.
   */
  private static final int MAX_RUN_COUNT = 67;

  /**
   * The maximum length of run in merge sort.
   */
  private static final int MAX_RUN_LENGTH = 33;

  /**
   * If the length of an array to be sorted is less than this
   * constant, Quicksort is used in preference to merge sort.
   */
  private static final int QUICKSORT_THRESHOLD = 286;

  /**
   * If the length of an array to be sorted is less than this
   * constant, insertion sort is used in preference to Quicksort.
   */
  private static final int INSERTION_SORT_THRESHOLD = 47;

  /**
   * If the length of a byte array to be sorted is greater than this
   * constant, counting sort is used in preference to insertion sort.
   */
  private static final int COUNTING_SORT_THRESHOLD_FOR_BYTE = 29;

  /**
   * If the length of a short or char array to be sorted is greater
   * than this constant, counting sort is used in preference to Quicksort.
   */
  private static final int COUNTING_SORT_THRESHOLD_FOR_SHORT_OR_CHAR = 3200;

  /*
   * Sorting methods for seven primitive types.
   */

  /**
   * Sorts the specified array.
   *
   * @param a the array to be sorted
   * @param order user-defined order
   */
  public static void sort(int[] a, @NotNull IntOrdering order) {
    sort(a, 0, a.length - 1, order);
  }

  /**
   * Sorts the specified range of the array.
   *
   * @param a the array to be sorted
   * @param left the index of the first element, inclusive, to be sorted
   * @param right the index of the last element, inclusive, to be sorted
   * @param order user-defined order
   */
  public static void sort(int[] a, int left, int right, @NotNull IntOrdering order) {
    // Use Quicksort on small arrays
    if (right - left < QUICKSORT_THRESHOLD) {
      sort(a, left, right, order, true);
      return;
    }

    /*
     * Index run[i] is the start of i-th run
     * (ascending or descending sequence).
     */
    final int[] run = new int[MAX_RUN_COUNT + 1];
    int count = 0; run[0] = left;

    // Check if the array is nearly sorted
    for (int k = left; k < right; run[count] = k) {
      switch (order.checkInt(a[k], a[k + 1])) {
      case Ascending:
        while (++k <= right && order.ascendingOrSame(a[k - 1], a[k]) ) ;
        break;
      case Descending:
        while (++k <= right && order.descendingOrSame(a[k - 1], a[k])) ;
        for (int lo = run[count] - 1, hi = k; ++lo < --hi; ) {
          final int t = a[lo];
          a[lo] = a[hi];
          a[hi] = t;
        }
        break;
      default:
        for (int m = MAX_RUN_LENGTH; ++k <= right && order.same(a[k - 1], a[k]); ) {
          if (--m == 0) {
            sort(a, left, right, order, true);
            return;
          }
        }
        break;
      }

      /*
       * The array is not highly structured,
       * use Quicksort instead of merge sort.
       */
      if (++count == MAX_RUN_COUNT) {
        sort(a, left, right, order, true);
        return;
      }
    }

    // Check special cases
    if (run[count] == right++) { // The last run contains one element
      run[++count] = right;
    } else if (count == 1) { // The array is already sorted
      return;
    }

    /*
     * Create temporary array, which is used for merging.
     * Implementation note: variable "right" is increased by 1.
     */
    int[] b; byte odd = 0;
    for (int n = 1; (n <<= 1) < count; odd ^= 1);

    if (odd == 0) {
      b = a; a = new int[b.length];
      for (int i = left - 1; ++i < right; a[i] = b[i]);
    } else {
      b = new int[a.length];
    }

    // Merging
    for (int last; count > 1; count = last) {
      for (int k = (last = 0) + 2; k <= count; k += 2) {
        int hi = run[k], mi = run[k - 1];
        for (int i = run[k - 2], p = i, q = mi; i < hi; ++i) {
          if (q >= hi || p < mi && order.ascending(a[p], a[q]))  {
            b[i] = a[p++];
          } else {
            b[i] = a[q++];
          }
        }
        run[++last] = hi;
      }
      if ((count & 1) != 0) {
        for (int i = right, lo = run[count - 1]; --i >= lo;
             b[i] = a[i]
        );
        run[++last] = right;
      }
      int[] t = a; a = b; b = t;
    }
  }

  /**
   * Sorts the specified range of the array by Dual-Pivot Quicksort.
   *
   * @param a the array to be sorted
   * @param left the index of the first element, inclusive, to be sorted
   * @param right the index of the last element, inclusive, to be sorted
   * @param leftmost indicates if this part is the leftmost in the range
   */
  private static void sort(int[] a, int left, int right,
                           @NotNull IntOrdering order,
                           boolean leftmost) {
    int length = right - left + 1;

    // Use insertion sort on tiny arrays
    if (length < INSERTION_SORT_THRESHOLD) {
      if (leftmost) {
        /*
         * Traditional (without sentinel) insertion sort,
         * optimized for server VM, is used in case of
         * the leftmost part.
         */
        for (int i = left, j = i; i < right; j = ++i) {
          int ai = a[i + 1];
          while (order.ascending(ai, a[j])) {
            a[j + 1] = a[j];
            if (j-- == left) {
              break;
            }
          }
          a[j + 1] = ai;
        }
      } else {
        /*
         * Skip the longest ascending sequence.
         */
        do {
          if (left >= right) {
            return;
          }
        } while (order.descendingOrSame(a[++left], a[left - 1]));

        /*
         * Every element from adjoining part plays the role
         * of sentinel, therefore this allows us to avoid the
         * left range check on each iteration. Moreover, we use
         * the more optimized algorithm, so called pair insertion
         * sort, which is faster (in the context of Quicksort)
         * than traditional implementation of insertion sort.
         */
        for (int k = left; ++left <= right; k = ++left) {
          int a1 = a[k], a2 = a[left];

          if (order.ascending(a1, a2)) {
            a2 = a1; a1 = a[left];
          }
          while (order.ascending(a1, a[--k])) {
            a[k + 2] = a[k];
          }
          a[++k + 1] = a1;

          while (order.ascending(a2, a[--k])) {
            a[k + 1] = a[k];
          }
          a[k + 1] = a2;
        }
        final int last = a[right];

        while (order.ascending(last, a[--right])) {
          a[right + 1] = a[right];
        }
        a[right + 1] = last;
      }
      return;
    }

    // Inexpensive approximation of length / 7
    int seventh = (length >> 3) + (length >> 6) + 1;

    /*
     * Sort five evenly spaced elements around (and including) the
     * center element in the range. These elements will be used for
     * pivot selection as described below. The choice for spacing
     * these elements was empirically determined to work well on
     * a wide variety of inputs.
     */
    int e3 = (left + right) >>> 1; // The midpoint
    int e2 = e3 - seventh;
    int e1 = e2 - seventh;
    int e4 = e3 + seventh;
    int e5 = e4 + seventh;

    // Sort these elements using insertion sort
    if (order.ascending(a[e2], a[e1])) { int t = a[e2];a[e2] = a[e1];a[e1] = t; }

    if (order.ascending(a[e3], a[e2])) { int t = a[e3];a[e3] = a[e2];a[e2] = t;
      if (order.ascending(t, a[e1])) {a[e2] = a[e1];a[e1] = t; }
    }
    if (order.ascending(a[e4], a[e3])) { int t = a[e4];a[e4] = a[e3];a[e3] = t;
      if (order.ascending(t, a[e2])) {a[e3] = a[e2];a[e2] = t;
        if (order.ascending(t, a[e1])) {a[e2] = a[e1];a[e1] = t; }
      }
    }
    if (order.ascending(a[e5], a[e4])) { int t = a[e5];a[e5] = a[e4];a[e4] = t;
      if (order.ascending(t, a[e3])) {a[e4] = a[e3];a[e3] = t;
        if (order.ascending(t, a[e2])) {a[e3] = a[e2];a[e2] = t;
          if (order.ascending(t, a[e1])) {a[e2] = a[e1];a[e1] = t; }
        }
      }
    }

    // Pointers
    int less  = left;  // The index of the first element of center part
    int great = right; // The index before the first element of right part

    if (order.different(a[e1], a[e2]) && order.different(a[e2], a[e3]) && order.different(a[e3], a[e4]) && order.different(a[e4], a[e5])) {
      /*
       * Use the second and fourth of the five sorted elements as pivots.
       * These values are inexpensive approximations of the first and
       * second terciles of the array. Note that pivot1 <= pivot2.
       */
      final int pivot1 = a[e2];
      final int pivot2 = a[e4];

      /*
       * The first and the last elements to be sorted are moved to the
       * locations formerly occupied by the pivots. When partitioning
       * is complete, the pivots are swapped back into their final
       * positions, and excluded from subsequent sorting.
       */
      a[e2] = a[left];
      a[e4] = a[right];

      /*
       * Skip elements, which are less or greater than pivot values.
       */
      while (order.ascending(a[++less], pivot1));
      while (order.descending(a[--great], pivot2));

      /*
       * Partitioning:
       *
       *   left part           center part                   right part
       * +--------------------------------------------------------------+
       * |  < pivot1  |  pivot1 <= && <= pivot2  |    ?    |  > pivot2  |
       * +--------------------------------------------------------------+
       *               ^                          ^       ^
       *               |                          |       |
       *              less                        k     great
       *
       * Invariants:
       *
       *              all in (left, less)   < pivot1
       *    pivot1 <= all in [less, k)     <= pivot2
       *              all in (great, right) > pivot2
       *
       * Pointer k is the first index of ?-part.
       */
      outer:
      for (int k = less - 1; ++k <= great; ) {
        int ak = a[k];
        if (order.ascending(ak, pivot1)) { // Move a[k] to left part
          a[k] = a[less];
          /*
           * Here and below we use "a[i] = b; i++;" instead
           * of "a[i++] = b;" due to performance issue.
           */
          a[less] = ak;
          ++less;
        } else if (order.descending(ak, pivot2)) { // Move a[k] to right part
          while (order.descending(a[great], pivot2)) {
            if (great-- == k) {
              break outer;
            }
          }
          if (order.ascending(a[great], pivot1)) { // a[great] <= pivot2
            a[k] = a[less];
            a[less] = a[great];
            ++less;
          } else { // pivot1 <= a[great] <= pivot2
            a[k] = a[great];
          }
          /*
           * Here and below we use "a[i] = b; i--;" instead
           * of "a[i--] = b;" due to performance issue.
           */
          a[great] = ak;
          --great;
        }
      }

      // Swap pivots into their final positions
      a[left]  = a[less  - 1]; a[less  - 1] = pivot1;
      a[right] = a[great + 1]; a[great + 1] = pivot2;

      // Sort left and right parts recursively, excluding known pivots
      sort(a, left, less - 2, order, leftmost);
      sort(a, great + 2, right, order, false);

      /*
       * If center part is too large (comprises > 4/7 of the array),
       * swap internal pivot values to ends.
       */
      if (less < e1 && e5 < great) {
        /*
         * Skip elements, which are equal to pivot values.
         */
        while (order.same(a[less], pivot1)) {
          ++less;
        }

        while (order.same(a[great], pivot2)) {
          --great;
        }

        /*
         * Partitioning:
         *
         *   left part         center part                  right part
         * +----------------------------------------------------------+
         * | == pivot1 |  pivot1 < && < pivot2  |    ?    | == pivot2 |
         * +----------------------------------------------------------+
         *              ^                        ^       ^
         *              |                        |       |
         *             less                      k     great
         *
         * Invariants:
         *
         *              all in (*,  less) == pivot1
         *     pivot1 < all in [less,  k)  < pivot2
         *              all in (great, *) == pivot2
         *
         * Pointer k is the first index of ?-part.
         */
        outer:
        for (int k = less - 1; ++k <= great; ) {
          int ak = a[k];
          if (order.same(ak, pivot1)) { // Move a[k] to left part
            a[k] = a[less];
            a[less] = ak;
            ++less;
          } else if (order.same(ak, pivot2)) { // Move a[k] to right part
            while (order.same(a[great], pivot2)) {
              if (great-- == k) {
                break outer;
              }
            }
            if (order.same(a[great], pivot1)) { // a[great] < pivot2
              a[k] = a[less];
              /*
               * Even though a[great] equals to pivot1, the
               * assignment a[less] = pivot1 may be incorrect,
               * if a[great] and pivot1 are floating-point zeros
               * of different signs. Therefore in float and
               * double sorting methods we have to use more
               * accurate assignment a[less] = a[great].
               */
              a[less] = pivot1;
              ++less;
            } else { // pivot1 < a[great] < pivot2
              a[k] = a[great];
            }
            a[great] = ak;
            --great;
          }
        }
      }

      // Sort center part recursively
      sort(a, less, great, order, false);

    } else { // Partitioning with one pivot
      /*
       * Use the third of the five sorted elements as pivot.
       * This value is inexpensive approximation of the median.
       */
      final int pivot = a[e3];

      /*
       * Partitioning degenerates to the traditional 3-way
       * (or "Dutch National Flag") schema:
       *
       *   left part    center part              right part
       * +-------------------------------------------------+
       * |  < pivot  |   == pivot   |     ?    |  > pivot  |
       * +-------------------------------------------------+
       *              ^              ^        ^
       *              |              |        |
       *             less            k      great
       *
       * Invariants:
       *
       *   all in (left, less)   < pivot
       *   all in [less, k)     == pivot
       *   all in (great, right) > pivot
       *
       * Pointer k is the first index of ?-part.
       */
      for (int k = less; k <= great; ++k) {
        if (order.same(a[k], pivot)) {
          continue;
        }
        int ak = a[k];
        if (order.ascending(ak, pivot)) { // Move a[k] to left part
          a[k] = a[less];
          a[less] = ak;
          ++less;
        } else { // a[k] > pivot - Move a[k] to right part
          while (order.descending(a[great], pivot)) {
            --great;
          }
          if (order.ascending(a[great], pivot)) { // a[great] <= pivot
            a[k] = a[less];
            a[less] = a[great];
            ++less;
          } else { // a[great] == pivot
            /*
             * Even though a[great] equals to pivot, the
             * assignment a[k] = pivot may be incorrect,
             * if a[great] and pivot are floating-point
             * zeros of different signs. Therefore in float
             * and double sorting methods we have to use
             * more accurate assignment a[k] = a[great].
             */
            a[k] = pivot;
          }
          a[great] = ak;
          --great;
        }
      }

      /*
       * Sort left and right parts recursively.
       * All elements from center part are equal
       * and, therefore, already sorted.
       */
      sort(a, left, less - 1, order, leftmost);
      sort(a, great + 1, right, order, false);
    }
  }

  /**
   * Sorts the specified array.
   *
   * @param a the array to be sorted
   * @param order sort order
   */
  public static void sort(long[] a, @NotNull LongOrdering order) {
    sort(a, 0, a.length - 1, order);
  }

  /**
   * Sorts the specified range of the array.
   *
   * @param a the array to be sorted
   * @param left the index of the first element, inclusive, to be sorted
   * @param right the index of the last element, inclusive, to be sorted
   * @param order sort order
   */
  public static void sort(long[] a, int left, int right, @NotNull LongOrdering order) {
    // Use Quicksort on small arrays
    if (right - left < QUICKSORT_THRESHOLD) {
      sort(a, left, right, order, true);
      return;
    }

    /*
     * Index run[i] is the start of i-th run
     * (ascending or descending sequence).
     */
    int[] run = new int[MAX_RUN_COUNT + 1];
    int count = 0; run[0] = left;

    // Check if the array is nearly sorted
    for (int k = left; k < right; run[count] = k) {
      switch (order.checkLong(a[k], a[k + 1])) {
      case Ascending:
        while (++k <= right && order.ascendingOrSame(a[k - 1], a[k])) ;
        break;
      case Descending:
        while (++k <= right && order.descendingOrSame(a[k - 1], a[k])) ;
        for (int lo = run[count] - 1, hi = k; ++lo < --hi; ) {
          long t = a[lo];
          a[lo] = a[hi];
          a[hi] = t;
        }
        break;
      default:
        for (int m = MAX_RUN_LENGTH; ++k <= right && order.same(a[k - 1], a[k]); ) {
          if (--m == 0) {
            sort(a, left, right, order, true);
            return;
          }
        }
        break;
      }

      /*
       * The array is not highly structured,
       * use Quicksort instead of merge sort.
       */
      if (++count == MAX_RUN_COUNT) {
        sort(a, left, right, order, true);
        return;
      }
    }

    // Check special cases
    if (run[count] == right++) { // The last run contains one element
      run[++count] = right;
    } else if (count == 1) { // The array is already sorted
      return;
    }

    /*
     * Create temporary array, which is used for merging.
     * Implementation note: variable "right" is increased by 1.
     */
    long[] b; byte odd = 0;
    for (int n = 1; (n <<= 1) < count; odd ^= 1);

    if (odd == 0) {
      b = a; a = new long[b.length];
      for (int i = left - 1; ++i < right; a[i] = b[i]);
    } else {
      b = new long[a.length];
    }

    // Merging
    for (int last; count > 1; count = last) {
      for (int k = (last = 0) + 2; k <= count; k += 2) {
        final int hi = run[k], mi = run[k - 1];
        for (int i = run[k - 2], p = i, q = mi; i < hi; ++i) {
          if (q >= hi || p < mi && order.ascendingOrSame(a[p], a[q])) {
            b[i] = a[p++];
          } else {
            b[i] = a[q++];
          }
        }
        run[++last] = hi;
      }
      if ((count & 1) != 0) {
        for (int i = right, lo = run[count - 1]; --i >= lo;
             b[i] = a[i]
        );
        run[++last] = right;
      }
      long[] t = a; a = b; b = t;
    }
  }

  /**
   * Sorts the specified range of the array by Dual-Pivot Quicksort.
   *
   * @param a the array to be sorted
   * @param left the index of the first element, inclusive, to be sorted
   * @param right the index of the last element, inclusive, to be sorted
   * @param order sort order
   * @param leftmost indicates if this part is the leftmost in the range
   */
  private static void sort(long[] a, int left, int right, @NotNull LongOrdering order, boolean leftmost) {
    final int length = right - left + 1;

    // Use insertion sort on tiny arrays
    if (length < INSERTION_SORT_THRESHOLD) {
      if (leftmost) {
        /*
         * Traditional (without sentinel) insertion sort,
         * optimized for server VM, is used in case of
         * the leftmost part.
         */
        for (int i = left, j = i; i < right; j = ++i) {
          final long ai = a[i + 1];
          while (order.ascending(ai, a[j])) {
            a[j + 1] = a[j];
            if (j-- == left) {
              break;
            }
          }
          a[j + 1] = ai;
        }
      } else {
        /*
         * Skip the longest ascending sequence.
         */
        do {
          if (left >= right) {
            return;
          }
        } while (order.descendingOrSame(a[++left], a[left - 1]));

        /*
         * Every element from adjoining part plays the role
         * of sentinel, therefore this allows us to avoid the
         * left range check on each iteration. Moreover, we use
         * the more optimized algorithm, so called pair insertion
         * sort, which is faster (in the context of Quicksort)
         * than traditional implementation of insertion sort.
         */
        for (int k = left; ++left <= right; k = ++left) {
          long a1 = a[k], a2 = a[left];

          if (order.ascending(a1, a2)) {
            a2 = a1; a1 = a[left];
          }
          while (order.ascending(a1, a[--k])) {
            a[k + 2] = a[k];
          }
          a[++k + 1] = a1;

          while (order.ascending(a2, a[--k])) {
            a[k + 1] = a[k];
          }
          a[k + 1] = a2;
        }
        final long last = a[right];

        while (order.ascending(last, a[--right])) {
          a[right + 1] = a[right];
        }
        a[right + 1] = last;
      }
      return;
    }

    // Inexpensive approximation of length / 7
    final int seventh = (length >> 3) + (length >> 6) + 1;

    /*
     * Sort five evenly spaced elements around (and including) the
     * center element in the range. These elements will be used for
     * pivot selection as described below. The choice for spacing
     * these elements was empirically determined to work well on
     * a wide variety of inputs.
     */
    final int e3 = (left + right) >>> 1; // The midpoint
    final int e2 = e3 - seventh;
    final int e1 = e2 - seventh;
    final int e4 = e3 + seventh;
    final int e5 = e4 + seventh;

    // Sort these elements using insertion sort
    if (order.ascending(a[e2], a[e1])) { final long t = a[e2]; a[e2] = a[e1]; a[e1] = t; }

    if (order.ascending(a[e3], a[e2])) { final long t = a[e3]; a[e3] = a[e2]; a[e2] = t;
      if (order.ascending(t, a[e1])) { a[e2] = a[e1]; a[e1] = t; }
    }
    if (order.ascending(a[e4], a[e3])) { final long t = a[e4]; a[e4] = a[e3]; a[e3] = t;
      if (order.ascending(t, a[e2])) { a[e3] = a[e2]; a[e2] = t;
        if (order.ascending(t, a[e1])) { a[e2] = a[e1]; a[e1] = t; }
      }
    }
    if (order.ascending(a[e5], a[e4])) { final long t = a[e5]; a[e5] = a[e4]; a[e4] = t;
      if (order.ascending(t, a[e3])) { a[e4] = a[e3]; a[e3] = t;
        if (order.ascending(t, a[e2])) { a[e3] = a[e2]; a[e2] = t;
          if (order.ascending(t, a[e1])) { a[e2] = a[e1]; a[e1] = t; }
        }
      }
    }

    // Pointers
    int less  = left;  // The index of the first element of center part
    int great = right; // The index before the first element of right part

    if (order.different(a[e1], a[e2]) && order.different(a[e2], a[e3]) && order.different(a[e3], a[e4]) && order.different(a[e4], a[e5])) {
      /*
       * Use the second and fourth of the five sorted elements as pivots.
       * These values are inexpensive approximations of the first and
       * second terciles of the array. Note that pivot1 <= pivot2.
       */
      final long pivot1 = a[e2];
      final long pivot2 = a[e4];

      /*
       * The first and the last elements to be sorted are moved to the
       * locations formerly occupied by the pivots. When partitioning
       * is complete, the pivots are swapped back into their final
       * positions, and excluded from subsequent sorting.
       */
      a[e2] = a[left];
      a[e4] = a[right];

      /*
       * Skip elements, which are less or greater than pivot values.
       */
      while (order.ascending(a[++less], pivot1));
      while (order.descending(a[--great], pivot2));

      /*
       * Partitioning:
       *
       *   left part           center part                   right part
       * +--------------------------------------------------------------+
       * |  < pivot1  |  pivot1 <= && <= pivot2  |    ?    |  > pivot2  |
       * +--------------------------------------------------------------+
       *               ^                          ^       ^
       *               |                          |       |
       *              less                        k     great
       *
       * Invariants:
       *
       *              all in (left, less)   < pivot1
       *    pivot1 <= all in [less, k)     <= pivot2
       *              all in (great, right) > pivot2
       *
       * Pointer k is the first index of ?-part.
       */
      outer:
      for (int k = less - 1; ++k <= great; ) {
        final long ak = a[k];
        if (order.ascending(ak,  pivot1)) { // Move a[k] to left part
          a[k] = a[less];
          /*
           * Here and below we use "a[i] = b; i++;" instead
           * of "a[i++] = b;" due to performance issue.
           */
          a[less] = ak;
          ++less;
        } else if (order.descending(ak, pivot2)) { // Move a[k] to right part
          while (order.descending(a[great], pivot2)) {
            if (great-- == k) {
              break outer;
            }
          }
          if (order.ascending(a[great], pivot1)) { // a[great] <= pivot2
            a[k] = a[less];
            a[less] = a[great];
            ++less;
          } else { // pivot1 <= a[great] <= pivot2
            a[k] = a[great];
          }
          /*
           * Here and below we use "a[i] = b; i--;" instead
           * of "a[i--] = b;" due to performance issue.
           */
          a[great] = ak;
          --great;
        }
      }

      // Swap pivots into their final positions
      a[left]  = a[less  - 1]; a[less  - 1] = pivot1;
      a[right] = a[great + 1]; a[great + 1] = pivot2;

      // Sort left and right parts recursively, excluding known pivots
      sort(a, left, less - 2, order, leftmost);
      sort(a, great + 2, right, order, false);

      /*
       * If center part is too large (comprises > 4/7 of the array),
       * swap internal pivot values to ends.
       */
      if (less < e1 && e5 < great) {
        /*
         * Skip elements, which are equal to pivot values.
         */
        while (order.same(a[less], pivot1)) {
          ++less;
        }

        while (order.same(a[great], pivot2)) {
          --great;
        }

        /*
         * Partitioning:
         *
         *   left part         center part                  right part
         * +----------------------------------------------------------+
         * | == pivot1 |  pivot1 < && < pivot2  |    ?    | == pivot2 |
         * +----------------------------------------------------------+
         *              ^                        ^       ^
         *              |                        |       |
         *             less                      k     great
         *
         * Invariants:
         *
         *              all in (*,  less) == pivot1
         *     pivot1 < all in [less,  k)  < pivot2
         *              all in (great, *) == pivot2
         *
         * Pointer k is the first index of ?-part.
         */
        outer:
        for (int k = less - 1; ++k <= great; ) {
          final long ak = a[k];
          if (order.same(ak, pivot1)) { // Move a[k] to left part
            a[k] = a[less];
            a[less] = ak;
            ++less;
          } else if (order.same(ak, pivot2)) { // Move a[k] to right part
            while (order.same(a[great], pivot2)) {
              if (great-- == k) {
                break outer;
              }
            }
            if (order.same(a[great], pivot1)) { // a[great] < pivot2
              a[k] = a[less];
              /*
               * Even though a[great] equals to pivot1, the
               * assignment a[less] = pivot1 may be incorrect,
               * if a[great] and pivot1 are floating-point zeros
               * of different signs. Therefore in float and
               * double sorting methods we have to use more
               * accurate assignment a[less] = a[great].
               */
              a[less] = pivot1;
              ++less;
            } else { // pivot1 < a[great] < pivot2
              a[k] = a[great];
            }
            a[great] = ak;
            --great;
          }
        }
      }

      // Sort center part recursively
      sort(a, less, great, order, false);

    } else { // Partitioning with one pivot
      /*
       * Use the third of the five sorted elements as pivot.
       * This value is inexpensive approximation of the median.
       */
      final long pivot = a[e3];

      /*
       * Partitioning degenerates to the traditional 3-way
       * (or "Dutch National Flag") schema:
       *
       *   left part    center part              right part
       * +-------------------------------------------------+
       * |  < pivot  |   == pivot   |     ?    |  > pivot  |
       * +-------------------------------------------------+
       *              ^              ^        ^
       *              |              |        |
       *             less            k      great
       *
       * Invariants:
       *
       *   all in (left, less)   < pivot
       *   all in [less, k)     == pivot
       *   all in (great, right) > pivot
       *
       * Pointer k is the first index of ?-part.
       */
      for (int k = less; k <= great; ++k) {
        if (order.same(a[k], pivot)) {
          continue;
        }
        final long ak = a[k];
        if (order.ascending(ak, pivot)) { // Move a[k] to left part
          a[k] = a[less];
          a[less] = ak;
          ++less;
        } else { // a[k] > pivot - Move a[k] to right part
          while (order.descending(a[great], pivot)) {
            --great;
          }
          if (order.ascending(a[great], pivot)) { // a[great] <= pivot
            a[k] = a[less];
            a[less] = a[great];
            ++less;
          } else { // a[great] == pivot
            /*
             * Even though a[great] equals to pivot, the
             * assignment a[k] = pivot may be incorrect,
             * if a[great] and pivot are floating-point
             * zeros of different signs. Therefore in float
             * and double sorting methods we have to use
             * more accurate assignment a[k] = a[great].
             */
            a[k] = pivot;
          }
          a[great] = ak;
          --great;
        }
      }

      /*
       * Sort left and right parts recursively.
       * All elements from center part are equal
       * and, therefore, already sorted.
       */
      sort(a, left, less - 1, order, leftmost);
      sort(a, great + 1, right, order, false);
    }
  }

  /**
   * Sorts the specified array.
   *
   * @param a the array to be sorted
   * @param order sort order
   */
  public static void sort(@NotNull short[] a, @NotNull ShortOrdering order) {
    sort(a, 0, a.length - 1, order);
  }

  /**
   * Sorts the specified range of the array.
   *
   * @param a the array to be sorted
   * @param left the index of the first element, inclusive, to be sorted
   * @param right the index of the last element, inclusive, to be sorted
   * @param order sort order
   */
  public static void sort(@NotNull short[] a, int left, int right, @NotNull ShortOrdering order) {
    // Use counting sort on large arrays
    if (right - left > COUNTING_SORT_THRESHOLD_FOR_SHORT_OR_CHAR) {
      // todo: having an order is making counting sort a bit more complex,
      //       so the threshold is probably wrong

      // Count and collect the values which appear in unique, then sort it and add
      // elements by count to the original array. Fallback to standard sorting if
      // all values are unique because then we can just sort the original array.
      final int length = right - left + 1;
      final int[] count = new int[NUM_SHORT_VALUES];
      final short[] unique = new short[Math.min(NUM_SHORT_VALUES, length)];
      int uCount = 0;
      for (int i = left;  i <= right;  ++i) {
        if (count[a[i] - Short.MIN_VALUE]++ == 0) {
          unique[uCount++] = a[i];
        }
      }
      if (uCount < length) {
        doSort(unique, 0, uCount - 1, order);

        int idx = left;
        for (int i = 0;  i < uCount;  ++i) {
          final short v = unique[i];
          int cnt = count[v - Short.MIN_VALUE];
          assert cnt > 0;
          while (--cnt >= 0) {
            a[idx++] = v;
          }
        }
        return;
      }
      // else: counting didn't help as each value only appeared once, so instead of sorting unique sort the array
    }
    // Use Dual-Pivot Quicksort on small arrays
    doSort(a, left, right, order);
  }

  /** The number of distinct short values. */
  private static final int NUM_SHORT_VALUES = 1 << 16;

  /**
   * Sorts the specified range of the array.
   *
   * @param a the array to be sorted
   * @param left the index of the first element, inclusive, to be sorted
   * @param right the index of the last element, inclusive, to be sorted
   * @param order order to apply
   */
  private static void doSort(short[] a, int left, int right, @NotNull ShortOrdering order) {
    // Use Quicksort on small arrays
    if (right - left < QUICKSORT_THRESHOLD) {
      sort(a, left, right, order, true);
      return;
    }

    /*
     * Index run[i] is the start of i-th run
     * (ascending or descending sequence).
     */
    final int[] run = new int[MAX_RUN_COUNT + 1];
    int count = 0; run[0] = left;

    // Check if the array is nearly sorted
    for (int k = left; k < right; run[count] = k) {
      switch (order.checkShort(a[k], a[k + 1])) {
      case Ascending:
        while (++k <= right && order.ascendingOrSame(a[k - 1], a[k]));
        break;
      case Descending:
        while (++k <= right && a[k - 1] >= a[k]);
        for (int lo = run[count] - 1, hi = k; ++lo < --hi; ) {
          final short t = a[lo]; a[lo] = a[hi]; a[hi] = t;
        }
        break;
      default:
        for (int m = MAX_RUN_LENGTH; ++k <= right && a[k - 1] == a[k]; ) {
          if (--m == 0) {
            sort(a, left, right, order, true);
            return;
          }
        }
        break;
      }

      /*
       * The array is not highly structured,
       * use Quicksort instead of merge sort.
       */
      if (++count == MAX_RUN_COUNT) {
        sort(a, left, right, order, true);
        return;
      }
    }

    // Check special cases
    if (run[count] == right++) { // The last run contains one element
      run[++count] = right;
    } else if (count == 1) { // The array is already sorted
      return;
    }

    /*
     * Create temporary array, which is used for merging.
     * Implementation note: variable "right" is increased by 1.
     */
    short[] b; byte odd = 0;
    for (int n = 1; (n <<= 1) < count; odd ^= 1);

    if (odd == 0) {
      b = a; a = new short[b.length];
      for (int i = left - 1; ++i < right; a[i] = b[i]);
    } else {
      b = new short[a.length];
    }

    // Merging
    for (int last; count > 1; count = last) {
      for (int k = (last = 0) + 2; k <= count; k += 2) {
        int hi = run[k], mi = run[k - 1];
        for (int i = run[k - 2], p = i, q = mi; i < hi; ++i) {
          if (q >= hi || p < mi && order.ascendingOrSame(a[p], a[q])) {
            b[i] = a[p++];
          } else {
            b[i] = a[q++];
          }
        }
        run[++last] = hi;
      }
      if ((count & 1) != 0) {
        for (int i = right, lo = run[count - 1]; --i >= lo;
             b[i] = a[i]
        );
        run[++last] = right;
      }
      short[] t = a; a = b; b = t;
    }
  }

  /**
   * Sorts the specified range of the array by Dual-Pivot Quicksort.
   *
   * @param a the array to be sorted
   * @param left the index of the first element, inclusive, to be sorted
   * @param right the index of the last element, inclusive, to be sorted
   * @param order sort order
   * @param leftmost indicates if this part is the leftmost in the range
   */
  private static void sort(short[] a, int left, int right, @NotNull ShortOrdering order, boolean leftmost) {
    final int length = right - left + 1;

    // Use insertion sort on tiny arrays
    if (length < INSERTION_SORT_THRESHOLD) {
      if (leftmost) {
        /*
         * Traditional (without sentinel) insertion sort,
         * optimized for server VM, is used in case of
         * the leftmost part.
         */
        for (int i = left, j = i; i < right; j = ++i) {
          final short ai = a[i + 1];
          while (order.ascending(ai , a[j])) {
            a[j + 1] = a[j];
            if (j-- == left) {
              break;
            }
          }
          a[j + 1] = ai;
        }
      } else {
        /*
         * Skip the longest ascending sequence.
         */
        do {
          if (left >= right) {
            return;
          }
        } while (order.descendingOrSame(a[++left], a[left - 1]));

        /*
         * Every element from adjoining part plays the role
         * of sentinel, therefore this allows us to avoid the
         * left range check on each iteration. Moreover, we use
         * the more optimized algorithm, so called pair insertion
         * sort, which is faster (in the context of Quicksort)
         * than traditional implementation of insertion sort.
         */
        for (int k = left; ++left <= right; k = ++left) {
          short a1 = a[k], a2 = a[left];

          if (order.ascending(a1, a2)) {
            a2 = a1; a1 = a[left];
          }
          while (order.ascending(a1, a[--k])) {
            a[k + 2] = a[k];
          }
          a[++k + 1] = a1;

          while (order.ascending(a2, a[--k])) {
            a[k + 1] = a[k];
          }
          a[k + 1] = a2;
        }
        final short last = a[right];

        while (order.ascending(last, a[--right])) {
          a[right + 1] = a[right];
        }
        a[right + 1] = last;
      }
      return;
    }

    // Inexpensive approximation of length / 7
    final int seventh = (length >> 3) + (length >> 6) + 1;

    /*
     * Sort five evenly spaced elements around (and including) the
     * center element in the range. These elements will be used for
     * pivot selection as described below. The choice for spacing
     * these elements was empirically determined to work well on
     * a wide variety of inputs.
     */
    final int e3 = (left + right) >>> 1; // The midpoint
    final int e2 = e3 - seventh;
    final int e1 = e2 - seventh;
    final int e4 = e3 + seventh;
    final int e5 = e4 + seventh;

    // Sort these elements using insertion sort
    if (order.ascending(a[e2], a[e1])) { final short t = a[e2]; a[e2] = a[e1]; a[e1] = t; }

    if (order.ascending(a[e3], a[e2])) { final short t = a[e3]; a[e3] = a[e2]; a[e2] = t;
      if (order.ascending(t, a[e1])) { a[e2] = a[e1]; a[e1] = t; }
    }
    if (order.ascending(a[e4], a[e3])) { final short t = a[e4]; a[e4] = a[e3]; a[e3] = t;
      if (order.ascending(t, a[e2])) { a[e3] = a[e2]; a[e2] = t;
        if (order.ascending(t, a[e1])) { a[e2] = a[e1]; a[e1] = t; }
      }
    }
    if (order.ascending(a[e5], a[e4])) { final short t = a[e5]; a[e5] = a[e4]; a[e4] = t;
      if (order.ascending(t, a[e3])) { a[e4] = a[e3]; a[e3] = t;
        if (order.ascending(t, a[e2])) { a[e3] = a[e2]; a[e2] = t;
          if (order.ascending(t, a[e1])) { a[e2] = a[e1]; a[e1] = t; }
        }
      }
    }

    // Pointers
    int less  = left;  // The index of the first element of center part
    int great = right; // The index before the first element of right part

    if (order.different(a[e1], a[e2]) && order.different(a[e2], a[e3]) && order.different(a[e3], a[e4]) && order.different(a[e4], a[e5])) {
      /*
       * Use the second and fourth of the five sorted elements as pivots.
       * These values are inexpensive approximations of the first and
       * second terciles of the array. Note that pivot1 <= pivot2.
       */
      final short pivot1 = a[e2];
      final short pivot2 = a[e4];

      /*
       * The first and the last elements to be sorted are moved to the
       * locations formerly occupied by the pivots. When partitioning
       * is complete, the pivots are swapped back into their final
       * positions, and excluded from subsequent sorting.
       */
      a[e2] = a[left];
      a[e4] = a[right];

      /*
       * Skip elements, which are less or greater than pivot values.
       */
      while (order.ascending(a[++less], pivot1));
      while (order.descending(a[--great], pivot2));

      /*
       * Partitioning:
       *
       *   left part           center part                   right part
       * +--------------------------------------------------------------+
       * |  < pivot1  |  pivot1 <= && <= pivot2  |    ?    |  > pivot2  |
       * +--------------------------------------------------------------+
       *               ^                          ^       ^
       *               |                          |       |
       *              less                        k     great
       *
       * Invariants:
       *
       *              all in (left, less)   < pivot1
       *    pivot1 <= all in [less, k)     <= pivot2
       *              all in (great, right) > pivot2
       *
       * Pointer k is the first index of ?-part.
       */
      outer:
      for (int k = less - 1; ++k <= great; ) {
        final short ak = a[k];
        if (order.ascending(ak, pivot1)) { // Move a[k] to left part
          a[k] = a[less];
          /*
           * Here and below we use "a[i] = b; i++;" instead
           * of "a[i++] = b;" due to performance issue.
           */
          a[less] = ak;
          ++less;
        } else if (order.descending(ak, pivot2)) { // Move a[k] to right part
          while (order.descending(a[great], pivot2)) {
            if (great-- == k) {
              break outer;
            }
          }
          if (order.ascending(a[great], pivot1)) { // a[great] <= pivot2
            a[k] = a[less];
            a[less] = a[great];
            ++less;
          } else { // pivot1 <= a[great] <= pivot2
            a[k] = a[great];
          }
          /*
           * Here and below we use "a[i] = b; i--;" instead
           * of "a[i--] = b;" due to performance issue.
           */
          a[great] = ak;
          --great;
        }
      }

      // Swap pivots into their final positions
      a[left]  = a[less  - 1]; a[less  - 1] = pivot1;
      a[right] = a[great + 1]; a[great + 1] = pivot2;

      // Sort left and right parts recursively, excluding known pivots
      sort(a, left, less - 2, order, leftmost);
      sort(a, great + 2, right, order, false);

      /*
       * If center part is too large (comprises > 4/7 of the array),
       * swap internal pivot values to ends.
       */
      if (less < e1 && e5 < great) {
        /*
         * Skip elements, which are equal to pivot values.
         */
        while (order.same(a[less], pivot1)) {
          ++less;
        }

        while (order.same(a[great], pivot2)) {
          --great;
        }

        /*
         * Partitioning:
         *
         *   left part         center part                  right part
         * +----------------------------------------------------------+
         * | == pivot1 |  pivot1 < && < pivot2  |    ?    | == pivot2 |
         * +----------------------------------------------------------+
         *              ^                        ^       ^
         *              |                        |       |
         *             less                      k     great
         *
         * Invariants:
         *
         *              all in (*,  less) == pivot1
         *     pivot1 < all in [less,  k)  < pivot2
         *              all in (great, *) == pivot2
         *
         * Pointer k is the first index of ?-part.
         */
        outer:
        for (int k = less - 1; ++k <= great; ) {
          final short ak = a[k];
          if (order.same(ak, pivot1)) { // Move a[k] to left part
            a[k] = a[less];
            a[less] = ak;
            ++less;
          } else if (order.same(ak, pivot2)) { // Move a[k] to right part
            while (order.same(a[great], pivot2)) {
              if (great-- == k) {
                break outer;
              }
            }
            if (order.same(a[great], pivot1)) { // a[great] < pivot2
              a[k] = a[less];
              /*
               * Even though a[great] equals to pivot1, the
               * assignment a[less] = pivot1 may be incorrect,
               * if a[great] and pivot1 are floating-point zeros
               * of different signs. Therefore in float and
               * double sorting methods we have to use more
               * accurate assignment a[less] = a[great].
               */
              a[less] = pivot1;
              ++less;
            } else { // pivot1 < a[great] < pivot2
              a[k] = a[great];
            }
            a[great] = ak;
            --great;
          }
        }
      }

      // Sort center part recursively
      sort(a, less, great, order, false);

    } else { // Partitioning with one pivot
      /*
       * Use the third of the five sorted elements as pivot.
       * This value is inexpensive approximation of the median.
       */
      final short pivot = a[e3];

      /*
       * Partitioning degenerates to the traditional 3-way
       * (or "Dutch National Flag") schema:
       *
       *   left part    center part              right part
       * +-------------------------------------------------+
       * |  < pivot  |   == pivot   |     ?    |  > pivot  |
       * +-------------------------------------------------+
       *              ^              ^        ^
       *              |              |        |
       *             less            k      great
       *
       * Invariants:
       *
       *   all in (left, less)   < pivot
       *   all in [less, k)     == pivot
       *   all in (great, right) > pivot
       *
       * Pointer k is the first index of ?-part.
       */
      for (int k = less; k <= great; ++k) {
        if (order.same(a[k], pivot)) {
          continue;
        }
        final short ak = a[k];
        if (order.ascending(ak, pivot)) { // Move a[k] to left part
          a[k] = a[less];
          a[less] = ak;
          ++less;
        } else { // a[k] > pivot - Move a[k] to right part
          while (order.descending(a[great], pivot)) {
            --great;
          }
          if (order.ascending(a[great], pivot)) { // a[great] <= pivot
            a[k] = a[less];
            a[less] = a[great];
            ++less;
          } else { // a[great] == pivot
            /*
             * Even though a[great] equals to pivot, the
             * assignment a[k] = pivot may be incorrect,
             * if a[great] and pivot are floating-point
             * zeros of different signs. Therefore in float
             * and double sorting methods we have to use
             * more accurate assignment a[k] = a[great].
             */
            a[k] = pivot;
          }
          a[great] = ak;
          --great;
        }
      }

      /*
       * Sort left and right parts recursively.
       * All elements from center part are equal
       * and, therefore, already sorted.
       */
      sort(a, left, less - 1, order, leftmost);
      sort(a, great + 1, right, order, false);
    }
  }

  /**
   * Sorts the specified array.
   *
   * @param a the array to be sorted
   * @param order sort order
   */
  public static void sort(@NotNull char[] a, @NotNull CharOrdering order) {
    sort(a, 0, a.length - 1, order);
  }

  /**
   * Sorts the specified range of the array.
   *
   * @param a the array to be sorted
   * @param left the index of the first element, inclusive, to be sorted
   * @param right the index of the last element, inclusive, to be sorted
   * @param order sort order
   */
  public static void sort(@NotNull char[] a, int left, int right, @NotNull CharOrdering order) {
    // Use counting sort on large arrays
    if (right - left > COUNTING_SORT_THRESHOLD_FOR_SHORT_OR_CHAR) {
      // todo: having an order is making counting sort a bit more complex,
      //       so the threshold is probably wrong

      // Count and collect the values which appear in unique, then sort it and add
      // elements by count to the original array. Fallback to standard sorting if
      // all values are unique because then we can just sort the original array.
      final int length = right - left + 1;
      final int[] count = new int[NUM_CHAR_VALUES];
      final char[] unique = new char[Math.min(NUM_CHAR_VALUES, length)];
      int uCount = 0;
      for (int i = left;  i <= right;  ++i) {
        if (count[a[i]]++ == 0) {
          unique[uCount++] = a[i];
        }
      }
      if (uCount < length) {
        doSort(unique, 0, uCount - 1, order);

        int idx = left;
        for (int i = 0;  i < uCount;  ++i) {
          final char v = unique[i];
          int cnt = count[v];
          assert cnt > 0;
          while (--cnt >= 0) {
            a[idx++] = v;
          }
        }
        return;
      }
      // else: counting didn't help as each value only appeared once, so instead of sorting `unique` sort the array a directly
    }
    // Use Dual-Pivot Quicksort on small arrays or arrays where each value only appears once
    doSort(a, left, right, order);
  }

  /** The number of distinct char values. */
  private static final int NUM_CHAR_VALUES = 1 << 16;

  /**
   * Sorts the specified range of the array.
   *
   * @param a the array to be sorted
   * @param left the index of the first element, inclusive, to be sorted
   * @param right the index of the last element, inclusive, to be sorted
   * @param order sort order
   */
  private static void doSort(@NotNull char[] a, int left, int right, @NotNull CharOrdering order) {
    // Use Quicksort on small arrays
    if (right - left < QUICKSORT_THRESHOLD) {
      sort(a, left, right, order, true);
      return;
    }

    /*
     * Index run[i] is the start of i-th run
     * (ascending or descending sequence).
     */
    final int[] run = new int[MAX_RUN_COUNT + 1];
    int count = 0; run[0] = left;

    // Check if the array is nearly sorted
    for (int k = left; k < right; run[count] = k) {
      switch (order.checkChar(a[k], a[k + 1])) {
      case Ascending:
        while (++k <= right && order.ascendingOrSame(a[k - 1], a[k]));
        break;
      case Descending:
        while (++k <= right && order.descendingOrSame(a[k - 1], a[k]));
        for (int lo = run[count] - 1, hi = k; ++lo < --hi; ) {
          final char t = a[lo]; a[lo] = a[hi]; a[hi] = t;
        }
        break;
      default:
        for (int m = MAX_RUN_LENGTH; ++k <= right && order.same(a[k - 1], a[k]); ) {
          if (--m == 0) {
            sort(a, left, right, order, true);
            return;
          }
        }
      }

      /*
       * The array is not highly structured,
       * use Quicksort instead of merge sort.
       */
      if (++count == MAX_RUN_COUNT) {
        sort(a, left, right, order, true);
        return;
      }
    }

    // Check special cases
    if (run[count] == right++) { // The last run contains one element
      run[++count] = right;
    } else if (count == 1) { // The array is already sorted
      return;
    }

    /*
     * Create temporary array, which is used for merging.
     * Implementation note: variable "right" is increased by 1.
     */
    char[] b; byte odd = 0;
    for (int n = 1; (n <<= 1) < count; odd ^= 1);

    if (odd == 0) {
      b = a; a = new char[b.length];
      for (int i = left - 1; ++i < right; a[i] = b[i]);
    } else {
      b = new char[a.length];
    }

    // Merging
    for (int last; count > 1; count = last) {
      for (int k = (last = 0) + 2; k <= count; k += 2) {
        int hi = run[k], mi = run[k - 1];
        for (int i = run[k - 2], p = i, q = mi; i < hi; ++i) {
          if (q >= hi || p < mi && order.ascendingOrSame(a[p], a[q])) {
            b[i] = a[p++];
          } else {
            b[i] = a[q++];
          }
        }
        run[++last] = hi;
      }
      if ((count & 1) != 0) {
        for (int i = right, lo = run[count - 1]; --i >= lo;
             b[i] = a[i]
        );
        run[++last] = right;
      }
      final char[] t = a; a = b; b = t;
    }
  }

  /**
   * Sorts the specified range of the array by Dual-Pivot Quicksort.
   *
   * @param a the array to be sorted
   * @param left the index of the first element, inclusive, to be sorted
   * @param right the index of the last element, inclusive, to be sorted
   * @param leftmost indicates if this part is the leftmost in the range
   */
  private static void sort(@NotNull char[] a, int left, int right, @NotNull CharOrdering order, boolean leftmost) {
    final int length = right - left + 1;

    // Use insertion sort on tiny arrays
    if (length < INSERTION_SORT_THRESHOLD) {
      if (leftmost) {
        /*
         * Traditional (without sentinel) insertion sort,
         * optimized for server VM, is used in case of
         * the leftmost part.
         */
        for (int i = left, j = i; i < right; j = ++i) {
          final char ai = a[i + 1];
          while (order.ascending(ai, a[j])) {
            a[j + 1] = a[j];
            if (j-- == left) {
              break;
            }
          }
          a[j + 1] = ai;
        }
      } else {
        /*
         * Skip the longest ascending sequence.
         */
        do {
          if (left >= right) {
            return;
          }
        } while (order.descendingOrSame(a[++left], a[left - 1]));

        /*
         * Every element from adjoining part plays the role
         * of sentinel, therefore this allows us to avoid the
         * left range check on each iteration. Moreover, we use
         * the more optimized algorithm, so called pair insertion
         * sort, which is faster (in the context of Quicksort)
         * than traditional implementation of insertion sort.
         */
        for (int k = left; ++left <= right; k = ++left) {
          char a1 = a[k], a2 = a[left];

          if (order.ascending(a1, a2)) {
            a2 = a1; a1 = a[left];
          }
          while (order.ascending(a1, a[--k])) {
            a[k + 2] = a[k];
          }
          a[++k + 1] = a1;

          while (order.ascending(a2, a[--k])) {
            a[k + 1] = a[k];
          }
          a[k + 1] = a2;
        }
        final char last = a[right];

        while (order.ascending(last, a[--right])) {
          a[right + 1] = a[right];
        }
        a[right + 1] = last;
      }
      return;
    }

    // Inexpensive approximation of length / 7
    final int seventh = (length >> 3) + (length >> 6) + 1;

    /*
     * Sort five evenly spaced elements around (and including) the
     * center element in the range. These elements will be used for
     * pivot selection as described below. The choice for spacing
     * these elements was empirically determined to work well on
     * a wide variety of inputs.
     */
    final int e3 = (left + right) >>> 1; // The midpoint
    final int e2 = e3 - seventh;
    final int e1 = e2 - seventh;
    final int e4 = e3 + seventh;
    final int e5 = e4 + seventh;

    // Sort these elements using insertion sort
    if (order.ascending(a[e2], a[e1])) { final char t = a[e2]; a[e2] = a[e1]; a[e1] = t; }

    if (order.ascending(a[e3], a[e2])) { final char t = a[e3]; a[e3] = a[e2]; a[e2] = t;
      if (order.ascending(t, a[e1])) { a[e2] = a[e1]; a[e1] = t; }
    }
    if (order.ascending(a[e4], a[e3])) { final char t = a[e4]; a[e4] = a[e3]; a[e3] = t;
      if (order.ascending(t, a[e2])) { a[e3] = a[e2]; a[e2] = t;
        if (order.ascending(t, a[e1])) { a[e2] = a[e1]; a[e1] = t; }
      }
    }
    if (order.ascending(a[e5], a[e4])) { final char t = a[e5]; a[e5] = a[e4]; a[e4] = t;
      if (order.ascending(t, a[e3])) { a[e4] = a[e3]; a[e3] = t;
        if (order.ascending(t, a[e2])) { a[e3] = a[e2]; a[e2] = t;
          if (order.ascending(t, a[e1])) { a[e2] = a[e1]; a[e1] = t; }
        }
      }
    }

    // Pointers
    int less  = left;  // The index of the first element of center part
    int great = right; // The index before the first element of right part

    if (order.different(a[e1], a[e2]) && order.different(a[e2], a[e3]) && order.different(a[e3], a[e4]) && order.different(a[e4], a[e5])) {
      /*
       * Use the second and fourth of the five sorted elements as pivots.
       * These values are inexpensive approximations of the first and
       * second terciles of the array. Note that pivot1 <= pivot2.
       */
      final char pivot1 = a[e2];
      final char pivot2 = a[e4];

      /*
       * The first and the last elements to be sorted are moved to the
       * locations formerly occupied by the pivots. When partitioning
       * is complete, the pivots are swapped back into their final
       * positions, and excluded from subsequent sorting.
       */
      a[e2] = a[left];
      a[e4] = a[right];

      /*
       * Skip elements, which are less or greater than pivot values.
       */
      while (order.ascending(a[++less], pivot1));
      while (order.descending(a[--great], pivot2));

      /*
       * Partitioning:
       *
       *   left part           center part                   right part
       * +--------------------------------------------------------------+
       * |  < pivot1  |  pivot1 <= && <= pivot2  |    ?    |  > pivot2  |
       * +--------------------------------------------------------------+
       *               ^                          ^       ^
       *               |                          |       |
       *              less                        k     great
       *
       * Invariants:
       *
       *              all in (left, less)   < pivot1
       *    pivot1 <= all in [less, k)     <= pivot2
       *              all in (great, right) > pivot2
       *
       * Pointer k is the first index of ?-part.
       */
      outer:
      for (int k = less - 1; ++k <= great; ) {
        final char ak = a[k];
        if (order.ascending(ak, pivot1)) { // Move a[k] to left part
          a[k] = a[less];
          /*
           * Here and below we use "a[i] = b; i++;" instead
           * of "a[i++] = b;" due to performance issue.
           */
          a[less] = ak;
          ++less;
        } else if (order.descending(ak, pivot2)) { // Move a[k] to right part
          while (order.descending(a[great], pivot2)) {
            if (great-- == k) {
              break outer;
            }
          }
          if (order.ascending(a[great], pivot1)) { // a[great] <= pivot2
            a[k] = a[less];
            a[less] = a[great];
            ++less;
          } else { // pivot1 <= a[great] <= pivot2
            a[k] = a[great];
          }
          /*
           * Here and below we use "a[i] = b; i--;" instead
           * of "a[i--] = b;" due to performance issue.
           */
          a[great] = ak;
          --great;
        }
      }

      // Swap pivots into their final positions
      a[left]  = a[less  - 1]; a[less  - 1] = pivot1;
      a[right] = a[great + 1]; a[great + 1] = pivot2;

      // Sort left and right parts recursively, excluding known pivots
      sort(a, left, less - 2, order, leftmost);
      sort(a, great + 2, right, order, false);

      /*
       * If center part is too large (comprises > 4/7 of the array),
       * swap internal pivot values to ends.
       */
      if (less < e1 && e5 < great) {
        /*
         * Skip elements, which are equal to pivot values.
         */
        while (order.same(a[less], pivot1)) {
          ++less;
        }

        while (order.same(a[great], pivot2)) {
          --great;
        }

        /*
         * Partitioning:
         *
         *   left part         center part                  right part
         * +----------------------------------------------------------+
         * | == pivot1 |  pivot1 < && < pivot2  |    ?    | == pivot2 |
         * +----------------------------------------------------------+
         *              ^                        ^       ^
         *              |                        |       |
         *             less                      k     great
         *
         * Invariants:
         *
         *              all in (*,  less) == pivot1
         *     pivot1 < all in [less,  k)  < pivot2
         *              all in (great, *) == pivot2
         *
         * Pointer k is the first index of ?-part.
         */
        outer:
        for (int k = less - 1; ++k <= great; ) {
          final char ak = a[k];
          if (order.same(ak, pivot1)) { // Move a[k] to left part
            a[k] = a[less];
            a[less] = ak;
            ++less;
          } else if (order.same(ak, pivot2)) { // Move a[k] to right part
            while (order.same(a[great], pivot2)) {
              if (great-- == k) {
                break outer;
              }
            }
            if (order.same(a[great], pivot1)) { // a[great] < pivot2
              a[k] = a[less];
              /*
               * Even though a[great] equals to pivot1, the
               * assignment a[less] = pivot1 may be incorrect,
               * if a[great] and pivot1 are floating-point zeros
               * of different signs. Therefore in float and
               * double sorting methods we have to use more
               * accurate assignment a[less] = a[great].
               */
              a[less] = pivot1;
              ++less;
            } else { // pivot1 < a[great] < pivot2
              a[k] = a[great];
            }
            a[great] = ak;
            --great;
          }
        }
      }

      // Sort center part recursively
      sort(a, less, great, order, false);

    } else { // Partitioning with one pivot
      /*
       * Use the third of the five sorted elements as pivot.
       * This value is inexpensive approximation of the median.
       */
      final char pivot = a[e3];

      /*
       * Partitioning degenerates to the traditional 3-way
       * (or "Dutch National Flag") schema:
       *
       *   left part    center part              right part
       * +-------------------------------------------------+
       * |  < pivot  |   == pivot   |     ?    |  > pivot  |
       * +-------------------------------------------------+
       *              ^              ^        ^
       *              |              |        |
       *             less            k      great
       *
       * Invariants:
       *
       *   all in (left, less)   < pivot
       *   all in [less, k)     == pivot
       *   all in (great, right) > pivot
       *
       * Pointer k is the first index of ?-part.
       */
      for (int k = less; k <= great; ++k) {
        if (order.same(a[k], pivot)) {
          continue;
        }
        final char ak = a[k];
        if (order.ascending(ak, pivot)) { // Move a[k] to left part
          a[k] = a[less];
          a[less] = ak;
          ++less;
        } else { // a[k] > pivot - Move a[k] to right part
          while (order.descending(a[great], pivot)) {
            --great;
          }
          if (order.ascending(a[great], pivot)) { // a[great] <= pivot
            a[k] = a[less];
            a[less] = a[great];
            ++less;
          } else { // a[great] == pivot
            /*
             * Even though a[great] equals to pivot, the
             * assignment a[k] = pivot may be incorrect,
             * if a[great] and pivot are floating-point
             * zeros of different signs. Therefore in float
             * and double sorting methods we have to use
             * more accurate assignment a[k] = a[great].
             */
            a[k] = pivot;
          }
          a[great] = ak;
          --great;
        }
      }

      /*
       * Sort left and right parts recursively.
       * All elements from center part are equal
       * and, therefore, already sorted.
       */
      sort(a, left, less - 1, order, leftmost);
      sort(a, great + 1, right, order, false);
    }
  }

  /** The number of distinct byte values. */
  private static final int NUM_BYTE_VALUES = 1 << 8;

  /**
   * Sorts the specified array in the specified order.
   *
   * @param a the array to be sorted
   * @param order sort order
   */
  public static void sort(@NotNull byte[] a, @NotNull ByteOrdering order) {
    sort(a, 0, a.length - 1, order);
  }

  /**
   * Sorts the specified range of the array in the specified order.
   *
   * @param a the array to be sorted
   * @param left the index of the first element, inclusive, to be sorted
   * @param right the index of the last element, inclusive, to be sorted
   * @param order sort order
   */
  public static void sort(@NotNull byte[] a, int left, int right, @NotNull ByteOrdering order) {
    // Use counting sort on large arrays
    if (right - left > COUNTING_SORT_THRESHOLD_FOR_BYTE) {
      // todo: having an order is making counting sort a bit more complex,
      //       so the threshold is probably wrong

      // Count and collect the values which appear in unique, then sort it and add
      // elements by count to the original array. Fallback to standard sorting if
      // all values are unique because then we can just sort the original array.
      final int length = right - left + 1;
      final int[] count = new int[NUM_BYTE_VALUES];
      final byte[] unique = new byte[Math.min(NUM_BYTE_VALUES, length)];
      int uCount = 0;
      for (int i = left;  i <= right;  ++i) {
        if (count[a[i] - Byte.MIN_VALUE]++ == 0) {
          unique[uCount++] = a[i];
        }
      }
      if (uCount < length) {
        if (uCount > COUNTING_SORT_THRESHOLD_FOR_BYTE) {
          FastSortByte.quickSort(MutableByteIndexable.viewArray(unique, 0, uCount), order);
        }
        else {
          insertionSort(unique, 0, uCount - 1, order);  // this may become 256 bytes long
        }
        int idx = left;
        for (int i = 0;  i < uCount;  ++i) {
          final byte v = unique[i];
          int cnt = count[v - Byte.MIN_VALUE];
          assert cnt > 0;
          while (--cnt >= 0) {
            a[idx++] = v;
          }
        }
        assert idx == length;
        return;
      }
      // else: counting didn't help as each value only appeared once, so instead of sorting unique sort the array
    }
    // Use insertion sort on small arrays
    insertionSort(a, left, right, order);
  }

  private static void insertionSort(@NotNull byte[] a, int left, int right, @NotNull ByteOrdering order)
  {
    for (int i = left, j = i; i < right; j = ++i) {
      final byte ai = a[i + 1];
      while (order.ascending(ai, a[j])) {
        a[j + 1] = a[j];
        if (j-- == left) {
          break;
        }
      }
      a[j + 1] = ai;
    }
  }

  /**
   * Sorts the specified array using the specified order.
   *
   * @param a the array to be sorted
   * @param order sort order
   */
  public static void sort(@NotNull float[] a, @NotNull FloatOrdering order) {
    sort(a, 0, a.length - 1, order);
  }

  /**
   * Sorts the specified range of the array.
   *
   * @param a the array to be sorted
   * @param left the index of the first element, inclusive, to be sorted
   * @param right the index of the last element, inclusive, to be sorted
   * @param order sort order
   */
  public static void sort(@NotNull float[] a, int left, int right, @NotNull FloatOrdering order) {
    // Note [Rammi]: we assume here that order is taking care of handling special values like NaN and infinities.
    //               This makes this method a lot easier, and we can just inline the former doSort() method.

    // Use Quicksort on small arrays
    if (right - left < QUICKSORT_THRESHOLD) {
      sort(a, left, right, order, true);
      return;
    }

    /*
     * Index run[i] is the start of i-th run
     * (ascending or descending sequence).
     */
    final int[] run = new int[MAX_RUN_COUNT + 1];
    int count = 0; run[0] = left;

    // Check if the array is nearly sorted
    for (int k = left; k < right; run[count] = k) {
      switch (order.checkFloat(a[k], a[k + 1])) {
      case Ascending:
        while (++k <= right && order.ascendingOrSame(a[k - 1], a[k]));
        break;
      case Descending:
        while (++k <= right && a[k - 1] >= a[k]);
        for (int lo = run[count] - 1, hi = k; ++lo < --hi; ) {
          float t = a[lo]; a[lo] = a[hi]; a[hi] = t;
        }
        break;
      default:
        for (int m = MAX_RUN_LENGTH; ++k <= right && a[k - 1] == a[k]; ) {
          if (--m == 0) {
            sort(a, left, right, order, true);
            return;
          }
        }
        break;
      }

      /*
       * The array is not highly structured,
       * use Quicksort instead of merge sort.
       */
      if (++count == MAX_RUN_COUNT) {
        sort(a, left, right, order, true);
        return;
      }
    }

    // Check special cases
    if (run[count] == right++) { // The last run contains one element
      run[++count] = right;
    } else if (count == 1) { // The array is already sorted
      return;
    }

    /*
     * Create temporary array, which is used for merging.
     * Implementation note: variable "right" is increased by 1.
     */
    float[] b; byte odd = 0;
    for (int n = 1; (n <<= 1) < count; odd ^= 1);

    if (odd == 0) {
      b = a; a = new float[b.length];
      for (int i = left - 1; ++i < right; a[i] = b[i]);
    } else {
      b = new float[a.length];
    }

    // Merging
    for (int last; count > 1; count = last) {
      for (int k = (last = 0) + 2; k <= count; k += 2) {
        final int hi = run[k], mi = run[k - 1];
        for (int i = run[k - 2], p = i, q = mi; i < hi; ++i) {
          if (q >= hi || p < mi && order.ascendingOrSame(a[p], a[q])) {
            b[i] = a[p++];
          } else {
            b[i] = a[q++];
          }
        }
        run[++last] = hi;
      }
      if ((count & 1) != 0) {
        for (int i = right, lo = run[count - 1]; --i >= lo;
             b[i] = a[i]
        );
        run[++last] = right;
      }
      final float[] t = a; a = b; b = t;
    }
  }

  /**
   * Sorts the specified range of the array by Dual-Pivot Quicksort.
   *
   * @param a the array to be sorted
   * @param left the index of the first element, inclusive, to be sorted
   * @param right the index of the last element, inclusive, to be sorted
   * @param leftmost indicates if this part is the leftmost in the range
   */
  private static void sort(@NotNull float[] a, int left, int right, @NotNull FloatOrdering order, boolean leftmost) {
    final int length = right - left + 1;

    // Use insertion sort on tiny arrays
    if (length < INSERTION_SORT_THRESHOLD) {
      if (leftmost) {
        /*
         * Traditional (without sentinel) insertion sort,
         * optimized for server VM, is used in case of
         * the leftmost part.
         */
        for (int i = left, j = i; i < right; j = ++i) {
          float ai = a[i + 1];
          while (order.ascending(ai, a[j])) {
            a[j + 1] = a[j];
            if (j-- == left) {
              break;
            }
          }
          a[j + 1] = ai;
        }
      } else {
        /*
         * Skip the longest ascending sequence.
         */
        do {
          if (left >= right) {
            return;
          }
        } while (order.descendingOrSame(a[++left], a[left - 1]));

        /*
         * Every element from adjoining part plays the role
         * of sentinel, therefore this allows us to avoid the
         * left range check on each iteration. Moreover, we use
         * the more optimized algorithm, so called pair insertion
         * sort, which is faster (in the context of Quicksort)
         * than traditional implementation of insertion sort.
         */
        for (int k = left; ++left <= right; k = ++left) {
          float a1 = a[k], a2 = a[left];

          if (order.ascending(a1, a2)) {
            a2 = a1; a1 = a[left];
          }
          while (order.ascending(a1, a[--k])) {
            a[k + 2] = a[k];
          }
          a[++k + 1] = a1;

          while (order.ascending(a2, a[--k])) {
            a[k + 1] = a[k];
          }
          a[k + 1] = a2;
        }
        final float last = a[right];

        while (order.ascending(last, a[--right])) {
          a[right + 1] = a[right];
        }
        a[right + 1] = last;
      }
      return;
    }

    // Inexpensive approximation of length / 7
    final int seventh = (length >> 3) + (length >> 6) + 1;

    /*
     * Sort five evenly spaced elements around (and including) the
     * center element in the range. These elements will be used for
     * pivot selection as described below. The choice for spacing
     * these elements was empirically determined to work well on
     * a wide variety of inputs.
     */
    final int e3 = (left + right) >>> 1; // The midpoint
    final int e2 = e3 - seventh;
    final int e1 = e2 - seventh;
    final int e4 = e3 + seventh;
    final int e5 = e4 + seventh;

    // Sort these elements using insertion sort
    if (order.ascending(a[e2], a[e1])) { final float t = a[e2]; a[e2] = a[e1]; a[e1] = t; }

    if (order.ascending(a[e3], a[e2])) { final float t = a[e3]; a[e3] = a[e2]; a[e2] = t;
      if (order.ascending(t, a[e1])) { a[e2] = a[e1]; a[e1] = t; }
    }
    if (order.ascending(a[e4], a[e3])) { final float t = a[e4]; a[e4] = a[e3]; a[e3] = t;
      if (order.ascending(t, a[e2])) { a[e3] = a[e2]; a[e2] = t;
        if (order.ascending(t, a[e1])) { a[e2] = a[e1]; a[e1] = t; }
      }
    }
    if (order.ascending(a[e5], a[e4])) { final float t = a[e5]; a[e5] = a[e4]; a[e4] = t;
      if (order.ascending(t, a[e3])) { a[e4] = a[e3]; a[e3] = t;
        if (order.ascending(t, a[e2])) { a[e3] = a[e2]; a[e2] = t;
          if (order.ascending(t, a[e1])) { a[e2] = a[e1]; a[e1] = t; }
        }
      }
    }

    // Pointers
    int less  = left;  // The index of the first element of center part
    int great = right; // The index before the first element of right part

    if (order.different(a[e1], a[e2]) && order.different(a[e2], a[e3]) && order.different(a[e3], a[e4]) && order.different(a[e4], a[e5])) {
      /*
       * Use the second and fourth of the five sorted elements as pivots.
       * These values are inexpensive approximations of the first and
       * second terciles of the array. Note that pivot1 <= pivot2.
       */
      final float pivot1 = a[e2];
      final float pivot2 = a[e4];

      /*
       * The first and the last elements to be sorted are moved to the
       * locations formerly occupied by the pivots. When partitioning
       * is complete, the pivots are swapped back into their final
       * positions, and excluded from subsequent sorting.
       */
      a[e2] = a[left];
      a[e4] = a[right];

      /*
       * Skip elements, which are less or greater than pivot values.
       */
      while (order.ascending(a[++less], pivot1));
      while (order.descending(a[--great], pivot2));

      /*
       * Partitioning:
       *
       *   left part           center part                   right part
       * +--------------------------------------------------------------+
       * |  < pivot1  |  pivot1 <= && <= pivot2  |    ?    |  > pivot2  |
       * +--------------------------------------------------------------+
       *               ^                          ^       ^
       *               |                          |       |
       *              less                        k     great
       *
       * Invariants:
       *
       *              all in (left, less)   < pivot1
       *    pivot1 <= all in [less, k)     <= pivot2
       *              all in (great, right) > pivot2
       *
       * Pointer k is the first index of ?-part.
       */
      outer:
      for (int k = less - 1; ++k <= great; ) {
        final float ak = a[k];
        if (order.ascending(ak, pivot1)) { // Move a[k] to left part
          a[k] = a[less];
          /*
           * Here and below we use "a[i] = b; i++;" instead
           * of "a[i++] = b;" due to performance issue.
           */
          a[less] = ak;
          ++less;
        } else if (order.descending(ak, pivot2)) { // Move a[k] to right part
          while (order.descending(a[great], pivot2)) {
            if (great-- == k) {
              break outer;
            }
          }
          if (order.ascending(a[great], pivot1)) { // a[great] <= pivot2
            a[k] = a[less];
            a[less] = a[great];
            ++less;
          } else { // pivot1 <= a[great] <= pivot2
            a[k] = a[great];
          }
          /*
           * Here and below we use "a[i] = b; i--;" instead
           * of "a[i--] = b;" due to performance issue.
           */
          a[great] = ak;
          --great;
        }
      }

      // Swap pivots into their final positions
      a[left]  = a[less  - 1]; a[less  - 1] = pivot1;
      a[right] = a[great + 1]; a[great + 1] = pivot2;

      // Sort left and right parts recursively, excluding known pivots
      sort(a, left, less - 2, order, leftmost);
      sort(a, great + 2, right, order, false);

      /*
       * If center part is too large (comprises > 4/7 of the array),
       * swap internal pivot values to ends.
       */
      if (less < e1 && e5 < great) {
        /*
         * Skip elements, which are equal to pivot values.
         */
        while (order.same(a[less], pivot1)) {
          ++less;
        }

        while (order.same(a[great], pivot2)) {
          --great;
        }

        /*
         * Partitioning:
         *
         *   left part         center part                  right part
         * +----------------------------------------------------------+
         * | == pivot1 |  pivot1 < && < pivot2  |    ?    | == pivot2 |
         * +----------------------------------------------------------+
         *              ^                        ^       ^
         *              |                        |       |
         *             less                      k     great
         *
         * Invariants:
         *
         *              all in (*,  less) == pivot1
         *     pivot1 < all in [less,  k)  < pivot2
         *              all in (great, *) == pivot2
         *
         * Pointer k is the first index of ?-part.
         */
        outer:
        for (int k = less - 1; ++k <= great; ) {
          final float ak = a[k];
          if (order.same(ak, pivot1)) { // Move a[k] to left part
            a[k] = a[less];
            a[less] = ak;
            ++less;
          } else if (order.same(ak, pivot2)) { // Move a[k] to right part
            while (order.same(a[great], pivot2)) {
              if (great-- == k) {
                break outer;
              }
            }
            if (order.same(a[great], pivot1)) { // a[great] < pivot2
              a[k] = a[less];
              /*
               * Even though a[great] equals to pivot1, the
               * assignment a[less] = pivot1 may be incorrect,
               * if a[great] and pivot1 are floating-point zeros
               * of different signs. Therefore in float and
               * double sorting methods we have to use more
               * accurate assignment a[less] = a[great].
               */
              a[less] = a[great];
              ++less;
            } else { // pivot1 < a[great] < pivot2
              a[k] = a[great];
            }
            a[great] = ak;
            --great;
          }
        }
      }

      // Sort center part recursively
      sort(a, less, great, order, false);

    } else { // Partitioning with one pivot
      /*
       * Use the third of the five sorted elements as pivot.
       * This value is inexpensive approximation of the median.
       */
      final float pivot = a[e3];

      /*
       * Partitioning degenerates to the traditional 3-way
       * (or "Dutch National Flag") schema:
       *
       *   left part    center part              right part
       * +-------------------------------------------------+
       * |  < pivot  |   == pivot   |     ?    |  > pivot  |
       * +-------------------------------------------------+
       *              ^              ^        ^
       *              |              |        |
       *             less            k      great
       *
       * Invariants:
       *
       *   all in (left, less)   < pivot
       *   all in [less, k)     == pivot
       *   all in (great, right) > pivot
       *
       * Pointer k is the first index of ?-part.
       */
      for (int k = less; k <= great; ++k) {
        if (order.same(a[k], pivot)) {
          continue;
        }
        final float ak = a[k];
        if (order.ascending(ak, pivot)) { // Move a[k] to left part
          a[k] = a[less];
          a[less] = ak;
          ++less;
        } else { // a[k] > pivot - Move a[k] to right part
          while (order.descending(a[great], pivot)) {
            --great;
          }
          if (order.ascending(a[great], pivot)) { // a[great] <= pivot
            a[k] = a[less];
            a[less] = a[great];
            ++less;
          } else { // a[great] == pivot
            /*
             * Even though a[great] equals to pivot, the
             * assignment a[k] = pivot may be incorrect,
             * if a[great] and pivot are floating-point
             * zeros of different signs. Therefore in float
             * and double sorting methods we have to use
             * more accurate assignment a[k] = a[great].
             */
            a[k] = a[great];
          }
          a[great] = ak;
          --great;
        }
      }

      /*
       * Sort left and right parts recursively.
       * All elements from center part are equal
       * and, therefore, already sorted.
       */
      sort(a, left, less - 1, order, leftmost);
      sort(a, great + 1, right, order, false);
    }
  }

  /**
   * Sorts the specified array according to the specified order.
   *
   * @param a the array to be sorted
   * @param order sort order
   */
  public static void sort(@NotNull double[] a, @NotNull DoubleOrdering order) {
    sort(a, 0, a.length - 1, order);
  }

  /**
   * Sorts the specified range of the array according to the specified order.
   *
   * @param a the array to be sorted
   * @param left the index of the first element, inclusive, to be sorted
   * @param right the index of the last element, inclusive, to be sorted
   * @param order sort order
   */
  public static void sort(double[] a, int left, int right, @NotNull DoubleOrdering order) {
    // Note [Rammi]: we assume here that order is taking care of handling special values like NaN and infinities.
    //               This makes this method a lot easier, and we can just inline the former doSort() method.

    // Use Quicksort on small arrays
    if (right - left < QUICKSORT_THRESHOLD) {
      sort(a, left, right, order, true);
      return;
    }

    /*
     * Index run[i] is the start of i-th run
     * (ascending or descending sequence).
     */
    final int[] run = new int[MAX_RUN_COUNT + 1];
    int count = 0; run[0] = left;

    // Check if the array is nearly sorted
    for (int k = left; k < right; run[count] = k) {
      switch (order.checkDouble(a[k], a[k + 1])) {
      case Ascending:
        while (++k <= right && order.ascendingOrSame(a[k - 1], a[k]));
        break;
      case Descending:
        while (++k <= right && order.descendingOrSame(a[k - 1], a[k]));
        for (int lo = run[count] - 1, hi = k; ++lo < --hi; ) {
          double t = a[lo]; a[lo] = a[hi]; a[hi] = t;
        }
        break;
      default:
        for (int m = MAX_RUN_LENGTH; ++k <= right && order.same(a[k - 1], a[k]); ) {
          if (--m == 0) {
            sort(a, left, right, order, true);
            return;
          }
        }
        break;
      }

      /*
       * The array is not highly structured,
       * use Quicksort instead of merge sort.
       */
      if (++count == MAX_RUN_COUNT) {
        sort(a, left, right, order, true);
        return;
      }
    }

    // Check special cases
    if (run[count] == right++) { // The last run contains one element
      run[++count] = right;
    } else if (count == 1) { // The array is already sorted
      return;
    }

    /*
     * Create temporary array, which is used for merging.
     * Implementation note: variable "right" is increased by 1.
     */
    double[] b; byte odd = 0;
    for (int n = 1; (n <<= 1) < count; odd ^= 1);

    if (odd == 0) {
      b = a; a = new double[b.length];
      for (int i = left - 1; ++i < right; a[i] = b[i]);
    } else {
      b = new double[a.length];
    }

    // Merging
    for (int last; count > 1; count = last) {
      for (int k = (last = 0) + 2; k <= count; k += 2) {
        int hi = run[k], mi = run[k - 1];
        for (int i = run[k - 2], p = i, q = mi; i < hi; ++i) {
          if (q >= hi || p < mi && order.ascendingOrSame(a[p], a[q])) {
            b[i] = a[p++];
          } else {
            b[i] = a[q++];
          }
        }
        run[++last] = hi;
      }
      if ((count & 1) != 0) {
        for (int i = right, lo = run[count - 1]; --i >= lo;
             b[i] = a[i]
        );
        run[++last] = right;
      }
      double[] t = a; a = b; b = t;
    }
  }

  /**
   * Sorts the specified range of the array by Dual-Pivot Quicksort.
   *
   * @param a the array to be sorted
   * @param left the index of the first element, inclusive, to be sorted
   * @param right the index of the last element, inclusive, to be sorted
   * @param leftmost indicates if this part is the leftmost in the range
   */
  private static void sort(@NotNull double[] a, int left, int right, @NotNull DoubleOrdering order, boolean leftmost) {
    final int length = right - left + 1;

    // Use insertion sort on tiny arrays
    if (length < INSERTION_SORT_THRESHOLD) {
      if (leftmost) {
        /*
         * Traditional (without sentinel) insertion sort,
         * optimized for server VM, is used in case of
         * the leftmost part.
         */
        for (int i = left, j = i; i < right; j = ++i) {
          final double ai = a[i + 1];
          while (order.ascending(ai, a[j])) {
            a[j + 1] = a[j];
            if (j-- == left) {
              break;
            }
          }
          a[j + 1] = ai;
        }
      } else {
        /*
         * Skip the longest ascending sequence.
         */
        do {
          if (left >= right) {
            return;
          }
        } while (order.descendingOrSame(a[++left], a[left - 1]));

        /*
         * Every element from adjoining part plays the role
         * of sentinel, therefore this allows us to avoid the
         * left range check on each iteration. Moreover, we use
         * the more optimized algorithm, so called pair insertion
         * sort, which is faster (in the context of Quicksort)
         * than traditional implementation of insertion sort.
         */
        for (int k = left; ++left <= right; k = ++left) {
          double a1 = a[k], a2 = a[left];

          if (order.ascending(a1, a2)) {
            a2 = a1; a1 = a[left];
          }
          while (order.ascending(a1, a[--k])) {
            a[k + 2] = a[k];
          }
          a[++k + 1] = a1;

          while (order.ascending(a2, a[--k])) {
            a[k + 1] = a[k];
          }
          a[k + 1] = a2;
        }
        final double last = a[right];

        while (order.ascending(last, a[--right])) {
          a[right + 1] = a[right];
        }
        a[right + 1] = last;
      }
      return;
    }

    // Inexpensive approximation of length / 7
    final int seventh = (length >> 3) + (length >> 6) + 1;

    /*
     * Sort five evenly spaced elements around (and including) the
     * center element in the range. These elements will be used for
     * pivot selection as described below. The choice for spacing
     * these elements was empirically determined to work well on
     * a wide variety of inputs.
     */
    final int e3 = (left + right) >>> 1; // The midpoint
    final int e2 = e3 - seventh;
    final int e1 = e2 - seventh;
    final int e4 = e3 + seventh;
    final int e5 = e4 + seventh;

    // Sort these elements using insertion sort
    if (order.ascending(a[e2], a[e1])) { final double t = a[e2]; a[e2] = a[e1]; a[e1] = t; }

    if (order.ascending(a[e3], a[e2])) { final double t = a[e3]; a[e3] = a[e2]; a[e2] = t;
      if (order.ascending(t, a[e1])) { a[e2] = a[e1]; a[e1] = t; }
    }
    if (order.ascending(a[e4], a[e3])) { final double t = a[e4]; a[e4] = a[e3]; a[e3] = t;
      if (order.ascending(t, a[e2])) { a[e3] = a[e2]; a[e2] = t;
        if (order.ascending(t, a[e1])) { a[e2] = a[e1]; a[e1] = t; }
      }
    }
    if (order.ascending(a[e5], a[e4])) { final double t = a[e5]; a[e5] = a[e4]; a[e4] = t;
      if (order.ascending(t, a[e3])) { a[e4] = a[e3]; a[e3] = t;
        if (order.ascending(t, a[e2])) { a[e3] = a[e2]; a[e2] = t;
          if (order.ascending(t, a[e1])) { a[e2] = a[e1]; a[e1] = t; }
        }
      }
    }

    // Pointers
    int less  = left;  // The index of the first element of center part
    int great = right; // The index before the first element of right part

    if (order.same(a[e1], a[e2]) && order.same(a[e2], a[e3]) && order.same(a[e3], a[e4]) && order.same(a[e4], a[e5])) {
      /*
       * Use the second and fourth of the five sorted elements as pivots.
       * These values are inexpensive approximations of the first and
       * second terciles of the array. Note that pivot1 <= pivot2.
       */
      final double pivot1 = a[e2];
      final double pivot2 = a[e4];

      /*
       * The first and the last elements to be sorted are moved to the
       * locations formerly occupied by the pivots. When partitioning
       * is complete, the pivots are swapped back into their final
       * positions, and excluded from subsequent sorting.
       */
      a[e2] = a[left];
      a[e4] = a[right];

      /*
       * Skip elements, which are less or greater than pivot values.
       */
      while (order.ascending(a[++less], pivot1));
      while (order.descending(a[--great], pivot2));

      /*
       * Partitioning:
       *
       *   left part           center part                   right part
       * +--------------------------------------------------------------+
       * |  < pivot1  |  pivot1 <= && <= pivot2  |    ?    |  > pivot2  |
       * +--------------------------------------------------------------+
       *               ^                          ^       ^
       *               |                          |       |
       *              less                        k     great
       *
       * Invariants:
       *
       *              all in (left, less)   < pivot1
       *    pivot1 <= all in [less, k)     <= pivot2
       *              all in (great, right) > pivot2
       *
       * Pointer k is the first index of ?-part.
       */
      outer:
      for (int k = less - 1; ++k <= great; ) {
        final double ak = a[k];
        if (order.ascending(ak, pivot1)) { // Move a[k] to left part
          a[k] = a[less];
          /*
           * Here and below we use "a[i] = b; i++;" instead
           * of "a[i++] = b;" due to performance issue.
           */
          a[less] = ak;
          ++less;
        } else if (order.descending(ak, pivot2)) { // Move a[k] to right part
          while (order.descending(a[great], pivot2)) {
            if (great-- == k) {
              break outer;
            }
          }
          if (order.ascending(a[great], pivot1)) { // a[great] <= pivot2
            a[k] = a[less];
            a[less] = a[great];
            ++less;
          } else { // pivot1 <= a[great] <= pivot2
            a[k] = a[great];
          }
          /*
           * Here and below we use "a[i] = b; i--;" instead
           * of "a[i--] = b;" due to performance issue.
           */
          a[great] = ak;
          --great;
        }
      }

      // Swap pivots into their final positions
      a[left]  = a[less  - 1]; a[less  - 1] = pivot1;
      a[right] = a[great + 1]; a[great + 1] = pivot2;

      // Sort left and right parts recursively, excluding known pivots
      sort(a, left, less - 2, order, leftmost);
      sort(a, great + 2, right, order, false);

      /*
       * If center part is too large (comprises > 4/7 of the array),
       * swap internal pivot values to ends.
       */
      if (less < e1 && e5 < great) {
        /*
         * Skip elements, which are equal to pivot values.
         */
        while (order.same(a[less], pivot1)) {
          ++less;
        }

        while (order.same(a[great], pivot2)) {
          --great;
        }

        /*
         * Partitioning:
         *
         *   left part         center part                  right part
         * +----------------------------------------------------------+
         * | == pivot1 |  pivot1 < && < pivot2  |    ?    | == pivot2 |
         * +----------------------------------------------------------+
         *              ^                        ^       ^
         *              |                        |       |
         *             less                      k     great
         *
         * Invariants:
         *
         *              all in (*,  less) == pivot1
         *     pivot1 < all in [less,  k)  < pivot2
         *              all in (great, *) == pivot2
         *
         * Pointer k is the first index of ?-part.
         */
        outer:
        for (int k = less - 1; ++k <= great; ) {
          final double ak = a[k];
          if (order.same(ak, pivot1)) { // Move a[k] to left part
            a[k] = a[less];
            a[less] = ak;
            ++less;
          } else if (order.same(ak, pivot2)) { // Move a[k] to right part
            while (order.same(a[great], pivot2)) {
              if (great-- == k) {
                break outer;
              }
            }
            if (order.same(a[great], pivot1)) { // a[great] < pivot2
              a[k] = a[less];
              /*
               * Even though a[great] equals to pivot1, the
               * assignment a[less] = pivot1 may be incorrect,
               * if a[great] and pivot1 are floating-point zeros
               * of different signs. Therefore in float and
               * double sorting methods we have to use more
               * accurate assignment a[less] = a[great].
               */
              a[less] = a[great];
              ++less;
            } else { // pivot1 < a[great] < pivot2
              a[k] = a[great];
            }
            a[great] = ak;
            --great;
          }
        }
      }

      // Sort center part recursively
      sort(a, less, great, order, false);

    } else { // Partitioning with one pivot
      /*
       * Use the third of the five sorted elements as pivot.
       * This value is inexpensive approximation of the median.
       */
      final double pivot = a[e3];

      /*
       * Partitioning degenerates to the traditional 3-way
       * (or "Dutch National Flag") schema:
       *
       *   left part    center part              right part
       * +-------------------------------------------------+
       * |  < pivot  |   == pivot   |     ?    |  > pivot  |
       * +-------------------------------------------------+
       *              ^              ^        ^
       *              |              |        |
       *             less            k      great
       *
       * Invariants:
       *
       *   all in (left, less)   < pivot
       *   all in [less, k)     == pivot
       *   all in (great, right) > pivot
       *
       * Pointer k is the first index of ?-part.
       */
      for (int k = less; k <= great; ++k) {
        if (order.same(a[k], pivot)) {
          continue;
        }
        final double ak = a[k];
        if (order.ascending(ak, pivot)) { // Move a[k] to left part
          a[k] = a[less];
          a[less] = ak;
          ++less;
        } else { // a[k] > pivot - Move a[k] to right part
          while (order.descending(a[great], pivot)) {
            --great;
          }
          if (order.ascending(a[great], pivot)) { // a[great] <= pivot
            a[k] = a[less];
            a[less] = a[great];
            ++less;
          } else { // a[great] == pivot
            /*
             * Even though a[great] equals to pivot, the
             * assignment a[k] = pivot may be incorrect,
             * if a[great] and pivot are floating-point
             * zeros of different signs. Therefore in float
             * and double sorting methods we have to use
             * more accurate assignment a[k] = a[great].
             */
            a[k] = a[great];
          }
          a[great] = ak;
          --great;
        }
      }

      /*
       * Sort left and right parts recursively.
       * All elements from center part are equal
       * and, therefore, already sorted.
       */
      sort(a, left, less - 1, order, leftmost);
      sort(a, great + 1, right, order, false);
    }
  }

  /**
   * Sorts the specified mutable int indexable by the given order.
   * This is some 40% slower than the array method, therefore it's only kept to allow testing.
   *
   * @param a the indexable to be sorted
   * @param order user-defined order
   * @param <T> element type
   */
  static <T> void sort(@NotNull MutableIndexable<T> a, @NotNull Ordering<? super T> order) {
    final int left = 0;
    int right = a.size() - 1;

    // Use Quicksort on small arrays
    if (right - left < QUICKSORT_THRESHOLD) {
      sort(a, order, true);
      return;
    }

    /*
     * Index run[i] is the start of i-th run
     * (ascending or descending sequence).
     */
    final int[] run = new int[MAX_RUN_COUNT + 1];
    int count = 0; run[0] = left;

    // Check if the array is nearly sorted
    for (int k = left; k < right; run[count] = k) {
      switch (order.check(a.get(k), a.get(k + 1))) {
      case Ascending:
        while (++k <= right && order.ascendingOrSame(a.get(k - 1), a.get(k)) ) ;
        break;
      case Descending:
        while (++k <= right && order.descendingOrSame(a.get(k - 1), a.get(k))) ;
        a.revert(run[count] -1, k);
        break;
      default:
        for (int m = MAX_RUN_LENGTH; ++k <= right && order.same(a.get(k - 1), a.get(k)); ) {
          if (--m == 0) {
            sort(a, order, true);
            return;
          }
        }
        break;
      }

      /*
       * The array is not highly structured,
       * use Quicksort instead of merge sort.
       */
      if (++count == MAX_RUN_COUNT) {
        sort(a, order, true);
        return;
      }
    }

    // Check special cases
    if (run[count] == right++) { // The last run contains one element
      run[++count] = right;
    } else if (count == 1) { // The array is already sorted
      return;
    }

    /*
     * Create temporary array, which is used for merging.
     * Implementation note: variable "right" is increased by 1.
     */
    MutableIndexable<T> b; byte odd = 0;
    for (int n = 1; (n <<= 1) < count; odd ^= 1);

    if (odd == 0) {
      b = a; a = b.getCopy();
    } else {
      b = MutableIndexable.nulled(a.size());
    }

    // Merging
    for (int last; count > 1; count = last) {
      for (int k = (last = 0) + 2; k <= count; k += 2) {
        final int hi = run[k], mi = run[k - 1];
        for (int i = run[k - 2], p = i, q = mi; i < hi; ++i) {
          if (q >= hi || p < mi && order.ascending(a.get(p), a.get(q)))  {
            b.set(i, a.get(p++));
          } else {
            b.set(i, a.get(q++));
          }
        }
        run[++last] = hi;
      }
      if ((count & 1) != 0) {
        for (int i = right, lo = run[count - 1]; --i >= lo;
             b.set(i, a.get(i))
        );
        run[++last] = right;
      }
      final MutableIndexable<T> t = a; a = b; b = t;
    }
  }

  /**
   * Sorts the specified mutable indexable by Dual-Pivot Quicksort.
   *
   * @param a the array to be sorted
   * @param leftmost indicates if this part is the leftmost in the range
   * @param <T> element type
   */
  private static <T> void sort(@NotNull MutableIndexable<T> a,
                               @NotNull Ordering<? super T> order,
                               boolean leftmost) {
    final int length = a.size();
    int left = 0;
    int right = length - 1;

    // Use insertion sort on tiny arrays
    if (length < INSERTION_SORT_THRESHOLD) {
      if (leftmost) {
        /*
         * Traditional (without sentinel) insertion sort,
         * optimized for server VM, is used in case of
         * the leftmost part.
         */
        for (int i = left, j = i; i < right; j = ++i) {
          final T ai = a.get(i + 1);
          while (order.ascending(ai, a.get(j))) {
            a.set(j + 1, a.get(j));
            if (j-- == left) {
              break;
            }
          }
          a.set(j + 1, ai);
        }
      } else {
        /*
         * Skip the longest ascending sequence.
         */
        do {
          if (left >= right) {
            return;
          }
        } while (order.descendingOrSame(a.get(++left), a.get(left - 1)));

        /*
         * Every element from adjoining part plays the role
         * of sentinel, therefore this allows us to avoid the
         * left range check on each iteration. Moreover, we use
         * the more optimized algorithm, so called pair insertion
         * sort, which is faster (in the context of Quicksort)
         * than traditional implementation of insertion sort.
         */
        for (int k = left; ++left <= right; k = ++left) {
          T a1 = a.get(k), a2 = a.get(left);

          if (order.ascending(a1, a2)) {
            a2 = a1; a1 = a.get(left);
          }
          while (--k >= 0  &&  order.ascending(a1, a.get(k))) { // Rammi: had to prevent indexing w/ -1 here which happens for test set with seed and length being 47
            a.set(k + 2, a.get(k));
          }
          a.set(++k + 1, a1);

          while (--k >= 0  &&  order.ascending(a2, a.get(k))) { // Rammi: had to prevent indexing w/ -1 here which happens for test set with seed and length being 47
            a.set(k + 1, a.get(k));
          }
          a.set(k + 1, a2);
        }
        final T last = a.get(right);

        while (--right >= 0  &&  order.ascending(last, a.get(right))) { // Rammi: had to prevent indexing w/ -1 here which happens for test set with seed and length being 52
          a.set(right + 1, a.get(right));
        }
        a.set(right + 1, last);
      }
      return;
    }

    // Inexpensive approximation of length / 7
    final int seventh = (length >> 3) + (length >> 6) + 1;

    /*
     * Sort five evenly spaced elements around (and including) the
     * center element in the range. These elements will be used for
     * pivot selection as described below. The choice for spacing
     * these elements was empirically determined to work well on
     * a wide variety of inputs.
     */
    final int e3 = (left + right) >>> 1; // The midpoint
    final int e2 = e3 - seventh;
    final int e1 = e2 - seventh;
    final int e4 = e3 + seventh;
    final int e5 = e4 + seventh;

    // Sort these elements using insertion sort
    if (order.ascending(a.get(e2), a.get(e1))) { a.swap(e1, e2); }

    if (order.ascending(a.get(e3), a.get(e2))) { final T t = a.get(e3);a.set(e3, a.get(e2));a.set(e2, t);
      if (order.ascending(t, a.get(e1))) {a.set(e2, a.get(e1));a.set(e1, t); }
    }
    if (order.ascending(a.get(e4), a.get(e3))) { final T t = a.get(e4);a.set(e4, a.get(e3));a.set(e3, t);
      if (order.ascending(t, a.get(e2))) {a.set(e3, a.get(e2));a.set(e2, t);
        if (order.ascending(t, a.get(e1))) {a.set(e2, a.get(e1));a.set(e1, t); }
      }
    }
    if (order.ascending(a.get(e5), a.get(e4))) { final T t = a.get(e5);a.set(e5, a.get(e4));a.set(e4, t);
      if (order.ascending(t, a.get(e3))) {a.set(e4, a.get(e3));a.set(e3, t);
        if (order.ascending(t, a.get(e2))) {a.set(e3, a.get(e2));a.set(e2, t);
          if (order.ascending(t, a.get(e1))) {a.set(e2, a.get(e1));a.set(e1, t); }
        }
      }
    }

    // Pointers
    int less  = left;  // The index of the first element of center part
    int great = right; // The index before the first element of right part

    if (order.different(a.get(e1), a.get(e2)) && order.different(a.get(e2), a.get(e3)) && order.different(a.get(e3), a.get(e4)) && order.different(a.get(e4), a.get(e5))) {
      /*
       * Use the second and fourth of the five sorted elements as pivots.
       * These values are inexpensive approximations of the first and
       * second terciles of the array. Note that pivot1 <= pivot2.
       */
      final T pivot1 = a.get(e2);
      final T pivot2 = a.get(e4);

      /*
       * The first and the last elements to be sorted are moved to the
       * locations formerly occupied by the pivots. When partitioning
       * is complete, the pivots are swapped back into their final
       * positions, and excluded from subsequent sorting.
       */
      a.set(e2, a.get(left));
      a.set(e4, a.get(right));

      /*
       * Skip elements, which are less or greater than pivot values.
       */
      while (order.ascending(a.get(++less), pivot1));
      while (order.descending(a.get(--great), pivot2));

      /*
       * Partitioning:
       *
       *   left part           center part                   right part
       * +--------------------------------------------------------------+
       * |  < pivot1  |  pivot1 <= && <= pivot2  |    ?    |  > pivot2  |
       * +--------------------------------------------------------------+
       *               ^                          ^       ^
       *               |                          |       |
       *              less                        k     great
       *
       * Invariants:
       *
       *              all in (left, less)   < pivot1
       *    pivot1 <= all in [less, k)     <= pivot2
       *              all in (great, right) > pivot2
       *
       * Pointer k is the first index of ?-part.
       */
      outer:
      for (int k = less - 1; ++k <= great; ) {
        final T ak = a.get(k);
        if (order.ascending(ak, pivot1)) { // Move a[k] to left part
          a.set(k, a.get(less));
          /*
           * Here and below we use "a[i] = b; i++;" instead
           * of "a[i++] = b;" due to performance issue.
           */
          a.set(less, ak);
          ++less;
        } else if (order.descending(ak, pivot2)) { // Move a[k] to right part
          while (order.descending(a.get(great), pivot2)) {
            if (great-- == k) {
              break outer;
            }
          }
          if (order.ascending(a.get(great), pivot1)) { // a[great] <= pivot2
            a.set(k, a.get(less));
            a.set(less, a.get(great));
            ++less;
          } else { // pivot1 <= a[great] <= pivot2
            a.set(k, a.get(great));
          }
          /*
           * Here and below we use "a[i] = b; i--;" instead
           * of "a[i--] = b;" due to performance issue.
           */
          a.set(great, ak);
          --great;
        }
      }

      // Swap pivots into their final positions
      a.set(left,  a.get(less  - 1)); a.set(less  - 1, pivot1);
      a.set(right, a.get(great + 1)); a.set(great + 1, pivot2);

      // Sort left and right parts recursively, excluding known pivots
      sort(a.subSet(left, less - 1), order, leftmost);
      sort(a.subSet(great + 2, right + 1), order, false);

      /*
       * If center part is too large (comprises > 4/7 of the array),
       * swap internal pivot values to ends.
       */
      if (less < e1 && e5 < great) {
        /*
         * Skip elements, which are equal to pivot values.
         */
        while (order.same(a.get(less), pivot1)) {
          ++less;
        }

        while (order.same(a.get(great), pivot2)) {
          --great;
        }

        /*
         * Partitioning:
         *
         *   left part         center part                  right part
         * +----------------------------------------------------------+
         * | == pivot1 |  pivot1 < && < pivot2  |    ?    | == pivot2 |
         * +----------------------------------------------------------+
         *              ^                        ^       ^
         *              |                        |       |
         *             less                      k     great
         *
         * Invariants:
         *
         *              all in (*,  less) == pivot1
         *     pivot1 < all in [less,  k)  < pivot2
         *              all in (great, *) == pivot2
         *
         * Pointer k is the first index of ?-part.
         */
        outer:
        for (int k = less - 1; ++k <= great; ) {
          final T ak = a.get(k);
          if (order.same(ak, pivot1)) { // Move a[k] to left part
            a.set(k, a.get(less));
            a.set(less, ak);
            ++less;
          } else if (order.same(ak, pivot2)) { // Move a[k] to right part
            while (order.same(a.get(great), pivot2)) {
              if (great-- == k) {
                break outer;
              }
            }
            if (order.same(a.get(great), pivot1)) { // a[great] < pivot2
              a.set(k, a.get(less));
              /*
               * Even though a[great] equals to pivot1, the
               * assignment a[less] = pivot1 may be incorrect,
               * if a[great] and pivot1 are floating-point zeros
               * of different signs. Therefore in float and
               * double sorting methods we have to use more
               * accurate assignment a[less] = a[great].
               */
              a.set(less, pivot1);
              ++less;
            } else { // pivot1 < a[great] < pivot2
              a.set(k, a.get(great));
            }
            a.set(great, ak);
            --great;
          }
        }
      }

      // Sort center part recursively
      sort(a.subSet(less, great + 1), order, false);

    } else { // Partitioning with one pivot
      /*
       * Use the third of the five sorted elements as pivot.
       * This value is inexpensive approximation of the median.
       */
      final T pivot = a.get(e3);

      /*
       * Partitioning degenerates to the traditional 3-way
       * (or "Dutch National Flag") schema:
       *
       *   left part    center part              right part
       * +-------------------------------------------------+
       * |  < pivot  |   == pivot   |     ?    |  > pivot  |
       * +-------------------------------------------------+
       *              ^              ^        ^
       *              |              |        |
       *             less            k      great
       *
       * Invariants:
       *
       *   all in (left, less)   < pivot
       *   all in [less, k)     == pivot
       *   all in (great, right) > pivot
       *
       * Pointer k is the first index of ?-part.
       */
      for (int k = less; k <= great; ++k) {
        if (order.same(a.get(k), pivot)) {
          continue;
        }
        final T ak = a.get(k);
        if (order.ascending(ak, pivot)) { // Move a[k] to left part
          a.set(k, a.get(less));
          a.set(less, ak);
          ++less;
        } else { // a[k] > pivot - Move a[k] to right part
          while (order.descending(a.get(great), pivot)) {
            --great;
          }
          if (order.ascending(a.get(great), pivot)) { // a[great] <= pivot
            a.set(k, a.get(less));
            a.set(less,a.get(great));
            ++less;
          } else { // a[great] == pivot
            /*
             * Even though a[great] equals to pivot, the
             * assignment a[k] = pivot may be incorrect,
             * if a[great] and pivot are floating-point
             * zeros of different signs. Therefore in float
             * and double sorting methods we have to use
             * more accurate assignment a[k] = a[great].
             */
            a.set(k, pivot);
          }
          a.set(great, ak);
          --great;
        }
      }

      /*
       * Sort left and right parts recursively.
       * All elements from center part are equal
       * and, therefore, already sorted.
       */
      sort(a.subSet(left, less), order, leftmost);
      sort(a.subSet(great + 1, right + 1), order, false);
    }
  }

  /**
   * Sorts the specified mutable int indexable by the natural order of the elements.
   * @param a the int indexable to be sorted
   */
  public static void sort(@NotNull MutableIntIndexable a)
  {
    sort(a, IntOrdering.ASCENDING);
  }

  /**
   * Sorts the specified mutable int indexable by the given order.
   *
   * @param a the indexable to be sorted
   * @param order user-defined order
   */
  public static void sort(@NotNull MutableIntIndexable a, @NotNull IntOrdering order) {
    final int left = 0;
    int right = a.size() - 1;

    // Use Quicksort on small arrays
    if (right - left < QUICKSORT_THRESHOLD) {
      sort(a, order, true);
      return;
    }

    /*
     * Index run[i] is the start of i-th run
     * (ascending or descending sequence).
     */
    final int[] run = new int[MAX_RUN_COUNT + 1];
    int count = 0; run[0] = left;

    // Check if the array is nearly sorted
    for (int k = left; k < right; run[count] = k) {
      switch (order.checkInt(a.get(k), a.get(k + 1))) {
      case Ascending:
        while (++k <= right && order.ascendingOrSame(a.get(k - 1), a.get(k)) ) ;
        break;
      case Descending:
        while (++k <= right && order.descendingOrSame(a.get(k - 1), a.get(k))) ;
        a.revert(run[count] -1, k);
        break;
      default:
        for (int m = MAX_RUN_LENGTH; ++k <= right && order.same(a.get(k - 1), a.get(k)); ) {
          if (--m == 0) {
            sort(a, order, true);
            return;
          }
        }
        break;
      }

      /*
       * The array is not highly structured,
       * use Quicksort instead of merge sort.
       */
      if (++count == MAX_RUN_COUNT) {
        sort(a, order, true);
        return;
      }
    }

    // Check special cases
    if (run[count] == right++) { // The last run contains one element
      run[++count] = right;
    } else if (count == 1) { // The array is already sorted
      return;
    }

    /*
     * Create temporary array, which is used for merging.
     * Implementation note: variable "right" is increased by 1.
     */
    MutableIntIndexable b; byte odd = 0;
    for (int n = 1; (n <<= 1) < count; odd ^= 1);

    if (odd == 0) {
      b = a; a = b.getCopy();
    } else {
      b = MutableIntIndexable.zeroed(a.size());
    }

    // Merging
    for (int last; count > 1; count = last) {
      for (int k = (last = 0) + 2; k <= count; k += 2) {
        final int hi = run[k], mi = run[k - 1];
        for (int i = run[k - 2], p = i, q = mi; i < hi; ++i) {
          if (q >= hi || p < mi && order.ascending(a.get(p), a.get(q)))  {
            b.set(i, a.get(p++));
          } else {
            b.set(i, a.get(q++));
          }
        }
        run[++last] = hi;
      }
      if ((count & 1) != 0) {
        for (int i = right, lo = run[count - 1]; --i >= lo;
             b.set(i, a.get(i))
        );
        run[++last] = right;
      }
      final MutableIntIndexable t = a; a = b; b = t;
    }
  }

  /**
   * Sorts the specified mutable indexable by Dual-Pivot Quicksort.
   *
   * @param a the array to be sorted
   * @param order sort order
   * @param leftmost indicates if this part is the leftmost in the range
   */
  private static <T> void sort(@NotNull MutableIntIndexable a,
                               @NotNull IntOrdering order,
                               boolean leftmost) {
    final int length = a.size();
    int left = 0;
    int right = length - 1;

    // Use insertion sort on tiny arrays
    if (length < INSERTION_SORT_THRESHOLD) {
      if (leftmost) {
        /*
         * Traditional (without sentinel) insertion sort,
         * optimized for server VM, is used in case of
         * the leftmost part.
         */
        for (int i = left, j = i; i < right; j = ++i) {
          final int ai = a.get(i + 1);
          while (order.ascending(ai, a.get(j))) {
            a.set(j + 1, a.get(j));
            if (j-- == left) {
              break;
            }
          }
          a.set(j + 1, ai);
        }
      } else {
        /*
         * Skip the longest ascending sequence.
         */
        do {
          if (left >= right) {
            return;
          }
        } while (order.descendingOrSame(a.get(++left), a.get(left - 1)));

        /*
         * Every element from adjoining part plays the role
         * of sentinel, therefore this allows us to avoid the
         * left range check on each iteration. Moreover, we use
         * the more optimized algorithm, so called pair insertion
         * sort, which is faster (in the context of Quicksort)
         * than traditional implementation of insertion sort.
         */
        for (int k = left; ++left <= right; k = ++left) {
          int a1 = a.get(k), a2 = a.get(left);

          if (order.ascending(a1, a2)) {
            a2 = a1; a1 = a.get(left);
          }
          while (--k >= 0  &&  order.ascending(a1, a.get(k))) { // Rammi: had to prevent indexing w/ -1 here which happens for test set with seed and length being 47
            a.set(k + 2, a.get(k));
          }
          a.set(++k + 1, a1);

          while (--k >= 0  &&  order.ascending(a2, a.get(k))) { // Rammi: had to prevent indexing w/ -1 here which happens for test set with seed and length being 47
            a.set(k + 1, a.get(k));
          }
          a.set(k + 1, a2);
        }
        final int last = a.get(right);

        while (--right >= 0  &&  order.ascending(last, a.get(right))) { // Rammi: had to prevent indexing w/ -1 here which happens for test set with seed and length being 52
          a.set(right + 1, a.get(right));
        }
        a.set(right + 1, last);
      }
      return;
    }

    // Inexpensive approximation of length / 7
    final int seventh = (length >> 3) + (length >> 6) + 1;

    /*
     * Sort five evenly spaced elements around (and including) the
     * center element in the range. These elements will be used for
     * pivot selection as described below. The choice for spacing
     * these elements was empirically determined to work well on
     * a wide variety of inputs.
     */
    final int e3 = (left + right) >>> 1; // The midpoint
    final int e2 = e3 - seventh;
    final int e1 = e2 - seventh;
    final int e4 = e3 + seventh;
    final int e5 = e4 + seventh;

    // Sort these elements using insertion sort
    if (order.ascending(a.get(e2), a.get(e1))) { a.swap(e1, e2); }

    if (order.ascending(a.get(e3), a.get(e2))) { final int t = a.get(e3);a.set(e3, a.get(e2));a.set(e2, t);
      if (order.ascending(t, a.get(e1))) {a.set(e2, a.get(e1));a.set(e1, t); }
    }
    if (order.ascending(a.get(e4), a.get(e3))) { final int t = a.get(e4);a.set(e4, a.get(e3));a.set(e3, t);
      if (order.ascending(t, a.get(e2))) {a.set(e3, a.get(e2));a.set(e2, t);
        if (order.ascending(t, a.get(e1))) {a.set(e2, a.get(e1));a.set(e1, t); }
      }
    }
    if (order.ascending(a.get(e5), a.get(e4))) { final int t = a.get(e5);a.set(e5, a.get(e4));a.set(e4, t);
      if (order.ascending(t, a.get(e3))) {a.set(e4, a.get(e3));a.set(e3, t);
        if (order.ascending(t, a.get(e2))) {a.set(e3, a.get(e2));a.set(e2, t);
          if (order.ascending(t, a.get(e1))) {a.set(e2, a.get(e1));a.set(e1, t); }
        }
      }
    }

    // Pointers
    int less  = left;  // The index of the first element of center part
    int great = right; // The index before the first element of right part

    if (order.different(a.get(e1), a.get(e2)) && order.different(a.get(e2), a.get(e3)) && order.different(a.get(e3), a.get(e4)) && order.different(a.get(e4), a.get(e5))) {
      /*
       * Use the second and fourth of the five sorted elements as pivots.
       * These values are inexpensive approximations of the first and
       * second terciles of the array. Note that pivot1 <= pivot2.
       */
      final int pivot1 = a.get(e2);
      final int pivot2 = a.get(e4);

      /*
       * The first and the last elements to be sorted are moved to the
       * locations formerly occupied by the pivots. When partitioning
       * is complete, the pivots are swapped back into their final
       * positions, and excluded from subsequent sorting.
       */
      a.set(e2, a.get(left));
      a.set(e4, a.get(right));

      /*
       * Skip elements, which are less or greater than pivot values.
       */
      while (order.ascending(a.get(++less), pivot1));
      while (order.descending(a.get(--great), pivot2));

      /*
       * Partitioning:
       *
       *   left part           center part                   right part
       * +--------------------------------------------------------------+
       * |  < pivot1  |  pivot1 <= && <= pivot2  |    ?    |  > pivot2  |
       * +--------------------------------------------------------------+
       *               ^                          ^       ^
       *               |                          |       |
       *              less                        k     great
       *
       * Invariants:
       *
       *              all in (left, less)   < pivot1
       *    pivot1 <= all in [less, k)     <= pivot2
       *              all in (great, right) > pivot2
       *
       * Pointer k is the first index of ?-part.
       */
      outer:
      for (int k = less - 1; ++k <= great; ) {
        final int ak = a.get(k);
        if (order.ascending(ak, pivot1)) { // Move a[k] to left part
          a.set(k, a.get(less));
          /*
           * Here and below we use "a[i] = b; i++;" instead
           * of "a[i++] = b;" due to performance issue.
           */
          a.set(less, ak);
          ++less;
        } else if (order.descending(ak, pivot2)) { // Move a[k] to right part
          while (order.descending(a.get(great), pivot2)) {
            if (great-- == k) {
              break outer;
            }
          }
          if (order.ascending(a.get(great), pivot1)) { // a[great] <= pivot2
            a.set(k, a.get(less));
            a.set(less, a.get(great));
            ++less;
          } else { // pivot1 <= a[great] <= pivot2
            a.set(k, a.get(great));
          }
          /*
           * Here and below we use "a[i] = b; i--;" instead
           * of "a[i--] = b;" due to performance issue.
           */
          a.set(great, ak);
          --great;
        }
      }

      // Swap pivots into their final positions
      a.set(left,  a.get(less  - 1)); a.set(less  - 1, pivot1);
      a.set(right, a.get(great + 1)); a.set(great + 1, pivot2);

      // Sort left and right parts recursively, excluding known pivots
      sort(a.subSet(left, less - 1), order, leftmost);
      sort(a.subSet(great + 2, right + 1), order, false);

      /*
       * If center part is too large (comprises > 4/7 of the array),
       * swap internal pivot values to ends.
       */
      if (less < e1 && e5 < great) {
        /*
         * Skip elements, which are equal to pivot values.
         */
        while (order.same(a.get(less), pivot1)) {
          ++less;
        }

        while (order.same(a.get(great), pivot2)) {
          --great;
        }

        /*
         * Partitioning:
         *
         *   left part         center part                  right part
         * +----------------------------------------------------------+
         * | == pivot1 |  pivot1 < && < pivot2  |    ?    | == pivot2 |
         * +----------------------------------------------------------+
         *              ^                        ^       ^
         *              |                        |       |
         *             less                      k     great
         *
         * Invariants:
         *
         *              all in (*,  less) == pivot1
         *     pivot1 < all in [less,  k)  < pivot2
         *              all in (great, *) == pivot2
         *
         * Pointer k is the first index of ?-part.
         */
        outer:
        for (int k = less - 1; ++k <= great; ) {
          final int ak = a.get(k);
          if (order.same(ak, pivot1)) { // Move a[k] to left part
            a.set(k, a.get(less));
            a.set(less, ak);
            ++less;
          } else if (order.same(ak, pivot2)) { // Move a[k] to right part
            while (order.same(a.get(great), pivot2)) {
              if (great-- == k) {
                break outer;
              }
            }
            if (order.same(a.get(great), pivot1)) { // a[great] < pivot2
              a.set(k, a.get(less));
              /*
               * Even though a[great] equals to pivot1, the
               * assignment a[less] = pivot1 may be incorrect,
               * if a[great] and pivot1 are floating-point zeros
               * of different signs. Therefore in float and
               * double sorting methods we have to use more
               * accurate assignment a[less] = a[great].
               */
              a.set(less, pivot1);
              ++less;
            } else { // pivot1 < a[great] < pivot2
              a.set(k, a.get(great));
            }
            a.set(great, ak);
            --great;
          }
        }
      }

      // Sort center part recursively
      sort(a.subSet(less, great + 1), order, false);

    } else { // Partitioning with one pivot
      /*
       * Use the third of the five sorted elements as pivot.
       * This value is inexpensive approximation of the median.
       */
      final int pivot = a.get(e3);

      /*
       * Partitioning degenerates to the traditional 3-way
       * (or "Dutch National Flag") schema:
       *
       *   left part    center part              right part
       * +-------------------------------------------------+
       * |  < pivot  |   == pivot   |     ?    |  > pivot  |
       * +-------------------------------------------------+
       *              ^              ^        ^
       *              |              |        |
       *             less            k      great
       *
       * Invariants:
       *
       *   all in (left, less)   < pivot
       *   all in [less, k)     == pivot
       *   all in (great, right) > pivot
       *
       * Pointer k is the first index of ?-part.
       */
      for (int k = less; k <= great; ++k) {
        if (order.same(a.get(k), pivot)) {
          continue;
        }
        final int ak = a.get(k);
        if (order.ascending(ak, pivot)) { // Move a[k] to left part
          a.set(k, a.get(less));
          a.set(less, ak);
          ++less;
        } else { // a[k] > pivot - Move a[k] to right part
          while (order.descending(a.get(great), pivot)) {
            --great;
          }
          if (order.ascending(a.get(great), pivot)) { // a[great] <= pivot
            a.set(k, a.get(less));
            a.set(less,a.get(great));
            ++less;
          } else { // a[great] == pivot
            /*
             * Even though a[great] equals to pivot, the
             * assignment a[k] = pivot may be incorrect,
             * if a[great] and pivot are floating-point
             * zeros of different signs. Therefore in float
             * and double sorting methods we have to use
             * more accurate assignment a[k] = a[great].
             */
            a.set(k, pivot);
          }
          a.set(great, ak);
          --great;
        }
      }

      /*
       * Sort left and right parts recursively.
       * All elements from center part are equal
       * and, therefore, already sorted.
       */
      sort(a.subSet(left, less), order, leftmost);
      sort(a.subSet(great + 1, right + 1), order, false);
    }
  }
}
