// ============================================================================
// File:               WrappedFragileException
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2023-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            7/6/23 5:55 PM
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

/**
 * A runtime exception wrapper for a checked exception which occurred
 * in places where there is no way to handle it.
 * <p>
 * In such cases, this runtime exception is used to wrap the given exception,
 * and thrown instead.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since July 06, 2023
 */
public class WrappedFragileException
        extends RuntimeException
{
  private static final long serialVersionUID = -5957051677945478538L;

  /** Exception caught. */
  @NotNull
  private final Exception exception;

  /**
   * Create a wrapped fragile exception.
   * @param exception  fragile exception to be wrapped
   */
  public WrappedFragileException(@NotNull Exception exception)
  {
    super(exception);
    this.exception = exception;
  }

  /**
   * Create a wrapped fragile exception with an additional message.
   * @param exception  fragile exception to be wrapped
   * @param additionalMessage additional message
   */
  public WrappedFragileException(@NotNull Exception exception,
                                 @NotNull String additionalMessage)
  {
    super(additionalMessage, exception);
    this.exception = exception;
  }

  /**
   * Get the exception.
   * @return wrapped exception
   */
  @NotNull
  public Exception getException()
  {
    return exception;
  }

  /**
   * Throw the wrapped exception.
   * @throws Exception the original exception
   */
  public void rethrow()
          throws Exception
  {
    throw exception;
  }

  /**
   * Rethrow the wrapped exception as an expected type.
   * If the wrapped {@link #getException() exception} is not
   * of the correct type this method will throw a
   * {@link ClassCastException}.
   *
   * @param xType expected type as class
   * @param <E> expected type
   * @throws E expected exception type
   * @throws ClassCastException if wrapped exception is of a different type
   */
  public <E extends Exception> void rethrow(@NotNull Class<E> xType)
    throws E
  {
    throw xType.cast(exception);
  }
}
