// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.util.*;

/**
 *  Let an array appear as an unmodifiable list.
 * <p>
 *  This is cheaper than {@code java.util.Arrays.asList()} and allows
 *  for sub arrays to be accessed.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class UnmodifiableListLongArrayWrapper
        extends AbstractBasicUnmodifiableList<Long>
{
  private class MyIterator
          implements Iterator<Long>
  {
    /** Current index. */
    protected int index = start;

    /**
     * Removes from the underlying collection the last element returned by the
     * iterator (optional operation).  This method can be called only once per
     * call to <tt>next</tt>.  The behavior of an iterator is unspecified if
     * the underlying collection is modified while the iteration is in
     * progress in any way other than by calling this method.
     *
     * @throws UnsupportedOperationException if the <tt>remove</tt>
     *                                       operation is not supported by this Iterator.
     * @throws IllegalStateException         if the <tt>next</tt> method has not
     *                                       yet been called, or the <tt>remove</tt> method has already
     *                                       been called after the last call to the <tt>next</tt>
     *                                       method.
     */
    @Override
    public void remove()
    {
      throw new UnsupportedOperationException();
    }

    /**
     * Returns <tt>true</tt> if the iteration has more elements. (In other
     * words, returns <tt>true</tt> if <tt>next</tt> would return an element
     * rather than throwing an exception.)
     *
     * @return <tt>true</tt> if the iterator has more elements.
     */
    @Override
    public boolean hasNext()
    {
      return index - start < length;
    }

    /**
     * Returns the next element in the iteration.
     *
     * @return the next element in the iteration.
     * @throws java.util.NoSuchElementException
     *          iteration has no more elements.
     */
    @Override
    public Long next()
    {
      if (index - start == length) {
        throw new NoSuchElementException();
      }
      return array[index++];
    }
  }
  private class MyListIterator
          extends MyIterator
          implements ListIterator<Long>
  {
    private MyListIterator()
    {
    }

    private MyListIterator(int start)
    {
      index += start;
    }

    /**
     * Returns the index of the element that would be returned by a subsequent
     * call to <tt>next</tt>. (Returns list size if the list iterator is at the
     * end of the list.)
     *
     * @return the index of the element that would be returned by a subsequent
     *         call to <tt>next</tt>, or list size if list iterator is at end
     *         of list.
     */
    @Override
    public int nextIndex()
    {
      return index - start;
    }

    /**
     * Returns the index of the element that would be returned by a subsequent
     * call to <tt>previous</tt>. (Returns -1 if the list iterator is at the
     * beginning of the list.)
     *
     * @return the index of the element that would be returned by a subsequent
     *         call to <tt>previous</tt>, or -1 if list iterator is at
     *         beginning of list.
     */
    @Override
    public int previousIndex()
    {
      return index - start - 1;
    }

    /**
     * Returns <tt>true</tt> if this list iterator has more elements when
     * traversing the list in the reverse direction.  (In other words, returns
     * <tt>true</tt> if <tt>previous</tt> would return an element rather than
     * throwing an exception.)
     *
     * @return <tt>true</tt> if the list iterator has more elements when
     *         traversing the list in the reverse direction.
     */
    @Override
    public boolean hasPrevious()
    {
      return index - start >= 1;
    }

    /**
     * Returns the previous element in the list.  This method may be called
     * repeatedly to iterate through the list backwards, or intermixed with
     * calls to <tt>next</tt> to go back and forth.  (Note that alternating
     * calls to <tt>next</tt> and <tt>previous</tt> will return the same
     * element repeatedly.)
     *
     * @return the previous element in the list.
     * @throws java.util.NoSuchElementException
     *          if the iteration has no previous
     *          element.
     */
    @Override
    public Long previous()
    {
      if (index - start < 1) {
        throw new NoSuchElementException();
      }
      return array[--index];
    }

    /**
     * Inserts the specified element into the list (optional operation).  The
     * element is inserted immediately before the next element that would be
     * returned by <tt>next</tt>, if any, and after the next element that
     * would be returned by <tt>previous</tt>, if any.  (If the list contains
     * no elements, the new element becomes the sole element on the list.)
     * The new element is inserted before the implicit cursor: a subsequent
     * call to <tt>next</tt> would be unaffected, and a subsequent call to
     * <tt>previous</tt> would return the new element.  (This call increases
     * by one the value that would be returned by a call to <tt>nextIndex</tt>
     * or <tt>previousIndex</tt>.)
     *
     * @param o the element to insert.
     * @throws UnsupportedOperationException if the <tt>add</tt> method is
     *                                       not supported by this list iterator.
     * @throws ClassCastException            if the class of the specified element
     *                                       prevents it from being added to this list.
     * @throws IllegalArgumentException      if some aspect of this element
     *                                       prevents it from being added to this list.
     */
    @Override
    public void add(Long o)
    {
      throw new UnsupportedOperationException();
    }

    /**
     * Replaces the last element returned by <tt>next</tt> or
     * <tt>previous</tt> with the specified element (optional operation).
     * This call can be made only if neither <tt>ListIterator.remove</tt> nor
     * <tt>ListIterator.add</tt> have been called after the last call to
     * <tt>next</tt> or <tt>previous</tt>.
     *
     * @param o the element with which to replace the last element returned by
     *          <tt>next</tt> or <tt>previous</tt>.
     * @throws UnsupportedOperationException if the <tt>set</tt> operation
     *                                       is not supported by this list iterator.
     * @throws ClassCastException            if the class of the specified element
     *                                       prevents it from being added to this list.
     * @throws IllegalArgumentException      if some aspect of the specified
     *                                       element prevents it from being added to this list.
     * @throws IllegalStateException         if neither <tt>next</tt> nor
     *                                       <tt>previous</tt> have been called, or <tt>remove</tt> or
     *                                       <tt>add</tt> have been called after the last call to
     *                                       <tt>next</tt> or <tt>previous</tt>.
     */
    @Override
    public void set(Long o)
    {
      throw new UnsupportedOperationException();
    }
  }

  /** The underlying array. */
  private final long[] array;
  /** The start position. */
  private final int start;
  /** The length to use. */
  private final int length;

  /**
   * Constructor.
   * @param array   array to wrap
   * @param start   start index to use
   * @param length  length to use
   */
  public UnmodifiableListLongArrayWrapper(@NotNull long[] array, int start, int length)
  {
    if (array == null) {
      throw new IllegalArgumentException("'null' array not allowed!");
    }
    if (start < 0  ||  start > array.length) {
      throw new IllegalArgumentException("'start' does not point into array!");
    }
    if (length < 0  ||  start + length > array.length) {
      throw new IllegalArgumentException("'length' invalid!");
    }
    this.array = array;
    this.start = start;
    this.length = length;
  }

  /**
   * Constructor.
   * This uses the complete array.
   * @param array array to wrap
   */
  public UnmodifiableListLongArrayWrapper(@NotNull long[] array)
  {
    this(array, 0, array.length);
  }

  /**
   * Constructor.
   * This uses only length elements of the complete array, starting with the zeroth element..
   * @param array array to wrap
   * @param length length to use
   */
  public UnmodifiableListLongArrayWrapper(@NotNull long[] array, int length)
  {
    this(array, 0, length);
  }

  /**
   * Returns the number of elements in this list.  If this list contains
   * more than <tt>Integer.MAX_VALUE</tt> elements, returns
   * <tt>Integer.MAX_VALUE</tt>.
   *
   * @return the number of elements in this list.
   */
  @Override
  public int size()
  {
    return length;
  }

  /**
   * Returns <tt>true</tt> if this list contains no elements.
   *
   * @return <tt>true</tt> if this list contains no elements.
   */
  @Override
  public boolean isEmpty()
  {
    return length == 0;
  }

  /**
   * Returns an array containing all of the elements in this list in proper
   * sequence.  Obeys the general contract of the
   * <tt>Collection.toArray</tt> method.
   *
   * @return an array containing all of the elements in this list in proper
   *         sequence.
   * @see java.util.Arrays#asList(Object[])
   */
  @Override
  @NotNull
  public Object[] toArray()
  {
    Object[] result = new Object[length];
    if (!isEmpty()) {
      for (int i = 0;  i < length;  ++i) {
        result[i] = array[start + i];
      }
    }
    return result;
  }

  /**
   * Returns the element at the specified position in this list.
   *
   * @param index index of element to return.
   * @return the element at the specified position in this list.
   * @throws IndexOutOfBoundsException if the index is out of range (index
   *                                   &lt; 0 || index &gt;= size()).
   */
  @Override
  public Long get(int index)
  {
    if (index < 0  ||  index >= length) {
      throw new IndexOutOfBoundsException("Invalid index "+index);
    }
    return array[start + index];
  }

  /**
   * Returns the index in this list of the first occurrence of the specified
   * element, or -1 if this list does not contain this element.
   * More formally, returns the lowest index <tt>i</tt> such that
   * <tt>(o==null ? get(i)==null : o.equals(get(i)))</tt>,
   * or -1 if there is no such index.
   *
   * @param o element to search for.
   * @return the index in this list of the first occurrence of the specified
   *         element, or -1 if this list does not contain this element.
   * @throws ClassCastException   if the type of the specified element
   *                              is incompatible with this list (optional).
   * @throws NullPointerException if the specified element is null and this
   *                              list does not support null elements (optional).
   */
  @Override
  public int indexOf(Object o)
  {
    long v = ((Long)o).longValue();
    for (int i = 0;  i < length;  ++i) {
      if (array[start + i] == v) {
        return i;
      }
    }
    return -1;
  }

  /**
   * Returns the index in this list of the last occurrence of the specified
   * element, or -1 if this list does not contain this element.
   * More formally, returns the highest index <tt>i</tt> such that
   * <tt>(o==null ? get(i)==null : o.equals(get(i)))</tt>,
   * or -1 if there is no such index.
   *
   * @param o element to search for.
   * @return the index in this list of the last occurrence of the specified
   *         element, or -1 if this list does not contain this element.
   * @throws ClassCastException   if the type of the specified element
   *                              is incompatible with this list (optional).
   * @throws NullPointerException if the specified element is null and this
   *                              list does not support null elements (optional).
   */
  @Override
  public int lastIndexOf(Object o)
  {
    long v = ((Long)o).longValue();
    for (int i = length - 1;  i >= 0;  --i) {
      if (array[start + i] == v) {
        return i;
      }
    }
    return -1;
  }

  /**
   * Returns <tt>true</tt> if this list contains the specified element.
   * More formally, returns <tt>true</tt> if and only if this list contains
   * at least one element <tt>e</tt> such that
   * <tt>(o==null&nbsp;?&nbsp;e==null&nbsp;:&nbsp;o.equals(e))</tt>.
   *
   * @param o element whose presence in this list is to be tested.
   * @return <tt>true</tt> if this list contains the specified element.
   * @throws ClassCastException   if the type of the specified element
   *                              is incompatible with this list (optional).
   * @throws NullPointerException if the specified element is null and this
   *                              list does not support null elements (optional).
   */
  @Override
  public boolean contains(Object o)
  {
    return indexOf(o) >= 0;
  }

  /**
   * Returns <tt>true</tt> if this list contains all of the elements of the
   * specified collection.
   *
   * @param c collection to be checked for containment in this list.
   * @return <tt>true</tt> if this list contains all of the elements of the
   *         specified collection.
   * @throws ClassCastException   if the types of one or more elements
   *                              in the specified collection are incompatible with this
   *                              list (optional).
   * @throws NullPointerException if the specified collection contains one
   *                              or more null elements and this list does not support null
   *                              elements (optional).
   *                              or if the specified collection is
   *                              <tt>null</tt>.
   * @see #contains(Object)
   */
  @Override
  @SuppressWarnings("rawtypes")
  public boolean containsAll(@NotNull Collection c)
  {
    for (Object aC : c) {
      if (!contains(aC)) {
        return false;
      }
    }
    return true;
  }

  /**
   * Returns an iterator over the elements in this list in proper sequence.
   *
   * @return an iterator over the elements in this list in proper sequence.
   */
  @Override
  @NotNull
  public Iterator<Long> iterator()
  {
    return new MyIterator();
  }

  /**
   * Returns a view of the portion of this list between the specified
   * <tt>fromIndex</tt>, inclusive, and <tt>toIndex</tt>, exclusive.  (If
   * <tt>fromIndex</tt> and <tt>toIndex</tt> are equal, the returned list is
   * empty.)  The returned list is backed by this list, so non-structural
   * changes in the returned list are reflected in this list, and vice-versa.
   * The returned list supports all of the optional list operations supported
   * by this list.<p>
   *
   * This method eliminates the need for explicit range operations (of
   * the sort that commonly exist for arrays).   Any operation that expects
   * a list can be used as a range operation by passing a subList view
   * instead of a whole list.  For example, the following idiom
   * removes a range of elements from a list:
   * <pre>
   * 	    list.subList(from, to).clear();
   * </pre>
   * Similar idioms may be constructed for <tt>indexOf</tt> and
   * <tt>lastIndexOf</tt>, and all of the algorithms in the
   * <tt>Collections</tt> class can be applied to a subList.<p>
   *
   * The semantics of the list returned by this method become undefined if
   * the backing list (i.e., this list) is <i>structurally modified</i> in
   * any way other than via the returned list.  (Structural modifications are
   * those that change the size of this list, or otherwise perturb it in such
   * a fashion that iterations in progress may yield incorrect results.)
   *
   * @param fromIndex low endpoint (inclusive) of the subList.
   * @param toIndex   high endpoint (exclusive) of the subList.
   * @return a view of the specified range within this list.
   * @throws IndexOutOfBoundsException for an illegal endpoint index value
   *                                   (fromIndex &lt; 0 || toIndex &gt; size || fromIndex &gt; toIndex).
   */
  @Override
  @NotNull
  public List<Long> subList(int fromIndex, int toIndex)
  {
    if (fromIndex < 0  ||  toIndex > length  ||  fromIndex > toIndex) {
      throw new IndexOutOfBoundsException();
    }
    final int length = toIndex - fromIndex;
    return length == 0
            ? Collections.emptyList()
            : new UnmodifiableListLongArrayWrapper(array, start + fromIndex, length);
  }

  /**
   * Returns a list iterator of the elements in this list (in proper
   * sequence).
   *
   * @return a list iterator of the elements in this list (in proper
   *         sequence).
   */
  @Override
  @NotNull
  public ListIterator<Long> listIterator()
  {
    return new MyListIterator();
  }

  /**
   * Returns a list iterator of the elements in this list (in proper
   * sequence), starting at the specified position in this list.  The
   * specified index indicates the first element that would be returned by
   * an initial call to the <tt>next</tt> method.  An initial call to
   * the <tt>previous</tt> method would return the element with the
   * specified index minus one.
   *
   * @param index index of first element to be returned from the
   *              list iterator (by a call to the <tt>next</tt> method).
   * @return a list iterator of the elements in this list (in proper
   *         sequence), starting at the specified position in this list.
   * @throws IndexOutOfBoundsException if the index is out of range (index
   *                                   &lt; 0 || index &gt; size()).
   */
  @Override
  @NotNull
  public ListIterator<Long> listIterator(int index)
  {
    return new MyListIterator(index);
  }

  /**
   * Returns an array containing all of the elements in this list in proper
   * sequence; the runtime type of the returned array is that of the
   * specified array.  Obeys the general contract of the
   * <tt>Collection.toArray(Object[])</tt> method.
   *
   * @param a the array into which the elements of this list are to
   *          be stored, if it is big enough; otherwise, a new array of the
   *          same runtime type is allocated for this purpose.
   * @return an array containing the elements of this list.
   * @throws ArrayStoreException  if the runtime type of the specified array
   *                              is not a supertype of the runtime type of every element in
   *                              this list.
   * @throws NullPointerException if the specified array is <tt>null</tt>.
   */
  @Override
  @NotNull
  @SuppressWarnings("unchecked")
  public <S> S[] toArray(@NotNull S[] a)
  {
    if (a.length < length) {
      // note: no better way to do this in 1.5
      a = (S[])java.lang.reflect.Array.
          newInstance(a.getClass().getComponentType(), length);
    }

    for (int i = 0;  i < length;  ++i) {
      a[i] = (S)Long.valueOf(array[start + i]);
    }

    if (a.length > length) {
      a[length] = null;
    }
    return a;
  }
}
