// ============================================================================
// File:               Unique2
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            7/26/24 3:24 PM
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.function.FragileProcedure1;

import java.util.Objects;
import java.util.function.Consumer;

/**
 * Helper interface for a union of two exclusive values of different types.
 * <p>
 * This is mainly used to be able to compare both {@link Union}
 * and {@link NullableUnion}.
 * It can also be used as a type where both can appear.
 *
 * To ensure the invariances of unions, objects of this class cannot be constructed outside of this package.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since July 26, 2024
 * @param <T1> first possible type
 * @param <T2> second possible type
 */
public abstract class UnionBase<T1, T2>
{
  /**
   * Make this class unconstructable outside of this package.
   * This is the only reason this is implemented as an abstract class instead of an interface.
   */
  UnionBase() {}

  /**
   * Does this union contain a value of type {@code T1}?
   * @return {@code true}: if this union contains a value of type {@code T1}<br>
   *         {@code false}: if this union contains a value of type {@code T2}
   * @see #get1()
   */
  public abstract boolean has1();

  /**
   * Does this union contain a value of type {@code T2}?
   * @return {@code true}: if this union contains a value of type {@code T2}<br>
   *         {@code false}: if this union contains a value of type {@code T1}
   * @see #get2()
   */
  public abstract boolean has2();

  /**
   * Get the value if it is of type {@code T1}.
   * Throw an exception otherwise.
   *
   * @return value of type {@code T1}
   * @throws IllegalStateException if this union does contain a value of type {@code T2}
   * @see #has1()
   */
  public abstract T1 get1() throws IllegalStateException;

  /**
   * Get the value if it is of type {@code T2}.
   * Throw an exception otherwise.
   *
   * @return value of type {@code T2}
   * @throws IllegalStateException if this union does contain a value of type {@code T1}
   * @see #has1()
   */
  public abstract T2 get2() throws IllegalStateException;

  /**
   * Get the value contained in this union.
   * @return the value of this union, having either type {@code T1} or {@code T2}
   */
  public abstract Object getValue();

  /**
   * Call the appropriate handler depending on this unions' value.
   * @param handler1 called if this union contains a value of type {@code T1}
   * @param handler2 called if this union contains a value of type {@code T2}
   */
  public abstract void dispose(@NotNull Consumer<? super T1> handler1,
                               @NotNull Consumer<? super T2> handler2);

  /**
   * Call the given handler if this union contains a value of type {@code T1}.
   * @param handler called with the value of this union if the contained type is {@code T1}, not called otherwise
   * @return this union for chaining, allowing to call {@linkplain #disposeOnly2(Consumer)} or
   *         {@linkplain #disposeFragile2(FragileProcedure1)} next
   */
  @NotNull
  public abstract UnionBase<T1, T2> disposeOnly1(@NotNull Consumer<? super T1> handler);

  /**
   * Call the given handler if this union contains a value of type {@code T2}.
   * @param handler called with the value of this union if the contained type is {@code T2}, not called otherwise
   * @return this union for chaining, allowing to call {@linkplain #disposeOnly1(Consumer)} or
   *         {@linkplain #disposeFragile1(FragileProcedure1)} next
   */
  @NotNull
  public abstract UnionBase<T1, T2> disposeOnly2(@NotNull Consumer<? super T2> handler);

  /**
   * Call the given handler if this union contains a value of type {@code T1}.
   * Allows for a handler which might throw a checked exception.
   * @param handler called with the value of this union if the contained type is {@code T1}, not called otherwise
   * @return this union for chaining, allowing to call {@linkplain #disposeOnly2(Consumer)} or
   *         {@linkplain #disposeFragile2(FragileProcedure1)} next
   * @param <E> exception the handler might throw
   * @throws E if {@code handler} throws it
   */
  @NotNull
  public abstract <E extends Exception> UnionBase<T1, T2> disposeFragile1(@NotNull FragileProcedure1<E, ? super T1> handler) throws E;

  /**
   * Call the given handler if this union contains a value of type {@code T2}.
   * Allows for a handler which might throw a checked exception.
   * @param handler called with the value of this union if the contained type is {@code T2}, not called otherwise
   * @return this union for chaining, allowing to call {@linkplain #disposeOnly1(Consumer)} or
   *         {@linkplain #disposeFragile1(FragileProcedure1)} next
   * @param <E> exception the handler might throw
   * @throws E if {@code handler} throws it
   */
  @NotNull
  public abstract <E extends Exception> UnionBase<T1, T2> disposeFragile2(@NotNull FragileProcedure1<E, ? super T2> handler) throws E;

  /**
   * Check whether two unions using their first type define the same value.
   * @param definitely1 a union which definitely contains a value of type {@code V1}
   * @param maybe1      a possible union which may contain a first value of type {@code V1}
   * @return {@code true}: {@code maybe1} is using its first type, and the object contained
   *                       is the same as the one from {@code definitely1}
   */
  static boolean isEqual1(@NotNull UnionBase<?, ?> definitely1, @Nullable Object maybe1)
  {
    if (maybe1 == definitely1) {
      return true;
    }
    if (!(maybe1 instanceof UnionBase)) {
      return false;
    }
    final UnionBase<?, ?> other = (UnionBase<?, ?>) maybe1;
    return other.has1() && Objects.equals(definitely1.getValue(), other.getValue());
  }

  /**
   * Check whether two unions using their first type define the same value.
   * @param definitely2 a union which definitely contains a value of type {@code V2}
   * @param maybe2      a possible union which may contain a first value of type {@code V2}
   * @return {@code true}: {@code maybe1} is using its second type, and the object contained
   *                       is the same as the one from {@code definitely1}
   */
  static boolean isEqual2(@NotNull UnionBase<?, ?> definitely2, @Nullable Object maybe2)
  {
    if (maybe2 == definitely2) {
      return true;
    }
    if (!(maybe2 instanceof UnionBase)) {
      return false;
    }
    final UnionBase<?, ?> other = (UnionBase<?, ?>) maybe2;
    return other.has2() && Objects.equals(definitely2.getValue(), other.getValue());
  }

}
