// ============================================================================
// File:               Union
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            7/26/24 2:02 PM
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.FragileProcedure1;

import java.util.Objects;
import java.util.function.Consumer;

/**
 * This class defines a union structure for two exclusive values of different types.
 * <p>
 * This is sometimes useful to transport errors from one part to the other, when
 * throwing them is no option. This is a type-safe implementation of a 2-member {@code union}
 * structure as defined in the C programming language.
 * <p>
 * This class allows no overriding to guarantee the following invariances:
 * <ul>
 *   <li>{@code u.has1() ^ u.has2()} is always {@code true}</li>
 * </ul>
 * A union is immutable in the sense that it itself does not change.
 * The contained values may change if they are mutable.
 * <p>
 * There is exactly one way to construct a union of either type:
 * <ul>
 *   <li>{@link #t1(Object)} will construct a union which contains a value of type {@code T1}</li>
 *   <li>{@link #t2(Object)} will construct a union which contains a value of type {@code T2}</li>
 * </ul>
 * <p>
 * This implementation does not allow {@code null} values.
 * Any consumers can safely assume that the values they achieve are non-{@code null}.
 * See {@link NullableUnion} if support for {@code null} is required.
 *
 * @param <T1> type of the first possible value
 * @param <T2> type of the second possible value
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since July 26, 2024
 */
public abstract class Union<T1, T2>
        extends UnionBase<T1, T2>
{
  /**
   * Don't allow construction from elsewhere.
   */
  private Union()
  {
  }

  /**
   * Get the value if it is of type {@code T1}.
   * Throw an exception otherwise.
   *
   * @return value of type {@code T1}
   * @throws IllegalStateException if this union does contain a value of type {@code T2}
   * @see #has1()
   */
  @NotNull
  public abstract T1 get1() throws IllegalStateException;

  /**
   * Get the value if it is of type {@code T2}.
   * Throw an exception otherwise.
   *
   * @return value of type {@code T2}
   * @throws IllegalStateException if this union does contain a value of type {@code T1}
   * @see #has1()
   */
  @NotNull
  public abstract T2 get2() throws IllegalStateException;

  /**
   * Get the value contained in this union.
   * @return the value of this union, having either type {@code T1} or {@code T2}
   */
  @Override
  @NotNull
  public abstract Object getValue();

  /**
   * Create a union containing a value of type V1.
   *
   * @param v1   value contained in this union
   * @param <V1> value type of contained value
   * @param <V2> implicit
   * @return union with value {@code v1}
   */
  @NotNull
  public static <V1, V2> Union<V1, V2> t1(@NotNull V1 v1)
  {
    if (v1 == null) {
      throw new IllegalArgumentException("null value is not allowed!");
    }
    return new Union<V1, V2>()
    {
      @Override
      public boolean has1()
      {
        return true;
      }

      @Override
      public boolean has2()
      {
        return false;
      }

      @NotNull
      @Override
      public V1 get1() throws IllegalStateException
      {
        return v1;
      }

      @NotNull
      @Override
      public V2 get2() throws IllegalStateException
      {
        throw new IllegalStateException("Requesting type 2 from a union containing type 1!");
      }

      @NotNull
      @Override
      public Object getValue()
      {
        return v1;
      }

      @Override
      public void dispose(@NotNull Consumer<? super V1> handler1, @NotNull Consumer<? super V2> handler2)
      {
        handler1.accept(v1);
      }

      @NotNull
      @Override
      public Union<V1, V2> disposeOnly1(@NotNull Consumer<? super V1> handler)
      {
        handler.accept(v1);
        return this;
      }

      @NotNull
      @Override
      public Union<V1, V2> disposeOnly2(@NotNull Consumer<? super V2> handler)
      {
        return this;
      }

      @NotNull
      @Override
      public <E extends Exception> Union<V1, V2> disposeFragile1(@NotNull FragileProcedure1<E, ? super V1> handler)
              throws E
      {
        handler.apply(v1);
        return this;
      }

      @NotNull
      @Override
      public <E extends Exception> Union<V1, V2> disposeFragile2(@NotNull FragileProcedure1<E, ? super V2> handler)
              throws E
      {
        return this;
      }

      @Override
      public int hashCode()
      {
        return Objects.hash(v1);
      }

      @Override
      public boolean equals(Object obj)
      {
        return UnionBase.isEqual1(this, obj);
      }

      @Override
      @NotNull
      public String toString()
      {
        return "Union{1:" + v1 + "}";
      }
    };
  }

  /**
   * Create a union containing a value of type V2.
   *
   * @param v2   value contained in this union
   * @param <V1> implicit
   * @param <V2> value type of contained value
   * @return union with value {@code v2}
   */
  @NotNull
  public static <V1, V2> Union<V1, V2> t2(@NotNull V2 v2)
  {
    if (v2 == null) {
      throw new IllegalArgumentException("null value is not allowed!");
    }
    return new Union<V1, V2>()
    {
      @Override
      public boolean has1()
      {
        return false;
      }

      @Override
      public boolean has2()
      {
        return true;
      }

      @NotNull
      @Override
      public V1 get1() throws IllegalStateException
      {
        throw new IllegalStateException("Requesting type 1 from a union containing type 2!");
      }

      @NotNull
      @Override
      public V2 get2() throws IllegalStateException
      {
        return v2;
      }

      @NotNull
      @Override
      public Object getValue()
      {
        return v2;
      }

      @Override
      public void dispose(@NotNull Consumer<? super V1> handler1, @NotNull Consumer<? super V2> handler2)
      {
        handler2.accept(v2);
      }

      @NotNull
      @Override
      public Union<V1, V2> disposeOnly1(@NotNull Consumer<? super V1> handler)
      {
        return this;
      }

      @NotNull
      @Override
      public Union<V1, V2> disposeOnly2(@NotNull Consumer<? super V2> handler)
      {
        handler.accept(v2);
        return this;
      }

      @NotNull
      @Override
      public <E extends Exception> Union<V1, V2> disposeFragile1(@NotNull FragileProcedure1<E, ? super V1> handler)
              throws E
      {
        return this;
      }

      @NotNull
      @Override
      public <E extends Exception> Union<V1, V2> disposeFragile2(@NotNull FragileProcedure1<E, ? super V2> handler)
              throws E
      {
        handler.apply(v2);
        return this;
      }

      @Override
      public int hashCode()
      {
        return Objects.hash(v2);
      }

      @Override
      public boolean equals(Object obj)
      {
        return UnionBase.isEqual1(this, obj);
      }

      @Override
      @NotNull
      public String toString()
      {
        return "Union{2:" + v2 + "}";
      }
    };
  }

}
