// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.util.Objects;

/**
 * A uniform matcher is a {@link Matcher} which can only compare
 * objects of the same type.
 * <p>
 * Besides {@link #IDENTICAL} which only accepts the same object as identical,
 * useful standard implementations are also
 * {@code Objects::equals} or {@code Objects::deepEquals}.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @see de.caff.generics.HashCodeCalculator
 */
@FunctionalInterface
public interface UniformMatcher<T>
        extends Matcher<T, T>
{
  /**
   * Uniform matcher which only accepts identical objects.
   * Don't use directly, prefer {@link #identical()}.
   */
  UniformMatcher<Object> IDENTICAL = (v1, v2) -> v1 == v2;
  /**
   * Uniform matcher which uses {@link Object#equals(Object)}.
   * Attention: this is fast but not {@code null}-safe.
   * Don't use directly, prefer {@link #natural()}.
   */
  UniformMatcher<Object> NATURAL = (v1, v2) -> v1.equals(v2);
  /**
   * Uniform matcher which uses {@link Objects#deepEquals(Object, Object)}.
   * Don't use directly, prefer {@link }
   */
  UniformMatcher<Object> NATURAL_NULL_SAFE = Objects::deepEquals;

  /**
   * Identical matcher.
   * This just returns {@link #IDENTICAL} but may be used in cases when
   * the interface does not accept matchers with a super type.
   * @param <V> value type
   * @return uniform matcher which only accepts identical objects
   */
  @NotNull
  @SuppressWarnings("unchecked")  // because everything is handled at object level read-only
  static <V> UniformMatcher<V> identical()
  {
    return (UniformMatcher<V>)IDENTICAL;
  }

  /**
   * Natural matcher.
   * This just returns {@link #NATURAL} but may be used in cases when
   * the interface does not accept matchers with a super type.
   * Attention: this is fast but not {@code null}-safe.
   * @param <V> value type
   * @return uniform matcher which only accepts identical objects
   */
  @NotNull
  @SuppressWarnings("unchecked")  // because everything is handled at object level read-only
  static <V> UniformMatcher<V> natural()
  {
    return (UniformMatcher<V>)NATURAL;
  }

  /**
   * Natural matcher.
   * This just returns {@link #NATURAL_NULL_SAFE} but may be used in cases when
   * the interface does not accept matchers with a super type.
   * @param <V> value type
   * @return uniform matcher which only accepts identical objects
   */
  @NotNull
  @SuppressWarnings("unchecked")  // because everything is handled at object level read-only
  static <V> UniformMatcher<V> naturalNullSafe()
  {
    return (UniformMatcher<V>)NATURAL_NULL_SAFE;
  }
}
