// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import java.util.*;
import java.util.function.Consumer;
import java.util.function.Function;

/**
 * Partial class used by Types, defining the map algorithms.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
final class TypesImplMap
{
  /**
   * Not to be constructed.
   */
  private TypesImplMap()
  {
  }

  /**
   * Add the mapped elements of an iterator into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link Types#map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   * <p>
   * See {@link Types#map(java.util.Collection, Iterable, Function)} for an easy to adapt example.
   *
   * @param target         collection where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  static <C extends Collection<? super T>, T, S> C mapImpl(@NotNull C target,
                                                           @NotNull Iterator<S> iterator,
                                                           @NotNull Function<? super S, T> typeConverter)
  {
    while (iterator.hasNext()) {
      target.add(typeConverter.apply(iterator.next()));
    }
    return target;
  }

  /**
   * Add the mapped elements of an iterator to a consumer.
   * <p>
   * This will add all mapped elements to the consumer.
   * See {@link Types#map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   *
   * @param target         consumer where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  static <C extends Consumer<? super T>, T, S> C mapImpl(@NotNull C target,
                                                         @NotNull Iterator<S> iterator,
                                                         @NotNull Function<? super S, T> typeConverter)
  {
    while (iterator.hasNext()) {
      target.accept(typeConverter.apply(iterator.next()));
    }
    return target;
  }

  /**
   * Add the mapped elements of an iterator into a map.
   * <p>
   * This will add all mapped elements to the target collection, using the
   * converted values as keys pointing to the incoming values.
   *
   * @param target         collection where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  static <M extends Map<? super T, ? super S>, T, S> M mapMImpl(@NotNull M target,
                                                                @NotNull Iterator<S> iterator,
                                                                @NotNull Function<? super S, T> typeConverter)
  {
    while (iterator.hasNext()) {
      final S value = iterator.next();
      target.put(typeConverter.apply(value), value);
    }
    return target;
  }

  /**
   * Add the mapped elements of an iterator into a map.
   * <p>
   * This will add all mapped elements to the target collection, using the
   * incoming values as keys pointing to the converted values.
   *
   * @param target         collection where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  static <M extends Map<? super S, ? super T>, T, S> M mapRImpl(@NotNull M target,
                                                                @NotNull Iterator<S> iterator,
                                                                @NotNull Function<? super S, T> typeConverter)
  {
    while (iterator.hasNext()) {
      final S key = iterator.next();
      target.put(key, typeConverter.apply(key));
    }
    return target;
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link Types#map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   * <p>
   * See {@link Types#map(java.util.Collection, Iterable, Function)} for an easy to adapt example.
   *
   * @param target         collection where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  static <C extends Collection<? super T>, T, S> C mapImpl(@NotNull C target,
                                                           @NotNull Enumeration<S> enumeration,
                                                           @NotNull Function<? super S, T> typeConverter)
  {
    while (enumeration.hasMoreElements()) {
      target.add(typeConverter.apply(enumeration.nextElement()));
    }
    return target;
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   * <p>
   * This will add all mapped elements to the target collection, using the
   * converted values as keys pointing to the incoming values.
   *
   * @param target         collection where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  static <M extends Map<? super T, ? super S>, T, S> M mapMImpl(@NotNull M target,
                                                                @NotNull Enumeration<S> enumeration,
                                                                @NotNull Function<? super S, T> typeConverter)
  {
    while (enumeration.hasMoreElements()) {
      S value = enumeration.nextElement();
      target.put(typeConverter.apply(value), value);
    }
    return target;
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   * <p>
   * This will add all mapped elements to the target collection, using the
   * converted values as keys pointing to the incoming values.
   *
   * @param target         collection where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  static <M extends Map<? super S, ? super T>, T, S> M mapRImpl(@NotNull M target,
                                                                @NotNull Enumeration<S> enumeration,
                                                                @NotNull Function<? super S, T> typeConverter)
  {
    while (enumeration.hasMoreElements()) {
      S key = enumeration.nextElement();
      target.put(key, typeConverter.apply(key));
    }
    return target;
  }

  /**
   * Map one map into another, while allowing to discard elements on the way.
   * <p>
   * An element is discarded if the type converter returns {@code null}.
   *
   * @param target         target map
   * @param source         source map
   * @param typeConverter  type converter converting map entries
   * @param <MT>           target map type
   * @param <KT>           target map key type
   * @param <VT>           target map value type
   * @param <KS>           source map key type
   * @param <VS>           source map value type
   * @return target map
   */
  @NotNull
  static <MT extends Map<? super KT, ? super VT>, KT, VT,
          KS, VS> MT
  mapImpl(@NotNull MT target,
          @NotNull Map<KS, VS> source,
          @NotNull Function<Map.Entry<KS, VS>, ? extends Map.Entry<KT, VT>> typeConverter)
  {
    for (Map.Entry<KS, VS> entry : source.entrySet()) {
      final Map.Entry<? extends KT, ? extends VT> converted = typeConverter.apply(entry);
      if (converted != null) {
        target.put(converted.getKey(), converted.getValue());
      }
    }
    return target;
  }

  /**
   * Map one map into another, allowing to discard or add elements on the way.
   *
   * @param target         target map
   * @param source         source map
   * @param typeConverter  type converter converting map entries
   * @param <MT>           target map type
   * @param <KT>           target map key type
   * @param <VT>           target map value type
   * @param <KS>           source map key type
   * @param <VS>           source map value type
   * @return target map
   */
  @NotNull
  static <MT extends Map<? super KT, ? super VT>, KT, VT,
          KS, VS> MT
  mapXImpl(@NotNull MT target,
           @NotNull Map<KS, VS> source,
           @NotNull Function<Map.Entry<? super KS, ? super VS>, ? extends Iterable<? extends Map.Entry<? extends KT, ? extends VT>>> typeConverter)
  {
    for (Map.Entry<KS, VS> entry : source.entrySet()) {
      final Iterable<? extends Map.Entry<? extends KT, ? extends VT>> iterable = typeConverter.apply(entry);
      for (Map.Entry<? extends KT, ? extends VT> result : iterable) {
        target.put(result.getKey(),  result.getValue());
      }
    }
    return target;
  }

  /**
   * Add the mapped elements of an iterator into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link Types#map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   * <p>
   * See {@link Types#map(java.util.Collection, Iterable, Function)} for an easy to adapt example.
   *
   * @param target         collection where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  static <C extends Collection<? super T>, T, S> C
  mapXImpl(@NotNull C target,
           @NotNull Iterator<S> iterator,
           @NotNull Function<? super S, ? extends Iterable<T>> typeConverter)
  {
    while (iterator.hasNext()) {
      for (T item : typeConverter.apply(iterator.next())) {
        target.add(item);
      }
    }
    return target;
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link Types#map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   * <p>
   * See {@link Types#map(java.util.Collection, Iterable, Function)} for an easy to adapt example.
   *
   * @param target         collection where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  static <C extends Collection<? super T>, T, S> C
  mapXImpl(@NotNull C target,
           @NotNull Enumeration<S> enumeration,
           @NotNull Function<? super S, ? extends Iterable<T>> typeConverter)
  {
    while (enumeration.hasMoreElements()) {
      for (T item : typeConverter.apply(enumeration.nextElement())) {
        target.add(item);
      }
    }
    return target;
  }

  /**
   * Add the mapped elements of an iterator into a collection.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         collection where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  static <C extends Collection<? super T>, T, S> C
  mapImpl(@NotNull C target,
          @NotNull Iterator<S> iterator,
          @NotNull Function<? super S, T> typeConverter,
          @Nullable T deleteMark)
  {
    while (iterator.hasNext()) {
      final T result = typeConverter.apply(iterator.next());
      if (deleteMark != result) {
        target.add(result);
      }
    }
    return target;
  }

  /**
   * Forward the mapped elements of an iterator to a consumer.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   *
   * @param target         consumer to which each element is mapped
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  static <C extends Consumer<? super T>, T, S> C
  mapImpl(@NotNull C target,
          @NotNull Iterator<S> iterator,
          @NotNull Function<? super S, T> typeConverter,
          @Nullable T deleteMark)
  {
    while (iterator.hasNext()) {
      final T result = typeConverter.apply(iterator.next());
      if (deleteMark != result) {
        target.accept(result);
      }
    }
    return target;
  }

  /**
   * Add the mapped elements of an iterator into a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  static <M extends Map<? super T, ? super S>, T, S> M
  mapMImpl(@NotNull M target,
           @NotNull Iterator<S> iterator,
           @NotNull Function<? super S, T> typeConverter,
           @Nullable T deleteMark)
  {
    while (iterator.hasNext()) {
      final S next = iterator.next();
      final T result = typeConverter.apply(next);
      if (deleteMark != result) {
        target.put(result, next);
      }
    }
    return target;
  }

  /**
   * Add the mapped elements of an iterator into a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  static <M extends Map<? super S, ? super T>, T, S> M
  mapRImpl(@NotNull M target,
           @NotNull Iterator<S> iterator,
           @NotNull Function<? super S, T> typeConverter,
           @Nullable T deleteMark)
  {
    while (iterator.hasNext()) {
      final S next = iterator.next();
      final T result = typeConverter.apply(next);
      if (deleteMark != result) {
        target.put(next, result);
      }
    }
    return target;
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         collection where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  static <C extends Collection<? super T>, T, S> C
  mapImpl(@NotNull C target,
          @NotNull Enumeration<S> enumeration,
          @NotNull Function<? super S, T> typeConverter,
          @Nullable T deleteMark)
  {
    while (enumeration.hasMoreElements()) {
      final T result = typeConverter.apply(enumeration.nextElement());
      if (deleteMark != result) {
        target.add(result);
      }
    }
    return target;
  }

  /**
   * Add the mapped elements of an enumeration into a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  static <M extends Map<? super T, ? super S>, T, S> M
  mapMImpl(@NotNull M target,
           @NotNull Enumeration<S> enumeration,
           @NotNull Function<? super S, T> typeConverter,
           @Nullable T deleteMark)
  {
    while (enumeration.hasMoreElements()) {
      final S next = enumeration.nextElement();
      final T result = typeConverter.apply(next);
      if (deleteMark != result) {
        target.put(result, next);
      }
    }
    return target;
  }

  /**
   * Add the mapped elements of an enumeration into a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  static <M extends Map<? super S, ? super T>, T, S> M
  mapRImpl(@NotNull M target,
           @NotNull Enumeration<S> enumeration,
           @NotNull Function<? super S, T> typeConverter,
           @Nullable T deleteMark)
  {
    while (enumeration.hasMoreElements()) {
      final S next = enumeration.nextElement();
      final T result = typeConverter.apply(next);
      if (deleteMark != result) {
        target.put(next, result);
      }
    }
    return target;
  }

  /**
   * Add the mapped elements of an iterator into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link Types#map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   * <p>
   * See {@link Types#map(java.util.Collection, Iterable, Function)} for an easy to adapt example.
   *
   * @param target         collection where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  static <C extends Collection<? super T>, T, S>
  C mapEImpl(@NotNull C target,
             @NotNull Iterator<S> iterator,
             @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    while (iterator.hasNext()) {
      target.add(typeConverter.convert(iterator.next()));
    }
    return target;
  }

  /**
   * Add the mapped elements of an iterator to a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This will add all mapped elements to the target map.
   * See {@link Types#mapM(java.util.Map, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   * <p>
   * See {@link Types#map(java.util.Collection, Iterable, Function)} for an easy to adapt example.
   *
   * @param target         map where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  static <M extends Map<? super T, ? super S>, T, S>
  M mapMEImpl(@NotNull M target,
              @NotNull Iterator<S> iterator,
              @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    while (iterator.hasNext()) {
      final S next = iterator.next();
      target.put(typeConverter.convert(next), next);
    }
    return target;
  }

  /**
   * Add the mapped elements of an iterator to a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This will add all mapped elements to the target map.
   * See {@link Types#mapR(java.util.Map, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  static <M extends Map<? super S, ? super T>, T, S>
  M mapREImpl(@NotNull M target,
              @NotNull Iterator<S> iterator,
              @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    while (iterator.hasNext()) {
      final S next = iterator.next();
      target.put(next, typeConverter.convert(next));
    }
    return target;
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link Types#map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         collection where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  static <C extends Collection<? super T>, T, S> C
  mapEImpl(@NotNull C target,
           @NotNull Enumeration<S> enumeration,
           @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    while (enumeration.hasMoreElements()) {
      target.add(typeConverter.convert(enumeration.nextElement()));
    }
    return target;
  }

  /**
   * Add the mapped elements of an enumeration to a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This will add all mapped elements to the target map.
   * See {@link Types#mapM(java.util.Map, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  static <M extends Map<? super T, ? super S>, T, S> M
  mapMEImpl(@NotNull M target,
            @NotNull Enumeration<S> enumeration,
            @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    while (enumeration.hasMoreElements()) {
      final S next = enumeration.nextElement();
      target.put(typeConverter.convert(next), next);
    }
    return target;
  }

  /**
   * Add the mapped elements of an enumeration to a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This will add all mapped elements to the target map.
   * See {@link Types#mapM(java.util.Map, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  static <M extends Map<? super S, ? super T>, T, S> M
  mapREImpl(@NotNull M target,
            @NotNull Enumeration<S> enumeration,
            @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    while (enumeration.hasMoreElements()) {
      final S next = enumeration.nextElement();
      target.put(next, typeConverter.convert(next));
    }
    return target;
  }

  /**
   * Add the mapped elements of an iterator into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link Types#map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * See {@link Types#map(java.util.Collection, Iterable, Function)} for an easy to adapt example.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         collection where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  static <C extends Collection<? super T>, T, S>
  C mapXEImpl(@NotNull C target,
              @NotNull Iterator<S> iterator,
              @NotNull FragileTypeConverter<? extends Iterable<T>, ? super S> typeConverter)
          throws TypeConverterException
  {
    while (iterator.hasNext()) {
      for (T item : typeConverter.convert(iterator.next())) {
        target.add(item);
      }
    }
    return target;
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link Types#map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   * <p>
   * See {@link Types#map(java.util.Collection, Iterable, Function)} for an easy to adapt example.
   *
   * @param target         collection where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  static <C extends Collection<? super T>, T, S> C
  mapXEImpl(@NotNull C target,
            @NotNull Enumeration<S> enumeration,
            @NotNull FragileTypeConverter<? extends Iterable<T>, ? super S> typeConverter)
          throws TypeConverterException
  {
    while (enumeration.hasMoreElements()) {
      for (T item : typeConverter.convert(enumeration.nextElement())) {
        target.add(item);
      }
    }
    return target;
  }

  /**
   * Add the mapped elements of an iterator into a collection.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         collection where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  static <C extends Collection<? super T>, T, S> C
  mapEImpl(@NotNull C target,
           @NotNull Iterator<S> iterator,
           @NotNull FragileTypeConverter<T, ? super S> typeConverter,
           @Nullable T deleteMark)
          throws TypeConverterException
  {
    while (iterator.hasNext()) {
      final T result = typeConverter.convert(iterator.next());
      if (deleteMark != result) {
        target.add(result);
      }
    }
    return target;
  }

  /**
   * Add the mapped elements of an iterator into a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  static <M extends Map<? super T, ? super S>, T, S> M
  mapMEImpl(@NotNull M target,
            @NotNull Iterator<S> iterator,
            @NotNull FragileTypeConverter<T, ? super S> typeConverter,
            @Nullable T deleteMark)
          throws TypeConverterException
  {
    while (iterator.hasNext()) {
      final S next = iterator.next();
      final T result = typeConverter.convert(next);
      if (deleteMark != result) {
        target.put(result, next);
      }
    }
    return target;
  }

  /**
   * Add the mapped elements of an iterator into a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  static <M extends Map<? super S, ? super T>, T, S> M
  mapREImpl(@NotNull M target,
            @NotNull Iterator<S> iterator,
            @NotNull FragileTypeConverter<T, ? super S> typeConverter,
            @Nullable T deleteMark)
          throws TypeConverterException
  {
    while (iterator.hasNext()) {
      final S next = iterator.next();
      final T result = typeConverter.convert(next);
      if (deleteMark != result) {
        target.put(next, result);
      }
    }
    return target;
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         collection where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  static <C extends Collection<? super T>, T, S> C
  mapEImpl(@NotNull C target,
           @NotNull Enumeration<S> enumeration,
           @NotNull FragileTypeConverter<T, ? super S> typeConverter,
           @Nullable T deleteMark)
          throws TypeConverterException
  {
    while (enumeration.hasMoreElements()) {
      final T result = typeConverter.convert(enumeration.nextElement());
      if (deleteMark != result) {
        target.add(result);
      }
    }
    return target;
  }

  /**
   * Add the mapped elements of an enumeration to a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  static <M extends Map<? super T, ? super S>, T, S> M
  mapMEImpl(@NotNull M target,
            @NotNull Enumeration<S> enumeration,
            @NotNull FragileTypeConverter<T, ? super S> typeConverter,
            @Nullable T deleteMark)
          throws TypeConverterException
  {
    while (enumeration.hasMoreElements()) {
      final S next = enumeration.nextElement();
      final T result = typeConverter.convert(next);
      if (deleteMark != result) {
        target.put(result, next);
      }
    }
    return target;
  }

  /**
   * Add the mapped elements of an enumeration to a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  static <M extends Map<? super S, ? super T>, T, S> M
  mapREImpl(@NotNull M target,
            @NotNull Enumeration<S> enumeration,
            @NotNull FragileTypeConverter<T, ? super S> typeConverter,
            @Nullable T deleteMark)
          throws TypeConverterException
  {
    while (enumeration.hasMoreElements()) {
      final S next = enumeration.nextElement();
      final T result = typeConverter.convert(next);
      if (deleteMark != result) {
        target.put(next, result);
      }
    }
    return target;
  }

}
