// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.util.*;
import java.util.function.BiFunction;

/**
 * Partial class used by Types, defining the folding algorithms.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
final class TypesImplFolding
{
  /**
   * Avoid creation.
   */
  private TypesImplFolding()
  {
  }

  /**
   * Do left-associative folding of an iterator.
   * <p>
   * Folding is the reduction of the elements of a collection into one value,
   * by applying the same function on each element with the result of the
   * last call.
   * <p>
   * See {@link de.caff.generics.Types#foldLeft(Iterable, Object, BiFunction)} for a code example.
   * @param iterator      iterator which elements are folded together
   * @param initialValue  start value
   * @param folder        folding function, the apply method gets the accumulated value
   *                      as its first and the element as its second argument,
   *                      and returns the result of their combination
   * @param <V>           accumulated value type
   * @param <T>           iterable type
   * @return accumulated value
   */
  public static <V, T> V foldLeftImpl(@NotNull Iterator<T> iterator,
                                      V initialValue,
                                      @NotNull BiFunction<? super V, ? super T, ? extends V> folder)
  {
    V value = initialValue;
    while (iterator.hasNext()) {
      value = folder.apply(value, iterator.next());
    }
    return value;
  }

  /**
   * Do left-associative folding of an enumeration.
   * <p>
   * Folding is the reduction of the elements of a collection into one value,
   * by applying the same function on each element with the result of the
   * last call.
   * <p>
   * See {@link de.caff.generics.Types#foldLeft(Iterable, Object, BiFunction)} for a code example.
   * @param enumeration   enumeration which elements are folded together
   * @param initialValue  start value
   * @param folder        folding function, the apply method gets the accumulated value
   *                      as its first and the element as its second argument,
   *                      and returns the result of their combination
   * @param <V>           accumulated value type
   * @param <T>           iterable type
   * @return accumulated value
   */
  public static <V, T> V foldLeftImpl(@NotNull Enumeration<T> enumeration,
                                      V initialValue,
                                      @NotNull BiFunction<? super V, ? super T, ? extends V> folder)
  {
    V value = initialValue;
    while (enumeration.hasMoreElements()) {
      value = folder.apply(value, enumeration.nextElement());
    }
    return value;
  }

  /**
   * Do right-associative folding of an iteration.
   * <p>
   * Folding is the reduction of the elements of a collection into one value,
   * by applying the same function on each element with the result of the
   * last call.
   *
   * @param list          list which elements are folded together
   * @param initialValue  start value
   * @param folder        folding function, the apply method gets the accumulated value
   *                      as its first and the element as its second argument,
   *                      and returns the result of their combination
   * @param <V>           accumulated value type
   * @param <T>           iterator type
   * @return accumulated value
   */
  public static <V, T> V foldRightImpl(@NotNull List<T> list,
                                       V initialValue,
                                       @NotNull BiFunction<? super V, ? super T, ? extends V> folder)
  {
    return foldRightImpl(list.listIterator(list.size() - 1), initialValue, folder);
  }

  /**
   * Do right-associative folding of a list iterator using it in reverse direction.
   * <p>
   * Folding is the reduction of the elements of a collection into one value,
   * by applying the same function on each element with the result of the
   * last call.
   *
   * @param reverseIterator iterator which is traverse in reverse direction
   * @param initialValue  start value
   * @param folder        folding function, the apply method gets the accumulated value
   *                      as its first and the element as its second argument,
   *                      and returns the result of their combination
   * @param <V>           accumulated value type
   * @param <T>           list type
   * @return accumulated value
   */
  public static <V, T> V foldRightImpl(@NotNull ListIterator<T> reverseIterator,
                                       V initialValue,
                                       @NotNull BiFunction<? super V, ? super T, ? extends V> folder)
  {
    V value = initialValue;
    while (reverseIterator.hasPrevious()) {
      value = folder.apply(value, reverseIterator.previous());
    }
    return value;
  }

}
