// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.util.Collection;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.function.Predicate;

/**
 * Partial class used by Types, defining the filter algorithms.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
final class TypesImpFilter
{
  /**
   * Avoid creation.
   */
  private TypesImpFilter()
  {
  }

  /**
   * Filter the elements of an iterator into a collection.
   * <p>
   * This will add all filtered elements to the target collection.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         collection where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param checker        value checker to decide which elements are kept
   * @param <C>            collection type
   * @param <T>            filtered type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <C extends Collection<? super T>, T> C
  filterImpl(@NotNull C target,
             @NotNull Iterator<T> iterator,
             @NotNull Predicate<? super T> checker)
  {
    while (iterator.hasNext()) {
      T elem = iterator.next();
      if (checker.test(elem)) {
        target.add(elem);
      }
    }
    return target;
  }

  /**
   * Filter the elements of an iterator into a collection.
   * <p>
   * This will add all filtered elements to the target collection.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         collection where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param checker        value checker to decide which elements are kept
   * @param <C>            collection type
   * @param <T>            filtered type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <C extends Collection<? super T>, T> C
  filterImpl(@NotNull C target,
             @NotNull Enumeration<T> enumeration,
             @NotNull Predicate<? super T> checker)
  {
    while (enumeration.hasMoreElements()) {
      T elem = enumeration.nextElement();
      if (checker.test(elem)) {
        target.add(elem);
      }
    }
    return target;
  }

}
