// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.checker.Check;
import de.caff.generics.converter.*;
import de.caff.generics.function.*;
import de.caff.generics.handler.CarelessLoopItemHandler;
import de.caff.generics.matcher.Match;
import de.caff.version.ModuleVersion;
import de.caff.version.ModuleVersionService;

import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.security.InvalidParameterException;
import java.util.*;
import java.util.function.*;

import static de.caff.generics.TypesImpFilter.filterImpl;
import static de.caff.generics.TypesImplFolding.foldLeftImpl;
import static de.caff.generics.TypesImplFolding.foldRightImpl;
import static de.caff.generics.TypesImplMap.*;

/**
 * Tool with helpful methods for type conversion.
 * <p>
 * Have a look at <a href="package.html">package generics</a> to get an overview.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public final class Types
{
  /**
   * Release of this file.
   * @deprecated Use {@link ModuleVersion#VERSION} instead, or
   * {@link ModuleVersionService#iterator()}/{@link ModuleVersionService#collectServices()} for all modules
   */
  @Deprecated
  public static final String RELEASE_DATE = ModuleVersion.getReleaseDate();
  /**
   * Version number of this file.
   * @deprecated use {@link ModuleVersion#VERSION} instead
   */
  @Deprecated
  public static final String VERSION = ModuleVersion.VERSION.withNoBuild().toString();

  /**
   * Comparator which compares Comparables in their natural order.
   * Will throw {@link NullPointerException} on {@code null} values.
   */
  @NotNull
  @SuppressWarnings("rawtypes")
  public static final Comparator<? extends Comparable> COMPARABLE_COMPARATOR =
          new Comparator<Comparable<?>>()
          {
            @Override
            @SuppressWarnings({ "unchecked", "rawtype" })
            public int compare(Comparable o1, Comparable o2)
            {
              return o1.compareTo(o2);
            }
          };

  /**
   * Comparator which compares the toString() representation.
   * Will throw {@link NullPointerException} on {@code null} values.
   */
  public static final Comparator<?> TO_STRING_COMPARATOR =
          Comparator.comparing(Object::toString);
  /**
   * Comparator which compares the toString() representation in case-insensitive order.
   * Will throw {@link NullPointerException} on {@code null} values.
   */
  public static final Comparator<?> TO_STRING_IGNORE_CASE_COMPARATOR =
          (Comparator<Object>)(o1, o2) -> o1.toString().compareToIgnoreCase(o2.toString());

  /**
   * Empty iterator which does not iterate.
   */
  private static final Iterator<Object> EMPTY_ITERATOR = new Iterator<Object>()
  {
    /**
     * Returns <tt>true</tt> if the iteration has more elements. (In other
     * words, returns <tt>true</tt> if <tt>next</tt> would return an element
     * rather than throwing an exception.)
     *
     * @return <tt>true</tt> if the iterator has more elements.
     */
    @Override
    public boolean hasNext()
    {
      return false;
    }

    /**
     * Returns the next element in the iteration.
     *
     * @return the next element in the iteration.
     * @throws java.util.NoSuchElementException iteration has no more elements.
     */
    @Override
    public Object next()
    {
      throw new NoSuchElementException("Empty iterator cannot iterate!");
    }

    /**
     * Removes from the underlying collection the last element returned by the
     * iterator (optional operation).  This method can be called only once per
     * call to <tt>next</tt>.  The behavior of an iterator is unspecified if
     * the underlying collection is modified while the iteration is in
     * progress in any way other than by calling this method.
     *
     * @throws UnsupportedOperationException if the <tt>remove</tt>
     *                                       operation is not supported by this Iterator.
     * @throws IllegalStateException         if the <tt>next</tt> method has not
     *                                       yet been called, or the <tt>remove</tt> method has already
     *                                       been called after the last call to the <tt>next</tt>
     *                                       method.
     */
    @Override
    public void remove()
    {
      throw new IllegalStateException("Cannot remove without iteration!");
    }

    @Override
    public void forEachRemaining(Consumer<? super Object> action)
    {
    }
  };

  /** Empty primitve double iterator. */
  public static final PrimitiveIterator.OfDouble EMPTY_DOUBLE_ITERATOR =
          new PrimitiveIterator.OfDouble()
          {
            @Override
            public double nextDouble()
            {
              throw new NoSuchElementException("Empty iterator has no elements!");
            }

            @Override
            public boolean hasNext()
            {
              return false;
            }

            @Override
            public void forEachRemaining(DoubleConsumer action)
            {
            }

            @Override
            public Double next()
            {
              throw new NoSuchElementException("Empty iterator has no elements!");
            }

            @Override
            public void forEachRemaining(Consumer<? super Double> action)
            {
            }
          };
  /** Empty primitive int iterator. */
  public static final PrimitiveIterator.OfInt EMPTY_INT_ITERATOR =
          new PrimitiveIterator.OfInt()
          {
            @Override
            public int nextInt()
            {
              throw new NoSuchElementException("Empty iterator has no elements!");
            }

            @Override
            public boolean hasNext()
            {
              return false;
            }

            @Override
            public void forEachRemaining(IntConsumer action)
            {
            }

            @Override
            public Integer next()
            {
              throw new NoSuchElementException("Empty iterator has no elements!");
            }

            @Override
            public void forEachRemaining(Consumer<? super Integer> action)
            {
            }
          };
  /** Empty primitive long iterator. */
  public static final PrimitiveIterator.OfLong EMPTY_LONG_ITERATOR =
          new PrimitiveIterator.OfLong()
          {
            @Override
            public long nextLong()
            {
              throw new NoSuchElementException("Empty iterator has no elements!");
            }

            @Override
            public boolean hasNext()
            {
              return false;
            }

            @Override
            public void forEachRemaining(LongConsumer action)
            {
            }

            @Override
            public Long next()
            {
              throw new NoSuchElementException("Empty iterator has no elements!");
            }

            @Override
            public void forEachRemaining(Consumer<? super Long> action)
            {
            }
          };
  /** Empty primitive boolean iterator. */
  public static final PrimitiveBooleanIterator EMPTY_BOOLEAN_ITERATOR = PrimitiveBooleanIterator.EMPTY;
  /** Empty primitive byte iterator. */
  public static final PrimitiveByteIterator EMPTY_BYTE_ITERATOR = PrimitiveByteIterator.EMPTY;
  /** Empty primitive short iterator. */
  public static final PrimitiveShortIterator EMPTY_SHORT_ITERATOR = PrimitiveShortIterator.EMPTY;
  /** Empty primitive char iterator. */
  public static final PrimitiveCharIterator EMPTY_CHAR_ITERATOR = PrimitiveCharIterator.EMPTY;
  /** Empty primitive float iterator. */
  public static final PrimitiveFloatIterator EMPTY_FLOAT_ITERATOR = PrimitiveFloatIterator.EMPTY;

  /** Hash value for empty collections. */
  public static final int EMPTY_HASH = 1;

  /**
   * Empty list iterator which does not iterate.
   */
  private static final ListIterator<Object> EMPTY_LIST_ITERATOR = new ListIterator<Object>()
  {
    @Override
    public boolean hasNext()
    {
      return false;
    }

    @Override
    public Object next()
    {
      throw new NoSuchElementException("Empty iterator cannot iterate!");
    }

    @Override
    public boolean hasPrevious()
    {
      return false;
    }

    @Override
    public Object previous()
    {
      throw new NoSuchElementException("Empty iterator cannot iterate!");
    }

    @Override
    public int nextIndex()
    {
      return 0;
    }

    @Override
    public int previousIndex()
    {
      return -EMPTY_HASH;
    }

    @Override
    public void remove()
    {
      throw new UnsupportedOperationException("remove");
    }

    @Override
    public void set(Object o)
    {
      throw new UnsupportedOperationException("set");
    }

    @Override
    public void add(Object o)
    {
      throw new UnsupportedOperationException("add");
    }

    @Override
    public void forEachRemaining(Consumer<? super Object> action)
    {
    }
  };

  /**
   * Empty enumeration which does not iterate.
   */
  private static final Enumeration<Object> EMPTY_ENUMERATION = new Enumeration<Object>()
  {
    @Override
    public boolean hasMoreElements()
    {
      return false;
    }

    @Override
    public Object nextElement()
    {
      throw new NoSuchElementException("Empty enumeration cannot iterate!");
    }
  };

  /**
   * Empty iterable which does not iterate.
   */
  private static final Iterable<Object> EMPTY_ITERABLE = () -> EMPTY_ITERATOR;

  /**
   * Not to be created.
   */
  private Types()
  {
  }

  /** The generic copy converter. */
  @SuppressWarnings("rawtypes")
  private static final CopyConverter<?> COPY_CONVERTER = new CopyConverter();

  /**
   * A type converter which converts any object into its toString() representation.
   * It is safe for {@code null} values.
   */
  @NotNull
  public static final Function1<String, Object> TO_STRING = Objects::toString;

  /** A type converter allowing any number based type to be converted to double. */
  @NotNull
  public static final Function1<Double, Number> NUMBER_TO_DOUBLE =
          object -> object != null ? object.doubleValue() : null;

  /** A type converter allowing any number based type to be converted to float. */
  @NotNull
  public static final Function1<Float, Number> NUMBER_TO_FLOAT =
          object -> object != null ? object.floatValue() : null;

  /** A type converter allowing any number based type to be converted to long. */
  @NotNull
  public static final Function1<Long, Number> NUMBER_TO_LONG =
          object -> object != null ? object.longValue() : null;

  /** A type converter allowing any number based type to be converted to integer. */
  @NotNull
  public static final Function1<Integer, Number> NUMBER_TO_INT =
          object -> object != null ? object.intValue() : null;

  /** A type converter allowing any number based type to be converted to short integer. */
  @NotNull
  public static final Function1<Short, Number> NUMBER_TO_SHORT =
          object -> object != null ? object.shortValue() : null;

  /** A type converter allowing any number based type to be converted to a byte value. */
  @NotNull
  public static final Function1<Byte, Number> NUMBER_TO_BYTE =
          object -> object != null ? object.byteValue() : null;

  /**
   * A converter which converts strings to integer, using base 10.
   * <p>
   * It may throw a {@link de.caff.generics.TypeConverterException} wrapping
   * a {@link java.lang.NumberFormatException}
   */
  @NotNull
  public static final FragileTypeConverter<Integer, String> STR_TO_INT =
          new StringToIntegerTypeConverter();
  /**
   * A converter which converts strings to integer, using base 16.
   * <p>
   * It may throw a {@link de.caff.generics.TypeConverterException} wrapping
   * a {@link java.lang.NumberFormatException}
   */
  @NotNull
  public static final FragileTypeConverter<Integer, String> STR_TO_INT_BASE_16 =
          new StringToIntegerTypeConverter(16);
  /**
   * A converter which converts strings to integer, using base 8.
   * <p>
   * It may throw a {@link de.caff.generics.TypeConverterException} wrapping
   * a {@link java.lang.NumberFormatException}
   */
  @NotNull
  public static final FragileTypeConverter<Integer, String> STR_TO_INT_BASE_8 =
          new StringToIntegerTypeConverter(8);
  /**
   * A converter which converts strings to integer, using base 2.
   * <p>
   * It may throw a {@link de.caff.generics.TypeConverterException} wrapping
   * a {@link java.lang.NumberFormatException}
   */
  @NotNull
  public static final FragileTypeConverter<Integer, String> STR_TO_INT_BASE_2 =
          new StringToIntegerTypeConverter(2);

  /**
   * A converter which converts strings to long integer, using base 10.
   * <p>
   * It may throw a {@link de.caff.generics.TypeConverterException} wrapping
   * a {@link java.lang.NumberFormatException}
   */
  @NotNull
  public static final FragileTypeConverter<Long, String> STR_TO_LONG =
          new StringToLongTypeConverter();
  /**
   * A converter which converts strings to long integer, using base 16.
   * <p>
   * It may throw a {@link de.caff.generics.TypeConverterException} wrapping
   * a {@link java.lang.NumberFormatException}
   */
  @NotNull
  public static final FragileTypeConverter<Long, String> STR_TO_LONG_BASE_16 =
          new StringToLongTypeConverter(16);
  /**
   * A converter which converts strings to long integer, using base 8.
   * <p>
   * It may throw a {@link de.caff.generics.TypeConverterException} wrapping
   * a {@link java.lang.NumberFormatException}
   */
  @NotNull
  public static final FragileTypeConverter<Long, String> STR_TO_LONG_BASE_8 =
          new StringToLongTypeConverter(8);
  /**
   * A converter which converts strings to long integer, using base 2.
   * <p>
   * It may throw a {@link de.caff.generics.TypeConverterException} wrapping
   * a {@link java.lang.NumberFormatException}
   */
  @NotNull
  public static final FragileTypeConverter<Long, String> STR_TO_LONG_BASE_2 =
          new StringToLongTypeConverter(2);

  /**
   * A converter which converts strings to short integer, using base 10.
   * <p>
   * It may throw a {@link de.caff.generics.TypeConverterException} wrapping
   * a {@link java.lang.NumberFormatException}
   */
  @NotNull
  public static final FragileTypeConverter<Short, String> STR_TO_SHORT =
          new StringToShortTypeConverter();
  /**
   * A converter which converts strings to short integer, using base 16.
   * <p>
   * It may throw a {@link de.caff.generics.TypeConverterException} wrapping
   * a {@link java.lang.NumberFormatException}
   */
  @NotNull
  public static final FragileTypeConverter<Short, String> STR_TO_SHORT_BASE_16 =
          new StringToShortTypeConverter(16);
  /**
   * A converter which converts strings to short integer, using base 8.
   * <p>
   * It may throw a {@link de.caff.generics.TypeConverterException} wrapping
   * a {@link java.lang.NumberFormatException}
   */
  @NotNull
  public static final FragileTypeConverter<Short, String> STR_TO_SHORT_BASE_8 =
          new StringToShortTypeConverter(8);
  /**
   * A converter which converts strings to short integer, using base 2.
   * <p>
   * It may throw a {@link de.caff.generics.TypeConverterException} wrapping
   * a {@link java.lang.NumberFormatException}
   */
  @NotNull
  public static final FragileTypeConverter<Short, String> STR_TO_SHORT_BASE_2 =
          new StringToShortTypeConverter(2);

  /**
   * A converter which converts strings to byte values, using base 10.
   * <p>
   * It may throw a {@link de.caff.generics.TypeConverterException} wrapping
   * a {@link java.lang.NumberFormatException}
   */
  @NotNull
  public static final FragileTypeConverter<Byte, String> STR_TO_BYTE =
          new StringToByteTypeConverter();
  /**
   * A converter which converts strings to byte values, using base 16.
   * <p>
   * It may throw a {@link de.caff.generics.TypeConverterException} wrapping
   * a {@link java.lang.NumberFormatException}
   */
  @NotNull
  public static final FragileTypeConverter<Byte, String> STR_TO_BYTE_BASE_16 =
          new StringToByteTypeConverter(16);
  /**
   * A converter which converts strings to byte values, using base 8.
   * <p>
   * It may throw a {@link de.caff.generics.TypeConverterException} wrapping
   * a {@link java.lang.NumberFormatException}
   */
  @NotNull
  public static final FragileTypeConverter<Byte, String> STR_TO_BYTE_BASE_8 =
          new StringToByteTypeConverter(8);
  /**
   * A converter which converts strings to byte values, using base 2.
   * <p>
   * It may throw a {@link de.caff.generics.TypeConverterException} wrapping
   * a {@link java.lang.NumberFormatException}
   */
  @NotNull
  public static final FragileTypeConverter<Byte, String> STR_TO_BYTE_BASE_2 =
          new StringToByteTypeConverter(2);

  /**
   * A converter which converts strings (in Java number format) to double values.
   * <p>
   * It may throw a {@link de.caff.generics.TypeConverterException} wrapping
   * a {@link java.lang.NumberFormatException}
   */
  @NotNull
  public static final FragileTypeConverter<Double, String> STR_TO_DOUBLE =
          new StringToDoubleTypeConverter();

  /**
   * A converter which converts strings (in Java number format) to float values.
   * <p>
   * It may throw a {@link de.caff.generics.TypeConverterException} wrapping
   * a {@link java.lang.NumberFormatException}
   */
  @NotNull
  public static final FragileTypeConverter<Float, String> STR_TO_FLOAT =
          new StringToFloatTypeConverter();

  /**
   * A converter which converts strings to boolean values.
   * <p>
   * It follows the semantics of {@link Boolean#valueOf(String)},
   * but will pass through {@code null} values.
   * .
   */
  @NotNull
  public static final Function1<Boolean, String> STR_TO_BOOLEAN =
          object -> object != null ? Boolean.valueOf(object) : null;

  /**
   * A converter which splits strings into words.
   */
  public static final Function1<List<String>, String> WORD_SPLIT =
          new StringSplitTypeConverter("\\W");

  /**
   * A converter which changes strings to lowercase.
   */
  public static final Function1<String, String> STR_TO_LOWER =
          String::toLowerCase;

  /**
   * A converter which changes strings to uppercase.
   */
  public static final Function1<String, String> STR_TO_UPPER =
          String::toUpperCase;

  /**
   * Predicate for filtering out null objects.
   */
  public static final Predicate1<Object> NOT_NULL =
          Objects::nonNull;

  /**
   * Get a comparator which sorts comparables in their natural order.
   * @param type comparable type class
   * @param <T>  comparable type
   * @return natural order comparator
   * @deprecated use {@link #naturalOrder()} instead
   */
  @NotNull
  @Deprecated
  public static <T extends Comparable<T>> Comparator<T> naturalOrder(@NotNull Class<T> type)
  {
    return naturalOrder();
  }

  /**
   * Get a comparator which sorts comparables in their natural order.
   * @param list collection with comparable type
   * @param <T>  comparable type
   * @return natural order comparator
   * @deprecated use {@link #naturalOrder()} instead
   */
  @NotNull
  @Deprecated
  public static <T extends Comparable<T>> Comparator<T> naturalOrder(@NotNull Collection<T> list)
  {
    return naturalOrder();
  }

  /**
   * Get a comparator which sorts comparables in their natural order.
   * @param <T> comparable tyoe
   * @return natural order comparator
   */
  @SuppressWarnings("unchecked")
  @NotNull
  public static <T extends Comparable<T>> Comparator<T> naturalOrder()
  {
    return (Comparator<T>)COMPARABLE_COMPARATOR;
  }

  /**
   * A converter which converts a type to a more basic type.
   * Java can do this alone in most circumstances, so this should rarely be necessary.
   * @param <S> source type which extends the target type
   * @param <T> target type (base type of source type)
   * @return type converter converting S to T
   */
  @NotNull
  public static <T, S extends T> Function1<T, S> downConverter()
  {
    return object -> object;
  }

  /**
   * A converter which converts a type to a any other type by
   * applying a cast.
   * Use this converter only if you are perfectly sure that
   * all objects passed to this converter fulfill the promise
   * that they are of type T, If not usage of this converter
   * will result in a {@link ClassCastException}.
   * @param <S> source type
   * @param <T> target type
   * @return type converter converting S to T
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T, S> Function1<T, S> castConverter()
  {
    return object -> (T)object;
  }

  /**
   * Type converter which does no conversion.
   * Use this converter when there is a type converter expected, but not necessary.
   * @param <T> any type
   * @return type converter not doing any conversion
   */
  @NotNull
  public static <T> Function1<T, T> nullConverter()
  {
    return object -> object;
  }

  /**
   * Cast an item to a given type if possible.
   * @param type target type
   * @param obj  object
   * @param <T>  target type
   * @return obj cast to target type, or {@code null}
   */
  @Nullable
  @SuppressWarnings("unchecked")
  public static <T> T as(@NotNull Class<T> type, @Nullable Object obj)
  {
    if (obj == null) {
      return null;
    }
    if (type.isInstance(obj)) {
      return (T)obj;
    }
    return null;
  }

  /**
   * Type converter which converts a map entry by simultaneously applying a
   * converter to the key and to the value.
   * @param keyConverter    key converter
   * @param valueConverter  value converter
   * @param <KT>            target key type
   * @param <VT>            target value type
   * @param <KS>            source key type
   * @param <VS>            source value type
   * @return the converter
   */
  @NotNull
  public static <KT, VT, KS, VS>
  Function1<Map.Entry<KT, VT>,
          Map.Entry<KS, VS>>
  mapEntryConverter(Function<KS, KT> keyConverter,
                    Function<VS, VT> valueConverter)
  {
    return new MapKeyValueConverter<>(keyConverter, valueConverter);
  }

  /**
   * Get a combined type converter using an intermediate value.
   * @param secondStep second step, converting the intermediate vale to the target value
   * @param firstStep  first step, converting the source value to the intermediate value
   * @param <T> target type
   * @param <I> intermediate type
   * @param <S> source type
   * @return combined converter
   */
  public static <T, I, S> Function1<T, S> combine(@NotNull Function<I, T> secondStep,
                                                  @NotNull Function<S, ? extends I> firstStep)
  {
    return new TransitiveTypeConverter<>(secondStep, firstStep);
  }

  /**
   * Get a combined type converter using an intermediate value.
   * @param secondStep second step, converting the intermediate vale to the target value
   * @param firstStep  first step, converting the source value to the intermediate value
   * @param <T> target type
   * @param <I> intermediate type
   * @param <S> source type
   * @return combined converter
   */
  public static <T, I, S> FragileTypeConverter<T, S> combineEF(@NotNull FragileTypeConverter<T, I> secondStep,
                                                               @NotNull Function<S, ? extends I> firstStep)
  {
    return new TransitiveFragileTypeConverter<>(secondStep,
                                                toFragile(firstStep));
  }

  /**
   * Get a combined type converter using an intermediate value.
   * @param secondStep second step, converting the intermediate vale to the target value
   * @param firstStep  first step, converting the source value to the intermediate value
   * @param <T> target type
   * @param <I> intermediate type
   * @param <S> source type
   * @return combined converter
   */
  public static <T, I, S> FragileTypeConverter<T, S> combineFE(@NotNull Function<I, T> secondStep,
                                                               @NotNull FragileTypeConverter<? extends I, S> firstStep)
  {
    return new TransitiveFragileTypeConverter<>(toFragile(secondStep),
                                                firstStep);
  }

  /**
   * Get a combined type converter using an intermediate value.
   * @param secondStep second step, converting the intermediate vale to the target value
   * @param firstStep  first step, converting the source value to the intermediate value
   * @param <T> target type
   * @param <I> intermediate type
   * @param <S> source type
   * @return combined converter
   */
  public static <T, I, S> FragileTypeConverter<T, S> combineEE(@NotNull FragileTypeConverter<T, I> secondStep,
                                                               @NotNull FragileTypeConverter<? extends I, S> firstStep)
  {
    return new TransitiveFragileTypeConverter<>(secondStep, firstStep);
  }

  /**
   * Get an inverted two way type converter from a given one.
   * @param typeConverter given type converter
   * @param <T> target type
   * @param <S> source tyoe
   * @return inverse type converter
   */
  @NotNull
  public static <T,S> TwoWayTypeConverter<T,S> invert(@NotNull final TwoWayTypeConverter<S,T> typeConverter)
  {
    return new TwoWayTypeConverter<T, S>()
    {
      @Override
      public S convertBack(T object)
      {
        return typeConverter.apply(object);
      }

      @Override
      public T apply(S object)
      {
        return typeConverter.convertBack(object);
      }
    };
  }

  /**
   * Create a two way type converter from two one way type converters.
   * @param toConverter   converter for normal direction (S to T)
   * @param backConverter converter for back direction (T to S)
   * @param <T> target type
   * @param <S> source type
   * @return two way type converter
   */
  @NotNull
  public static <T,S> TwoWayTypeConverter<T,S> createTwoWay(@NotNull final Function<S,T> toConverter,
                                                            @NotNull final Function<T,S> backConverter)
  {
    return new TwoWayTypeConverter<T, S>()
    {
      @Override
      public S convertBack(T object)
      {
        return backConverter.apply(object);
      }

      @Override
      public T apply(S object)
      {
        return toConverter.apply(object);
      }
    };
  }

  /**
   * Convert a fragile zero-argument function which could throw an exception into non-fragile one.
   * @param fragileFunction fragile function
   * @param <R> result type of incoming and outgoing function
   * @return non-fragile function which will throw a {@link RuntimeException} with the
   *         wrapped original exception if {@code fragileFunction} throws its declared exception
   */
  @NotNull
  public static <R> Function0<R> toRobustF0(@NotNull FragileFunction0<? extends R, ?> fragileFunction)
  {
    return () -> {
      try {
        return fragileFunction.apply();
      } catch (Exception x) {
        throw new RuntimeException("Fragile function threw exception!", x);
      }
    };
  }

  /**
   * Convert a fragile one-argument function which could throw an exception into non-fragile one.
   * @param fragileFunction fragile function
   * @param <R> result type of incoming and outgoing function
   * @param <P> parameter type of incoming and outgoing function
   * @return non-fragile function which will throw a {@link RuntimeException} with the
   *         wrapped original exception if {@code fragileFunction} throws its declared exception
   */
  @NotNull
  public static <R, P> Function1<R, P> toRobustF1(@NotNull FragileFunction1<? extends R, ?, ? super P> fragileFunction)
  {
    return arg -> {
      try {
        return fragileFunction.apply(arg);
      } catch (Exception x) {
        throw new RuntimeException("Fragile function threw exception!", x);
      }
    };
  }

  /**
   * Convert a fragile two-argument function which could throw an exception into non-fragile one.
   * @param fragileFunction fragile function
   * @param <R> result type of incoming and outgoing function
   * @param <P1> first parameter type of incoming and outgoing function
   * @param <P2> second parameter type of incoming and outgoing function
   * @return non-fragile function which will throw a {@link RuntimeException} with the
   *         wrapped original exception if {@code fragileFunction} throws its declared exception
   */
  @NotNull
  public static <R, P1, P2> Function2<R, P1, P2> toRobustF2(@NotNull FragileFunction2<? extends R, ?, ? super P1, ? super P2> fragileFunction)
  {
    return (arg1, arg2) -> {
      try {
        return fragileFunction.apply(arg1, arg2);
      } catch (Exception x) {
        throw new RuntimeException("Fragile function threw exception!", x);
      }
    };
  }

  /**
   * Convert a fragile 3-argument function which could throw an exception into non-fragile one.
   * @param fragileFunction fragile function
   * @param <R> result type of incoming and outgoing function
   * @param <P1> first parameter type of incoming and outgoing function
   * @param <P2> second parameter type of incoming and outgoing function
   * @param <P3> third parameter type of incoming and outgoing function
   * @return non-fragile function which will throw a {@link RuntimeException} with the
   *         wrapped original exception if {@code fragileFunction} throws its declared exception
   */
  @NotNull
  public static <R, P1, P2, P3> Function3<R, P1, P2, P3> toRobustF3(@NotNull FragileFunction3<? extends R, ?, ? super P1, ? super P2, ? super P3> fragileFunction)
  {
    return (arg1, arg2, arg3) -> {
      try {
        return fragileFunction.apply(arg1, arg2, arg3);
      } catch (Exception x) {
        throw new RuntimeException("Fragile function threw exception!", x);
      }
    };
  }

  /**
   * Convert a fragile zero-argument procedure which could throw an exception into non-fragile one.
   * @param fragileProc fragile procedure
   * @return non-fragile procedure which will throw a {@link RuntimeException} with the
   *         wrapped original exception if {@code fragileProc} throws its declared exception
   */
  @NotNull
  public static Procedure0 toRobustP0(@NotNull FragileProcedure0<?> fragileProc)
  {
    return () -> {
      try {
        fragileProc.apply();
      } catch (Exception x) {
        throw new RuntimeException("Fragile function threw exception!", x);
      }
    };
  }

  /**
   * Convert a fragile one-argument procedure which could throw an exception into non-fragile one.
   * @param fragileProc fragile procedure
   * @param <P> parameter type of incoming and outgoing procedure
   * @return non-fragile function which will throw a {@link RuntimeException} with the
   *         wrapped original exception if {@code fragileProc} throws its declared exception
   */
  @NotNull
  public static <P> Procedure1<P> toRobustP1(@NotNull FragileProcedure1<?, ? super P> fragileProc)
  {
    return arg -> {
      try {
        fragileProc.apply(arg);
      } catch (Exception x) {
        throw new RuntimeException("Fragile function threw exception!", x);
      }
    };
  }

  /**
   * Convert a fragile two-argument procedure which could throw an exception into non-fragile one.
   * @param fragileProc fragile procedure
   * @param <P1> first parameter type of incoming and outgoing procedure
   * @param <P2> second parameter type of incoming and outgoing procedure
   * @return non-fragile function which will throw a {@link RuntimeException} with the
   *         wrapped original exception if {@code fragileProc} throws its declared exception
   */
  @NotNull
  public static <P1, P2> Procedure2<P1, P2> toRobustP2(@NotNull FragileProcedure2<?, ? super P1, ? super P2> fragileProc)
  {
    return (arg1, arg2) -> {
      try {
        fragileProc.apply(arg1, arg2);
      } catch (Exception x) {
        throw new RuntimeException("Fragile function threw exception!", x);
      }
    };
  }

  /**
   * Convert a fragile 3-argument procedure which could throw an exception into non-fragile one.
   * @param fragileProc fragile procedure
   * @param <P1> first parameter type of incoming and outgoing procedure
   * @param <P2> second parameter type of incoming and outgoing procedure
   * @param <P3> third parameter type of incoming and outgoing procedure
   * @return non-fragile function which will throw a {@link RuntimeException} with the
   *         wrapped original exception if {@code fragileProc} throws its declared exception
   */
  @NotNull
  public static <P1, P2, P3> Procedure3<P1, P2, P3> toRobustP3(@NotNull FragileProcedure3<?, ? super P1, ? super P2, ? super P3> fragileProc)
  {
    return (arg1, arg2, arg3) -> {
      try {
        fragileProc.apply(arg1, arg2, arg3);
      } catch (Exception x) {
        throw new RuntimeException("Fragile function threw exception!", x);
      }
    };
  }

  /**
   * Make a fragile type converter appear as a standard type converter.
   *
   * @param fragileTypeConverter fragile type converter
   * @param <T> target type
   * @param <S> source type
   * @return fragile type converter wrapper, returning {@code null} on exceptions
   */
  @NotNull
  public static <T, S> CarelessTypeConverter<T, S> toNonFragile(@NotNull FragileTypeConverter<T, S> fragileTypeConverter)
  {
    return toNonFragile(fragileTypeConverter, null);
  }

  /**
   * Make a fragile type converter appear as a standard type converter.
   *
   * @param fragileTypeConverter fragile type converter
   * @param defaultValue default value returned when the fragile type converter throws an exception
   * @param <T> target type
   * @param <S> source type
   * @return fragile type converter wrapper, returning the default value on exceptions
   */
  @NotNull
  public static <T, S> CarelessTypeConverter<T, S> toNonFragile(@NotNull FragileTypeConverter<T, S> fragileTypeConverter,
                                                                @Nullable T defaultValue)
  {
    return new CarelessTypeConverter<>(fragileTypeConverter, defaultValue);
  }

  /**
   * Make a non-fragile type converter appear as fragile.
   * @param typeConverter standard non-fragile type converter
   * @param <T> target type
   * @param <S> source type
   * @return overcaring type converter
   */
  @NotNull
  public static <T, S> OverCaringTypeConverter<T, S> toFragile(@NotNull Function<S, T> typeConverter)
  {
    return new OverCaringTypeConverter<>(typeConverter);
  }

  /**
   * Get a down-casted iterable.
   * <p>
   * This allows to view any iterable as being an iterable of a base class of its generic type.
   * <p>
   * Although casting generic types is frowned upon in general, in this special case using this method
   * does not harm static type safety. I.e. if your program compiles the code without warnings,
   * your program will not throw {@link ClassCastException}s as the result of using this method.
   * <p>
   * Example:
   * <blockquote><pre>
   *   public void drawShapes(Iterable&lt;Shape&gt; shapes) {
   *     // ...
   *   }
   *
   *   // ...
   *   Collection&lt;Rectangle2D&gt; rectangles = // ...
   *   drawShapes(Types.&lt;Shape&gt;downCast(rectangles));
   *
   * </pre></blockquote>
   * @param iterable iterable with extended type
   * @param <T> base type
   * @return iterator with base type
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T> Iterable<T> downCast(@NotNull Iterable<? extends T> iterable)
  {
    return (Iterable<T>)iterable;
  }

  /**
   * Get a down-casted iterator.
   * <p>
   * This allows to view any iterator as being an iterator of a base class of its generic type.
   * <p>
   * Although casting generic types is frowned upon in general, in this special case using of this method
   * does not harm static type safety. I.e. if your program compiles the code without warnings,
   * your program will not throw {@link ClassCastException}s as the result of using this method.
   * <p>
   * Example:
   * <blockquote><pre>
   *   public void drawShapes(Iterator&lt;Shape&gt; shapes) {
   *     // ...
   *   }
   *
   *   // ...
   *   Collection&lt;Rectangle2D&gt; rectangles = // ...
   *   drawShapes(Types.&lt;Shape&gt;downCast(rectangles.iterator());
   *
   * </pre></blockquote>
   * @param iterator iterator with extended type
   * @param <T> base type
   * @return iterable with base type
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T> Iterator<T> downCast(@NotNull Iterator<? extends T> iterator)
  {
    return (Iterator<T>)iterator;
  }

  /**
   * Get a down-casted enumeration.
   * <p>
   * This allows to view any enumeration as being an enumeration of a base class of its generic type.
   * <p>
   * Although casting generic types is frowned upon in general, in this special case usage of this method
   * does not harm static type safety. I.e. if your program compiles the code without warnings,
   * your program will not throw {@link ClassCastException}s as the result of using this method.
   * <p>
   * Example:
   * <blockquote><pre>
   *   public void drawShapes(Enumeration&lt;Shape&gt; shapes) {
   *     // ...
   *   }
   *
   *   // ...
   *   GVector&lt;Rectangle2D&gt; rectangles = // ...
   *   drawShapes(Types.&lt;Shape&gt;downCast(rectangles.enumeration());
   * </pre></blockquote>
   * @param enumeration enumeration with extended type
   * @param <T> base type
   * @return enumerator with base type
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T> Enumeration<T> downCast(@NotNull Enumeration<? extends T> enumeration)
  {
    return (Enumeration<T>)enumeration;
  }

  /**
   * Downcast a type converter.
   * <p>
   * Using this method is typesafe.
   * <p>
   * Although casting generic types is frowned upon in general, in this special case usage of this method
   * does not harm static type safety. I.e. if your program compiles the code without warnings,
   * your program will not throw {@link ClassCastException}s as the result of using this method.
   *
   * @param typeConverter type converter to downcast
   * @param <T> base type of original target type
   * @param <S> extending type of original source type
   * @return casted type converter
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T, S> Function<S, T> downCast(@NotNull Function<? super S, ? extends T> typeConverter)
  {
    return (Function<S, T>)typeConverter;
  }

  /**
   * Downcast a type converter.
   * <p>
   * Using this method is typesafe.
   * @param typeConverter type converter to downcast
   * @param <T> base type of original target type
   * @param <S> extending type of original source type
   * @return casted type converter
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T, S> FragileTypeConverter<T, S> downCastT(@NotNull FragileTypeConverter<? extends T, ? super S> typeConverter)
  {
    return (FragileTypeConverter<T, S>)typeConverter;
  }

  /**
   * Get a casted iterable.
   * <p>
   * Be careful using this method, as it can easily harm static type safety.
   * I.e. only use when it is guaranteed that all elements returned by the iteration
   * fulfill the promise that they are of type T. Otherwise a {@link ClassCastException}
   * will be thrown when using the iterator if the iterable.
   * <p>
   * In other words: you can use this method to compile a program which will fail at runtime!
   * And that is usually a stupid idea. So you should rethink your design before using this method.
   * <p>
   * The reason for this method is that life isn't perfect, and especially when you have to use
   * libraries (even standard Java ones) this method may prove helpful.
   * <p>
   * Example:
   * <pre>
   *   // we have this naked collection, but we know all its elements are Strings
   *   Collection foo = otherLibObject.getCollection();
   *   for (String str : Types.&lt;String&gt;cast(foo)) {
   *     // ...
   *   }
   * </pre>
   *
   * @param iterable iterable with any tye
   * @param <T> target type
   * @return iterator with target type
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T> Iterable<T> cast(@NotNull Iterable<?> iterable)
  {
    return (Iterable<T>)iterable;
  }

  /**
   * Get a casted iterator.
   * <p>
   * Be careful using this method, as it can easily harm static type safety.
   * I.e. only use when it is guaranteed that all elements returned by the iterator
   * fulfill the promise that they are of type T. Otherwise a {@link ClassCastException}
   * will be thrown when using the iterator.
   * <p>
   * In other words: you can use this method to compile a program which will fail at runtime!
   * And that is usually a stupid idea. So you should rethink your design before using this method.
   * <p>
   * The reason for this method is that life isn't perfect, and especially when you have to use
   * libraries (even standard Java ones) this method may prove helpful.
   * <p>
   * See {@link de.caff.generics.Types#cast(java.lang.Iterable)} for an easy to adapt example.
   *
   * @param iterator iterator with any type
   * @param <T> target type
   * @return iterator with target type
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T> Iterator<T> cast(@NotNull Iterator<?> iterator)
  {
    return (Iterator<T>)iterator;
  }

  /**
   * Get a casted enumeration.
   * <p>
   * Be careful using this method, as it can easily harm static type safety.
   * I.e. only use when it is guaranteed that all elements returned by the enumeration
   * fulfill the promise that they are of type T. Otherwise a {@link ClassCastException}
   * will be thrown when using the enumeration.
   * <p>
   * In other words: you can use this method to compile a program which will fail at runtime!
   * And that is a stupid idea. So you should rethink your design before using this method.
   * <p>
   * The reason for this method is that life isn't perfect, and especially when you have to use
   * libraries (even standard Java ones) this method may prove helpful.
   * <p>
   * See {@link de.caff.generics.Types#cast(java.lang.Iterable)} for an easy to adapt example.
   *
   * @param enumeration enumeration with any type
   * @param <T> target type
   * @return enumerator with target type
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T> Enumeration<T> cast(@NotNull Enumeration<? extends T> enumeration)
  {
    return (Enumeration<T>)enumeration;
  }

  /**
   * Flatten a collection of collections.
   * This will create a single linked list which contains all elements contained in the incoming
   * collections.
   * @param collectionOfCollections collection to flatten
   * @param <T> basic type
   * @return flattened linked list with all elements
   */
  @NotNull
  public static <T> LinkedList<T> flatten(@NotNull Iterable<? extends Iterable<T>> collectionOfCollections)
  {
    return mapX(collectionOfCollections, Types.<Iterable<T>>nullConverter());
  }

  /**
   * Flatten a collection of collections.
   * This will create a single collection which contains all elements contained in the incoming
   * collections.
   * @param target target collection, where the elements are stored
   * @param collectionOfCollections collection to flatten
   * @param <T> basic type
   * @param <R> result type
   * @return flattened linked list with all elements
   */
  @NotNull
  public static <T, R extends Collection<? super T>> R
  flatten(@NotNull R target, @NotNull Iterable<? extends Iterable<T>> collectionOfCollections)
  {
    return mapX(target, collectionOfCollections, Types.<Iterable<T>>nullConverter());
  }

  /**
   * Add the mapped elements of an array into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link #map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Example:
   * <blockquote><pre>
   *    // convert an array of strings containing double values into a collection of doubles
   *    String[] values = { "1.2", "-7.07", "3.14250255358979" };
   *    List&lt;Double&gt; result = Types.map(new ArrayList&lt;Double&gt;(values.length),
   *                                    values,
   *                                    new Function1&lt;Double, String&gt;() {
   *                                      public Double convert(String object) {
   *                                        return Double.parseDouble(object);
   *                                      }
   *                                    });
   * </pre></blockquote>
   *
   * @param target         collection where the mapped elements are added
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <C extends Collection<? super T>, T, S> C map(@NotNull C target,
                                                              @NotNull S[] array,
                                                              @NotNull Function<? super S, T> typeConverter)
  {
    return map(target, asList(array), typeConverter);
  }

  /**
   * Add the mapped elements of an array into a map.
   * <p>
   * This will add all mapped elements to the target collection, using the
   * converted values as keys pointing to the incoming values.
   *
   * @param target         collection where the mapped elements are added
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <M extends Map<? super T, ? super S>, T, S> M mapM(@NotNull M target,
                                                                   @NotNull S[] array,
                                                                   @NotNull Function<? super S, T> typeConverter)
  {
    return mapM(target, asList(array), typeConverter);
  }

  /**
   * Add the mapped elements of an array into a map.
   * <p>
   * This will add all mapped elements to the target collection, using the
   * incoming values as keys pointing to the converted values.
   *
   * @param target         collection where the mapped elements are added
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <M extends Map<? super S, ? super T>, T, S> M mapR(@NotNull M target,
                                                                   @NotNull S[] array,
                                                                   @NotNull Function<? super S, T> typeConverter)
  {
    return mapR(target, asList(array), typeConverter);
  }

  /**
   * Add the mapped elements of an iteration into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link #map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that target and iterable are not allowed to be the same object, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   * <p>
   * Example:
   * <blockquote><pre>
   *    // convert an array of strings containing double values into a collection of doubles
   *    String[] values = { "1.2", "-7.07", "3.14250255358979" };
   *    List&lt;Double&gt; result = Types.map(new ArrayList&lt;Double&gt;(values.length),
   *                                    Types.asList(values),
   *                                    new Function1&lt;Double, String&gt;() {
   *                                      public Double convert(String object) {
   *                                        return Double.parseDouble(object);
   *                                      }
   *                                    });
   * </pre></blockquote>
   *
   * @param target         collection where the mapped elements are added
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <C extends Collection<? super T>, T, S> C map(@NotNull C target,
                                                              @NotNull Iterable<S> iterable,
                                                              @NotNull Function<? super S, T> typeConverter)
  {
    if (target == iterable) {
      throw new IllegalArgumentException("'target' and 'iterable' may not be the same object!");
    }
    return mapImpl(target, iterable.iterator(), typeConverter);
  }

  /**
   * Add the mapped elements of an iteration into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link #map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that target and iterable are not allowed to be the same object, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   * <p>
   * Example:
   * <blockquote><pre>
   *    // convert an array of strings containing double values into a collection of doubles
   *    String[] values = { "1.2", "-7.07", "3.14250255358979" };
   *    List&lt;Double&gt; result = Types.map(new ArrayList&lt;Double&gt;(values.length),
   *                                    Types.asList(values),
   *                                    new Function1&lt;Double, String&gt;() {
   *                                      public Double convert(String object) {
   *                                        return Double.parseDouble(object);
   *                                      }
   *                                    });
   * </pre></blockquote>
   *
   * @param target         collection where the mapped elements are added
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <C extends Consumer<? super T>, T, S> C map(@NotNull C target,
                                                            @NotNull Iterable<S> iterable,
                                                            @NotNull Function<? super S, T> typeConverter)
  {
    return mapImpl(target, iterable.iterator(), typeConverter);
  }

  /**
   * Add the mapped elements of an iteration into a map.
   * <p>
   * This will add all mapped elements to the target collection, using the
   * converted values as keys pointing to the incoming values.
   *
   * @param target         collection where the mapped elements are added
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <M extends Map<? super T, ? super S>, T, S> M mapM(@NotNull M target,
                                                                   @NotNull Iterable<S> iterable,
                                                                   @NotNull Function<? super S, T> typeConverter)
  {
    if (target == iterable) {
      throw new IllegalArgumentException("'target' and 'iterable' may not be the same object!");
    }
    return mapMImpl(target, iterable.iterator(), typeConverter);
  }

  /**
   * Add the mapped elements of an iteration into a map.
   * <p>
   * This will add all mapped elements to the target collection, using the
   * converted values as keys pointing to the incoming values.
   *
   * @param target         collection where the mapped elements are added
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <M extends Map<? super S, ? super T>, T, S> M mapR(@NotNull M target,
                                                                   @NotNull Iterable<S> iterable,
                                                                   @NotNull Function<? super S, T> typeConverter)
  {
    if (target == iterable) {
      throw new IllegalArgumentException("'target' and 'iterable' may not be the same object!");
    }
    return mapRImpl(target, iterable.iterator(), typeConverter);
  }

  /**
   * Add the mapped elements of an iterator into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link #map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   * <p>
   * See {@link #map(java.util.Collection, Iterable, Function)} for an easy to adapt example.
   *
   * @param target         collection where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <C extends Collection<? super T>, T, S> C map(@NotNull C target,
                                                              @NotNull Iterator<S> iterator,
                                                              @NotNull Function<? super S, T> typeConverter)
  {
    return mapImpl(target, iterator, typeConverter);
  }

  /**
   * Add the mapped elements of an iterator into a map.
   * <p>
   * This will add all mapped elements to the target collection, using the
   * converted values as keys pointing to the incoming values.
   *
   * @param target         collection where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <M extends Map<? super T, ? super S>, T, S> M mapM(@NotNull M target,
                                                                   @NotNull Iterator<S> iterator,
                                                                   @NotNull Function<? super S, T> typeConverter)
  {
    return mapMImpl(target, iterator, typeConverter);
  }

  /**
   * Add the mapped elements of an iterator into a map.
   * <p>
   * This will add all mapped elements to the target collection, using the
   * incoming values as keys pointing to the converted values.
   *
   * @param target         collection where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <M extends Map<? super S, ? super T>, T, S> M mapR(@NotNull M target,
                                                                   @NotNull Iterator<S> iterator,
                                                                   @NotNull Function<? super S, T> typeConverter)
  {
    return mapRImpl(target, iterator, typeConverter);
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link #map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   * <p>
   * See {@link #map(java.util.Collection, Iterable, Function)} for an easy to adapt example.
   *
   * @param target         collection where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <C extends Collection<? super T>, T, S> C map(@NotNull C target,
                                                              @NotNull Enumeration<S> enumeration,
                                                              @NotNull Function<? super S, T> typeConverter)
  {
    return mapImpl(target, enumeration, typeConverter);
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   * <p>
   * This will add all mapped elements to the target collection, using the
   * converted values as keys pointing to the incoming values.
   *
   * @param target         collection where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <M extends Map<? super T, ? super S>, T, S> M mapM(@NotNull M target,
                                                                   @NotNull Enumeration<S> enumeration,
                                                                   @NotNull Function<? super S, T> typeConverter)
  {
    return mapMImpl(target, enumeration, typeConverter);
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   * <p>
   * This will add all mapped elements to the target collection, using the
   * converted values as keys pointing to the incoming values.
   *
   * @param target         collection where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <M extends Map<? super S, ? super T>, T, S> M mapR(@NotNull M target,
                                                                   @NotNull Enumeration<S> enumeration,
                                                                   @NotNull Function<? super S, T> typeConverter)
  {
    return mapRImpl(target, enumeration, typeConverter);
  }

  /**
   * Map one map into another, while allowing to discard elements on the way.
   * <p>
   * An element is discarded if the type converter returns {@code null}.
   *
   * @param source         source map
   * @param typeConverter  type converter converting map entries
   * @param <KT>           target map key type
   * @param <VT>           target map value type
   * @param <KS>           source map key type
   * @param <VS>           source map value type
   * @return target map
   */
  @NotNull
  public static <KT, VT, KS, VS> HashMap<KT, VT>
  map(@NotNull Map<KS,VS> source,
      @NotNull Function<Map.Entry<KS, VS>, Map.Entry<KT, VT>> typeConverter)
  {
    return map(new HashMap<>(),
               source,
               typeConverter);
  }

  /**
   * Map one map into another, while allowing to discard elements on the way.
   * <p>
   * An element is discarded if the type converter returns {@code null}.
   *
   * @param target         target map
   * @param source         source map
   * @param typeConverter  type converter converting map entries
   * @param <MT>           target map type
   * @param <KT>           target map key type
   * @param <VT>           target map value type
   * @param <KS>           source map key type
   * @param <VS>           source map value type
   * @return target map
   */
  @NotNull
  public static <MT extends Map<KT, VT>, KT, VT,
          KS, VS> MT
  map(@NotNull MT target,
      @NotNull Map<KS, VS> source,
      @NotNull Function<Map.Entry<KS, VS>, ? extends Map.Entry<KT, VT>> typeConverter)
  {
    return mapImpl(target, source, typeConverter);
  }

  /**
   * Map one map into another, while allowing to discard elements on the way.
   * <p>
   * An element is discarded if the type converter returns {@code null}.
   *
   * @param source         source map
   * @param typeConverter  type converter converting map entries
   * @param <KT>           target map key type
   * @param <VT>           target map value type
   * @param <KS>           source map key type
   * @param <VS>           source map value type
   * @return target map
   */
  @NotNull
  public static <KT, VT, KS, VS> HashMap<KT, VT>
  mapX(@NotNull Map<KS,VS> source,
       @NotNull Function<Map.Entry<? super KS, ? super VS>, Iterable<Map.Entry<KT, VT>>> typeConverter)
  {
    return mapX(new HashMap<KT, VT>(),
                source,
                typeConverter);
  }

  /**
   * Map one map into another, allowing to discard or add elements on the way.
   *
   * @param target         target map
   * @param source         source map
   * @param typeConverter  type converter converting map entries
   * @param <MT>           target map type
   * @param <KT>           target map key type
   * @param <VT>           target map value type
   * @param <KS>           source map key type
   * @param <VS>           source map value type
   * @return target map
   */
  @NotNull
  public static <MT extends Map<? super KT, ? super VT>, KT, VT,
          KS, VS> MT
  mapX(@NotNull MT target,
       @NotNull Map<KS, VS> source,
       @NotNull Function<Map.Entry<? super KS, ? super VS>, ? extends Iterable<? extends Map.Entry<? extends KT, ? extends VT>>> typeConverter)
  {
    return mapXImpl(target, source, typeConverter);
  }

  /**
   * Filter the elements of an array into a collection.
   * <p>
   * This will add all filtered elements to the target collection.
   *
   * @param target         collection where the mapped elements are added
   * @param array          array providing the source elements
   * @param filter         filter to decide which elements are kept,
   *                       when returning {@code true} the element is kept,
   *                       otherwise it is discarded
   * @param <C>            collection type
   * @param <T>            filtered type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <C extends Collection<? super T>, T> C filter(@NotNull C target,
                                                              @NotNull T[] array,
                                                              @NotNull Predicate<? super T> filter)
  {
    return filter(target, asList(array), filter);
  }

  /**
   * Filter the elements of an iterable into a collection.
   * <p>
   * This will add all filtered elements to the target collection.
   * <p>
   * Note that target and iterable are not allowed to be the same object, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         collection where the mapped elements are added
   * @param iterable       iterable providing the source elements
   * @param checker        value checker to decide which elements are kept
   * @param <C>            collection type
   * @param <T>            filtered type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <C extends Collection<? super T>, T> C filter(@NotNull C target,
                                                              @NotNull Iterable<T> iterable,
                                                              @NotNull Predicate<? super T> checker)
  {
    return filter(target, iterable.iterator(), checker);
  }

  /**
   * Filter the elements of an iterator into a collection.
   * <p>
   * This will add all filtered elements to the target collection.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         collection where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param checker        value checker to decide which elements are kept
   * @param <C>            collection type
   * @param <T>            filtered type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <C extends Collection<? super T>, T> C filter(@NotNull C target,
                                                              @NotNull Iterator<T> iterator,
                                                              @NotNull Predicate<? super T> checker)
  {
    return filterImpl(target, iterator, checker);
  }

  /**
   * Filter the elements of an enumeration into a collection.
   * <p>
   * This will add all filtered elements to the target collection.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         collection where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param checker        value checker to decide which elements are kept
   * @param <C>            collection type
   * @param <T>            filtered type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <C extends Collection<? super T>, T> C filter(@NotNull C target,
                                                              @NotNull Enumeration<T> enumeration,
                                                              @NotNull Predicate<? super T> checker)
  {
    return filterImpl(target, enumeration, checker);
  }

  /**
   * Filter the elements of an array into a collection.
   * <p>
   * This will add all filtered elements to the target collection.
   *
   * @param array          array providing the source elements
   * @param checker        value checker to decide which elements are kept
   * @param <T>            filtered type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <T> LinkedList<T> filter(@NotNull T[] array,
                                         @NotNull Predicate<? super T> checker)
  {
    return filter(new LinkedList<T>(), array, checker);
  }

  /**
   * Filter the elements of an iterable into a collection.
   * <p>
   * This will add all filtered elements to the target collection.
   *
   * @param iterable       iterable providing the source elements
   * @param checker        value checker to decide which elements are kept
   * @param <T>            filtered type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <T> LinkedList<T> filter(@NotNull Iterable<T> iterable,
                                         @NotNull Predicate<? super T> checker)
  {
    return filter(new LinkedList<T>(), iterable, checker);
  }

  /**
   * Filter the elements of an iterator into a collection.
   * <p>
   * This will add all filtered elements to the target collection.
   *
   * @param iterator       iterator providing the source elements
   * @param checker        value checker to decide which elements are kept
   * @param <T>            filtered type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <T> LinkedList<T> filter(@NotNull Iterator<T> iterator,
                                         @NotNull Predicate<? super T> checker)
  {
    return filter(new LinkedList<T>(), iterator, checker);
  }

  /**
   * Filter the elements of an enumeration into a result list.
   *
   * @param enumeration    enumeration providing the source elements
   * @param checker        value checker to decide which elements are kept
   * @param <T>            filtered type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <T> LinkedList<T> filter(@NotNull Enumeration<T> enumeration,
                                         @NotNull Predicate<? super T> checker)
  {
    return filter(new LinkedList<T>(), enumeration, checker);
  }

  /**
   * Filter a modifiable iterable inline.
   * This removes elements with certain properties from the iterable.
   * @param iterable modifiable iterable expected to provide an iterator which implements
   *                 the {@link Iterator#remove()}  method
   * @param outCheck   predicate which returns {@code true} for elements which are expected to be removed
   * @param <T> collection element type
   * @return {@code true} if the collection was mdified<br>
   *         {@code false} if nothing was changed
   * @throws RuntimeException whatever the iterator of the iterable throws when calling its remove() method
   */
  public static <T> boolean filterOut(@NotNull Iterable<T> iterable,
                                      @NotNull Predicate<? super T> outCheck)
  {
    boolean changed = false;
    for (Iterator<T> it = iterable.iterator();  it.hasNext(); ) {
      if (outCheck.test(it.next())) {
        it.remove();
        changed = true;
      }
    }
    return changed;
  }

  /**
   * Does any element in the array fulfill the predicate?
   * <p>
   * This will check all array elements against the predicate, and return
   * {@code true} on the first positive check.
   *
   * @param array          array providing the source elements
   * @param checker        value checker
   * @param <T>            checked type
   * @return {@code true}: if any of the elements of the array matches the predicate<br>
   *         {@code false}: if none of the elements matches or the array is empty
   */
  public static <T> boolean any(@NotNull T[] array,
                                @NotNull Predicate<? super T> checker)
  {
    for (T elem : array) {
      if (checker.test(elem)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Does any element in the iterable fulfill the predicate?
   * <p>
   * This will check all elements against the predicate, and return
   * {@code true} on the first positive check.
   *
   * @param iterable       iterable providing the source elements
   * @param checker        value checker
   * @param <T>            filtered type
   * @return {@code true}: if any of the elements of the array matches the predicate<br>
   *         {@code false}: if none of the elements matches or the iterable is empty
   */
  public static <T> boolean any(@NotNull Iterable<T> iterable,
                                @NotNull Predicate<? super T> checker)
  {
    for (T elem: iterable) {
      if (checker.test(elem)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Does any element of the iterator fulfill the predicate?
   * <p>
   * This will check all elements against the predicate, and return
   * {@code true} on the first positive check.
   *
   * @param iterator       iterator providing the source elements
   * @param checker        value checker to decide which elements are kept
   * @param <T>            filtered type
   * @return {@code true}: if any of the elements of the array matches the predicate<br>
   *         {@code false}: if none of the elements matches or the iterator is empty
   */
  public static <T> boolean any(@NotNull Iterator<T> iterator,
                                @NotNull Predicate<? super T> checker)
  {
    while (iterator.hasNext()) {
      if (checker.test(iterator.next())) {
        return true;
      }
    }
    return false;
  }

  /**
   * Does any element of the enumeration fulfill the predicate?
   * <p>
   * This will check all elements against the predicate, and return
   * {@code true} on the first positive check.
   *
   * @param enumeration    enumeration providing the source elements
   * @param checker        value checker to decide which elements are kept
   * @param <T>            filtered type
   * @return {@code true}: if any of the elements of the array matches the predicate<br>
   *         {@code false}: if none of the elements matches or the enumeration is empty
   */
  public static <T> boolean any(@NotNull Enumeration<T> enumeration,
                                @NotNull Predicate<? super T> checker)
  {
    while (enumeration.hasMoreElements()) {
      if (checker.test(enumeration.nextElement())) {
        return true;
      }
    }
    return false;
  }

  /**
   * Does every element in the array fulfill the predicate?
   * <p>
   * This will check all array elements against the predicate, and return
   * {@code false} on the first negative check.
   *
   * @param array          array providing the source elements
   * @param checker        value checker
   * @param <T>            checked type
   * @return {@code true}: if all of the elements of the array match the predicate or the array is empty<br>
   *         {@code false}: if any of the elements does not match
   */
  public static <T> boolean all(@NotNull T[] array,
                                @NotNull Predicate<? super T> checker)
  {
    for (T elem : array) {
      if (!checker.test(elem)) {
        return false;
      }
    }
    return true;
  }

  /**
   * Does every element in the iterable fulfill the predicate?
   * <p>
   * This will check all elements against the predicate, and return
   * {@code false} on the first negative check.
   *
   * @param iterable       iterable providing the source elements
   * @param checker        value checker
   * @param <T>            filtered type
   * @return {@code true}: if all of the elements of the iterable match the predicate or the iterable is empty<br>
   *         {@code false}: if any of the elements does not match
   */
  public static <T> boolean all(@NotNull Iterable<T> iterable,
                                @NotNull Predicate<? super T> checker)
  {
    for (T elem: iterable) {
      if (!checker.test(elem)) {
        return false;
      }
    }
    return true;
  }

  /**
   * Does every element of the iterator fulfill the predicate?
   * <p>
   * This will check all elements against the predicate, and return
   * {@code false} on the first negative check.
   *
   * @param iterator       iterator providing the source elements
   * @param checker        value checker to decide which elements are kept
   * @param <T>            filtered type
   * @return {@code true}: if all of the elements of the iterator match the predicate or the iterator is empty<br>
   *         {@code false}: if any of the elements does not match
   */
  public static <T> boolean all(@NotNull Iterator<T> iterator,
                                @NotNull Predicate<? super T> checker)
  {
    while (iterator.hasNext()) {
      if (!checker.test(iterator.next())) {
        return false;
      }
    }
    return true;
  }

  /**
   * Does every element of the enumeration fulfill the predicate?
   * <p>
   * This will check all elements against the predicate, and return
   * {@code false} on the first negative check.
   *
   * @param enumeration    enumeration providing the source elements
   * @param checker        value checker to decide which elements are kept
   * @param <T>            filtered type
   * @return {@code true}: if all of the elements of the enumeration match the predicate or the enumeration is empty<br>
   *         {@code false}: if any of the elements does not match
   */
  public static <T> boolean all(@NotNull Enumeration<T> enumeration,
                                @NotNull Predicate<? super T> checker)
  {
    while (enumeration.hasMoreElements()) {
      if (!checker.test(enumeration.nextElement())) {
        return false;
      }
    }
    return true;
  }

  /**
   * Get the first element which fulfills a check.
   * @param iterable iterable
   * @param checker  value checker to decide which elements fit
   * @param <T> element type
   * @return first matching element of {@code null}
   */
  @Nullable
  public static <T> T first(@NotNull Iterable<T> iterable,
                            @NotNull Predicate<? super T> checker)
  {
    for (T elem : iterable) {
      if (checker.test(elem)) {
        return elem;
      }
    }
    return null;
  }

  /**
   * Get the last element which fulfills a check.
   * @param iterable iterable
   * @param checker  value checker to decide which elements fit
   * @param <T> element type
   * @return last matching element of {@code null}
   */
  @Nullable
  public static <T> T last(@NotNull Iterable<T> iterable,
                           @NotNull Predicate<? super T> checker)
  {
    T match = null;
    for (T elem : iterable) {
      if (checker.test(elem)) {
        match = elem;
      }
    }
    return match;
  }

  /**
   * Get the last element which fulfills a check.
   * @param list list to check
   * @param checker  value checker to decide which elements fit
   * @param startIndex start index of search
   * @param <T> element type
   * @return last matching element of {@code null}
   */
  @Nullable
  public static <T> T last(@NotNull List<T> list,
                           @NotNull Predicate<? super T> checker,
                           int startIndex)
  {
    for (ListIterator<T> it = list.listIterator(startIndex);  it.hasPrevious();  ) {
      T elem = it.previous();
      if (checker.test(elem)) {
        return elem;
      }
    }
    return null;
  }

  /**
   * Do left-associative folding of an array.
   * <p>
   * Folding is the reduction of the elements of a collection into one value,
   * by applying the same function on each element with the result of the
   * last call.
   * <p>
   * Eg to calculate the integer sum of an array of Numbers, you can use the following code
   * <blockquote><pre>
   *  Number[] array = { 1, Math.PI, 9L };
   *  double result = Types.foldLeft(array, 0.0,
   *                                new Function2&lt;Double, Double, Number&gt;() {
   *                                   public Double apply(Double arg1, Number arg2) {
   *                                     return arg1 + arg2.doubleValue();
   *                                   }
   *                                });
   * </pre></blockquote>
   * {@code result} will have a value of 13.14... after the call.
   * @param array         array which elements are folded together
   * @param initialValue  start value
   * @param folder        folding function, the apply method gets the accumulated value
   *                      as its first and the element as its second argument,
   *                      and returns the result of their combination
   * @param <V>           accumulated value type
   * @param <T>           array type
   * @return accumulated value
   */
  public static <V, T> V foldLeft(@NotNull T[] array,
                                  V initialValue,
                                  @NotNull BiFunction<? super V, ? super T, ? extends V> folder)
  {
    return foldLeft(asList(array), initialValue, folder);
  }

  /**
   * Do left-associative folding of an iterable.
   * <p>
   * Folding is the reduction of the elements of a collection into one value,
   * by applying the same function on each element with the result of the
   * last call.
   * <p>
   * Eg to calculate the integer sum of an array of Numbers, you can use the following code
   * <blockquote><pre>
   *  Collection&lt;Number&gt; numbers = Types.&lt;Number&gt;asList(1, Math.PI, 9L);
   *  double result = Types.foldLeft(numbers, 0.0,
   *                                new Function2&lt;Double, Double, Number&gt;() {
   *                                   public Double apply(Double arg1, Number arg2) {
   *                                     return arg1 + arg2.doubleValue();
   *                                   }
   *                                });
   * </pre></blockquote>
   * {@code result} will have a value of 13.14... after the call.
   * @param iterable      iterable which elements are folded together
   * @param initialValue  start value
   * @param folder        folding function, the apply method gets the accumulated value
   *                      as its first and the element as its second argument,
   *                      and returns the result of their combination
   * @param <V>           accumulated value type
   * @param <T>           iterable type
   * @return accumulated value
   */
  public static <V, T> V foldLeft(@NotNull Iterable<T> iterable,
                                  V initialValue,
                                  @NotNull BiFunction<? super V, ? super T, ? extends V> folder)
  {
    return foldLeft(iterable.iterator(), initialValue, folder);
  }

  /**
   * Do left-associative folding of an iterator.
   * <p>
   * Folding is the reduction of the elements of a collection into one value,
   * by applying the same function on each element with the result of the
   * last call.
   * <p>
   * See {@link #foldLeft(Iterable, Object, BiFunction)} for a code example.
   * @param iterator      iterator which elements are folded together
   * @param initialValue  start value
   * @param folder        folding function, the apply method gets the accumulated value
   *                      as its first and the element as its second argument,
   *                      and returns the result of their combination
   * @param <V>           accumulated value type
   * @param <T>           iterable type
   * @return accumulated value
   */
  public static <V, T> V foldLeft(@NotNull Iterator<T> iterator,
                                  V initialValue,
                                  @NotNull BiFunction<? super V, ? super T, ? extends V> folder)
  {
    return foldLeftImpl(iterator, initialValue, folder);
  }

  /**
   * Do left-associative folding of an enumeration.
   * <p>
   * Folding is the reduction of the elements of a collection into one value,
   * by applying the same function on each element with the result of the
   * last call.
   * <p>
   * See {@link #foldLeft(Iterable, Object, BiFunction)} for a code example.
   * @param enumeration   enumeration which elements are folded together
   * @param initialValue  start value
   * @param folder        folding function, the apply method gets the accumulated value
   *                      as its first and the element as its second argument,
   *                      and returns the result of their combination
   * @param <V>           accumulated value type
   * @param <T>           iterable type
   * @return accumulated value
   */
  public static <V, T> V foldLeft(@NotNull Enumeration<T> enumeration,
                                  V initialValue,
                                  @NotNull BiFunction<? super V, ? super T, ? extends V> folder)
  {
    return foldLeftImpl(enumeration, initialValue, folder);
  }

  /**
   * Do right-associative folding of an array.
   * <p>
   * Folding is the reduction of the elements of a collection into one value,
   * by applying the same function on each element with the result of the
   * last call.
   * <p>
   * Currently I have no useful example where right-folding in Java is making sense,
   * so you should prefer
   * {@link #foldLeft(Object[], Object, BiFunction)}.
   *
   * @param array         array which elements are folded together
   * @param initialValue  start value
   * @param folder        folding function, the apply method gets the accumulated value
   *                      as its first and the element as its second argument,
   *                      and returns the result of their combination
   * @param <V>           accumulated value type
   * @param <T>           array type
   * @return accumulated value
   */
  public static <V, T> V foldRight(@NotNull T[] array,
                                   V initialValue,
                                   @NotNull BiFunction<? super V, ? super T, ? extends V> folder)
  {
    return foldRight(asList(array), initialValue, folder);
  }

  /**
   * Do right-associative folding of an iterable.
   * <p>
   * Folding is the reduction of the elements of a collection into one value,
   * by applying the same function on each element with the result of the
   * last call.
   * <p>
   * Currently I have no useful example where right-folding in Java is making sense,
   * so you should prefer
   * {@link #foldLeft(Object[], Object, BiFunction)},
   * especially as the implementation for iterables is requiring an intermediate copy.
   * If you really find a use for right-folding try to use the
   * {@link #foldRight(java.util.List, Object, BiFunction)
   * foldRight method working on a list}, as it avoids the copy mentioned above.
   *
   * @param iterable      iterable which elements are folded together
   * @param initialValue  start value
   * @param folder        folding function, the apply method gets the accumulated value
   *                      as its first and the element as its second argument,
   *                      and returns the result of their combination
   * @param <V>           accumulated value type
   * @param <T>           iterable type
   * @return accumulated value
   */
  public static <V, T> V foldRight(@NotNull Iterable<T> iterable,
                                   V initialValue,
                                   @NotNull BiFunction<? super V, ? super T, ? extends V> folder)
  {
    return foldRight(iterable.iterator(), initialValue, folder);
  }

  /**
   * Do right-associative folding of an iteration.
   * <p>
   * Folding is the reduction of the elements of a collection into one value,
   * by applying the same function on each element with the result of the
   * last call.
   * <p>
   * Currently I have no useful example where right-folding in Java is making sense,
   * so you should prefer
   * {@link #foldLeft(Iterable, Object, BiFunction)},
   * especially as the implementation for iteration is requiring an intermediate copy.
   * If you really find a use for right-folding you should try to use the
   * {@link #foldRight(java.util.List, Object, BiFunction)
   * foldRight method working on a list}, as it avoids the copy mentioned above.
   *
   * @param iterator      iterator which elements are folded together
   * @param initialValue  start value
   * @param folder        folding function, the apply method gets the accumulated value
   *                      as its first and the element as its second argument,
   *                      and returns the result of their combination
   * @param <V>           accumulated value type
   * @param <T>           iterator type
   * @return accumulated value
   */
  public static <V, T> V foldRight(@NotNull Iterator<T> iterator,
                                   V initialValue,
                                   @NotNull BiFunction<? super V, ? super T, ? extends V> folder)
  {
    // to avoid overflowing the stack, copy into list first
    return foldRightImpl(Types.makeList(iterator), initialValue, folder);
  }

  /**
   * Do right-associative folding of an enumeration.
   * <p>
   * Folding is the reduction of the elements of a collection into one value,
   * by applying the same function on each element with the result of the
   * last call.
   * <p>
   * Currently I have no useful example where right-folding in Java is making sense,
   * so you should prefer
   * {@link #foldLeft(java.util.Iterator, Object, BiFunction)},
   * especially as the implementation for iteration is requiring an intermediate copy.
   * If you really find a use for right-folding you should try to use the
   * {@link #foldRight(java.util.List, Object, BiFunction)
   * foldRight method working on a list}, as it avoids the copy mentioned above.
   *
   * @param enumeration   enumeration which elements are folded together
   * @param initialValue  start value
   * @param folder        folding function, the apply method gets the accumulated value
   *                      as its first and the element as its second argument,
   *                      and returns the result of their combination
   * @param <V>           accumulated value type
   * @param <T>           enumeration type
   * @return accumulated value
   */
  public static <V, T> V foldRight(@NotNull Enumeration<T> enumeration,
                                   V initialValue,
                                   @NotNull BiFunction<? super V, ? super T, ? extends V> folder)
  {
    // to avoid overflowing the stack, copy into list first
    return foldRightImpl(Types.makeList(enumeration), initialValue, folder);
  }

  /**
   * Do right-associative folding of a list.
   * <p>
   * Folding is the reduction of the elements of a collection into one value,
   * by applying the same function on each element with the result of the
   * last call.
   * <p>
   * Currently I have no useful example where right-folding in Java is making sense,
   * so you should prefer
   * {@link #foldLeft(java.util.Enumeration, Object, BiFunction)},
   * especially as the implementation for iteration is requiring an intermediate copy.
   *
   * @param list          list which elements are folded together
   * @param initialValue  start value
   * @param folder        folding function, the apply method gets the accumulated value
   *                      as its first and the element as its second argument,
   *                      and returns the result of their combination
   * @param <V>           accumulated value type
   * @param <T>           list type
   * @return accumulated value
   */
  public static <V, T> V foldRight(@NotNull List<T> list,
                                   V initialValue,
                                   @NotNull BiFunction<? super V, ? super T, ? extends V> folder)
  {
    return foldRightImpl(list, initialValue, folder);
  }

  /**
   * Do right-associative folding of a list iterator using it in reverse direction.
   * <p>
   * Folding is the reduction of the elements of a collection into one value,
   * by applying the same function on each element with the result of the
   * last call.
   * <p>
   * Currently I have no useful example where right-folding in Java is making sense,
   * so you should prefer
   * {@link #foldLeft(Iterable, Object, BiFunction)},
   * especially as the implementation for iteration is requiring an intermediate copy.
   *
   * @param reverseIterator iterator which is traverse in reverse direction
   * @param initialValue  start value
   * @param folder        folding function, the apply method gets the accumulated value
   *                      as its first and the element as its second argument,
   *                      and returns the result of their combination
   * @param <V>           accumulated value type
   * @param <T>           list type
   * @return accumulated value
   */
  public static <V, T> V foldRight(@NotNull ListIterator<T> reverseIterator,
                                   V initialValue,
                                   @NotNull BiFunction<? super V, ? super T, ? extends V> folder)
  {
    return foldRightImpl(reverseIterator, initialValue, folder);
  }

  /**
   * Add the mapped elements of an array into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link #map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Example:
   * <blockquote><pre>
   *    // convert an array of strings containing double values into a collection of doubles
   *    String[] values = { "1.2", "-7.07", "3.14250255358979" };
   *    List&lt;Double&gt; result = Types.map(new ArrayList&lt;Double&gt;(values.length),
   *                                    values,
   *                                    new Function1&lt;Double, String&gt;() {
   *                                      public Double convert(String object) {
   *                                        return Double.parseDouble(object);
   *                                      }
   *                                    });
   * </pre></blockquote>
   *
   * @param target         collection where the mapped elements are added
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <C extends Collection<? super T>, T, S> C mapX(@NotNull C target,
                                                               @NotNull S[] array,
                                                               @NotNull Function<? super S, ? extends Iterable<T>> typeConverter)
  {
    return mapX(target, asList(array), typeConverter);
  }

  /**
   * Add the mapped elements of an iteration into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link #map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that target and iterable are not allowed to be the same object, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   * <p>
   * Note that target and iterable are not allowed to be the same object, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   * <p>
   * Example:
   * <blockquote><pre>
   *    // convert an array of strings containing double values into a collection of doubles
   *    String[] values = { "1.2", "-7.07", "3.14250255358979" };
   *    List&lt;Double&gt; result = Types.map(new ArrayList&lt;Double&gt;(values.length),
   *                                    Types.asList(values),
   *                                    new Function1&lt;Double, String&gt;() {
   *                                      public Double convert(String object) {
   *                                        return Double.parseDouble(object);
   *                                      }
   *                                    });
   * </pre></blockquote>
   *
   * @param target         collection where the mapped elements are added
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <C extends Collection<? super T>, T, S> C mapX(@NotNull C target,
                                                               @NotNull Iterable<S> iterable,
                                                               @NotNull Function<? super S, ? extends Iterable<T>> typeConverter)
  {
    if (target == iterable) {
      throw new IllegalArgumentException("'target' and 'iterable' may not be the same object!");
    }
    return mapX(target, iterable.iterator(), typeConverter);
  }

  /**
   * Add the mapped elements of an iterator into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link #map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   * <p>
   * See {@link #map(java.util.Collection, Iterable, Function)} for an easy to adapt example.
   *
   * @param target         collection where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <C extends Collection<? super T>, T, S> C mapX(@NotNull C target,
                                                               @NotNull Iterator<S> iterator,
                                                               @NotNull Function<? super S, ? extends Iterable<T>> typeConverter)
  {
    return mapXImpl(target, iterator, typeConverter);
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link #map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   * <p>
   * See {@link #map(java.util.Collection, Iterable, Function)} for an easy to adapt example.
   *
   * @param target         collection where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <C extends Collection<? super T>, T, S> C mapX(@NotNull C target,
                                                               @NotNull Enumeration<S> enumeration,
                                                               @NotNull Function<? super S, ? extends Iterable<T>> typeConverter)
  {
    return mapXImpl(target, enumeration, typeConverter);
  }

  /**
   * Get the mapped elements of an iterable.
   * <p>
   * See {@link #map(java.util.Collection, Iterable, Function)} for a method which allows to define the
   * result collection.
   * <p>
   * See {@link #map(Iterable, Function, Object)} for a method which allows to discard elements.
   * <p>
   * Example:
   * <blockquote><pre>
   *    // convert an array of strings containing double values into a collection of doubles
   *    String[] values = { "1.2", "-7.07", "3.14250255358979" };
   *    List&lt;Double&gt; result = Types.map(Types.asList(values),
   *                                    new Function1&lt;Double, String&gt;() {
   *                                      public Double convert(String object) {
   *                                        return Double.parseDouble(object);
   *                                      }
   *                                    });
   * </pre></blockquote>
   *
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   */
  @NotNull
  public static <T, S> LinkedList<T> map(@NotNull Iterable<S> iterable,
                                         @NotNull Function<? super S, T> typeConverter)
  {
    return map(new LinkedList<>(), iterable, typeConverter);
  }

  /**
   * Get the mapped elements of a collection.
   * <p>
   * This method has the advantage that the size of the collection is known, so an {@link ArrayList}
   * is used for the resulting collection.
   * <p>
   * See {@link #map(java.util.Collection, Iterable, Function)} for a method which allows to define the
   * result collection.
   * <p>
   * See {@link #map(Iterable, Function, Object)} for a method which allows to discard elements.
   * @param collection     collection to be mapped
   * @param typeConverter  type converter doing the mapping
   * @param <T>            target (mapped) type
   * @param <S>            source target
   * @return linked list with mapping results
   */
  @NotNull
  public static <T, S> ArrayList<T> map(@NotNull Collection<S> collection,
                                        @NotNull Function<? super S, T> typeConverter)
  {
    return map(new ArrayList<>(collection.size()), collection, typeConverter);
  }

  /**
   * Get the mapped elements of a countable.
   * <p>
   * This method has the advantage that the size of the countable is known, so an {@link ArrayList}
   * is used for the resulting collection.
   * <p>
   * See {@link #map(java.util.Collection, Iterable, Function)} for a method which allows to define the
   * result collection.
   * <p>
   * See {@link #map(Iterable, Function, Object)} for a method which allows to discard elements.
   * @param countable     collection to be mapped
   * @param typeConverter  type converter doing the mapping
   * @param <T>            target (mapped) type
   * @param <S>            source target
   * @return linked list with mapping results
   */
  @NotNull
  public static <T, S> ArrayList<T> map(@NotNull Countable<S> countable,
                                        @NotNull Function<? super S, T> typeConverter)
  {
    return map(new ArrayList<>(countable.size()), countable, typeConverter);
  }

  /**
   * Get the mapped elements of an iterable as a hash map..
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * See {@link #mapM(java.util.Map, Iterable, Function)} for a method which allows to define the
   * result map.
   * <p>
   * See {@link #mapM(Iterable, Function, Object)} for a method which allows to discard elements.
   *
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   */
  @NotNull
  public static <T, S> HashMap<T, S> mapM(@NotNull Iterable<S> iterable,
                                          @NotNull Function<? super S, T> typeConverter)
  {
    return mapM(new HashMap<T, S>(), iterable, typeConverter);
  }

  /**
   * Get the mapped elements of an iterable as a hash map..
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * See {@link #mapR(java.util.Map, Iterable, Function)} for a method which allows to define the
   * result map.
   * <p>
   * See {@link #mapR(Iterable, Function, Object)} for a method which allows to discard elements.
   *
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   */
  @NotNull
  public static <T, S> HashMap<S, T> mapR(@NotNull Iterable<S> iterable,
                                          @NotNull Function<? super S, T> typeConverter)
  {
    return mapR(new HashMap<>(), iterable, typeConverter);
  }

  /**
   * Get the mapped elements of an array.
   * <p>
   * See {@link #map(java.util.Collection, Object[], Function)} for a method which allows to define the
   * result collection.
   * <p>
   * See {@link #map(Object[], Function, Object)} for a method which allows to discard elements.
   * <p>
   * Example:
   * <blockquote><pre>
   *    // convert an array of strings containing double values into a collection of doubles
   *    String[] values = { "1.2", "-7.07", "3.14250255358979" };
   *    List&lt;Double&gt; result = Types.map(values,
   *                                    new Function1&lt;Double, String&gt;() {
   *                                      public Double convert(String object) {
   *                                        return Double.parseDouble(object);
   *                                      }
   *                                    });
   * </pre></blockquote>
   *
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   */
  @NotNull
  public static <T, S> ArrayList<T> map(@NotNull S[] array,
                                        @NotNull Function<? super S, T> typeConverter)
  {
    return map(new ArrayList<>(array.length), asList(array), typeConverter);
  }

  /**
   * Get the mapped elements of an array as a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * See {@link #mapM(java.util.Map, Object[], Function)} for a method which allows to define the
   * result collection.
   * <p>
   * See {@link #mapM(Object[], Function, Object)} for a method which allows to discard elements.
   *
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   */
  @NotNull
  public static <T, S> HashMap<T, S> mapM(@NotNull S[] array,
                                          @NotNull Function<? super S, T> typeConverter)
  {
    return mapM(new HashMap<>(), asList(array), typeConverter);
  }

  /**
   * Get the mapped elements of an array as a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * See {@link #mapR(java.util.Map, Object[], Function)} for a method which allows to define the
   * result collection.
   * <p>
   * See {@link #mapR(Object[], Function, Object)} for a method which allows to discard elements.
   *
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   */
  @NotNull
  public static <T, S> HashMap<S, T> mapR(@NotNull S[] array,
                                          @NotNull Function<? super S, T> typeConverter)
  {
    return mapR(new HashMap<>(), asList(array), typeConverter);
  }

  /**
   * Get the mapped elements of an iterator.
   * <p>
   * See {@link #map(java.util.Collection, Iterable, Function)} for a method which allows to define the
   * result collection.
   * <p>
   * See {@link #map(Iterable, Function, Object)} for a method which allows to discard elements.
   * <p>
   * See {@link #map(Iterable, Function)} for an easy to adapt example.
   *
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   * @see #view(java.util.Iterator, Function)
   */
  @NotNull
  public static <T, S> LinkedList<T> map(@NotNull Iterator<S> iterator,
                                         @NotNull Function<? super S, T> typeConverter)
  {
    return map(new LinkedList<T>(), iterator, typeConverter);
  }

  /**
   * Get the mapped elements of an iterator as a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * See {@link #mapM(java.util.Map, Iterable, Function)} for a method which allows to define the
   * result collection.
   * <p>
   * See {@link #mapM(Iterable, Function, Object)} for a method which allows to discard elements.
   *
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   */
  @NotNull
  public static <T, S> HashMap<T, S> mapM(@NotNull Iterator<S> iterator,
                                          @NotNull Function<? super S, T> typeConverter)
  {
    return mapM(new HashMap<T, S>(), iterator, typeConverter);
  }

  /**
   * Get the mapped elements of an iterator as a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * See {@link #mapR(java.util.Map, Iterable, Function)} for a method which allows to define the
   * result collection.
   * <p>
   * See {@link #mapR(Iterable, Function, Object)} for a method which allows to discard elements.
   *
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   */
  @NotNull
  public static <T, S> HashMap<S, T> mapR(@NotNull Iterator<S> iterator,
                                          @NotNull Function<? super S, T> typeConverter)
  {
    return mapR(new HashMap<S, T>(), iterator, typeConverter);
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   *
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <T, S> LinkedList<T> map(@NotNull Enumeration<S> enumeration,
                                         @NotNull Function<? super S, T> typeConverter)
  {
    return map(new LinkedList<T>(), enumeration, typeConverter);
  }

  /**
   * Add the mapped elements of an enumeration into a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   *
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <T, S> HashMap<T, S> mapM(@NotNull Enumeration<S> enumeration,
                                          @NotNull Function<? super S, T> typeConverter)
  {
    return mapM(new HashMap<>(), enumeration, typeConverter);
  }

  /**
   * Add the mapped elements of an enumeration into a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   *
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <T, S> HashMap<S, T> mapR(@NotNull Enumeration<S> enumeration,
                                          @NotNull Function<? super S, T> typeConverter)
  {
    return mapR(new HashMap<>(), enumeration, typeConverter);
  }

  /**
   * Get the mapped elements of an iterable.
   * <p>
   * See {@link #map(java.util.Collection, Iterable, Function)} for a method which allows to define the
   * result collection.
   * <p>
   * See {@link #map(Iterable, Function, Object)} for a method which allows to discard elements.
   * <p>
   * Example:
   * <blockquote><pre>
   *    // convert an array of strings containing double values into a collection of doubles
   *    String[] values = { "1.2", "-7.07", "3.14250255358979" };
   *    List&lt;Double&gt; result = Types.map(Types.asList(values),
   *                                    new Function1&lt;Double, String&gt;() {
   *                                      public Double convert(String object) {
   *                                        return Double.parseDouble(object);
   *                                      }
   *                                    });
   * </pre></blockquote>
   *
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   */
  @NotNull
  public static <T, S> LinkedList<T> mapX(@NotNull Iterable<S> iterable,
                                          @NotNull Function<? super S, ? extends Iterable<T>> typeConverter)
  {
    return mapX(new LinkedList<T>(), iterable, typeConverter);
  }

  /**
   * Get the mapped elements of an array.
   * <p>
   * See {@link #map(java.util.Collection, Object[], Function)} for a method which allows to define the
   * result collection.
   * <p>
   * See {@link #map(Object[], Function, Object)} for a method which allows to discard elements.
   * <p>
   * Example:
   * <blockquote><pre>
   *    // convert an array of strings containing double values into a collection of doubles
   *    String[] values = { "1.2", "-7.07", "3.14250255358979" };
   *    List&lt;Double&gt; result = Types.map(values,
   *                                    new Function1&lt;Double, String&gt;() {
   *                                      public Double convert(String object) {
   *                                        return Double.parseDouble(object);
   *                                      }
   *                                    });
   * </pre></blockquote>
   *
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   */
  @NotNull
  public static <T, S> LinkedList<T> mapX(@NotNull S[] array,
                                          @NotNull Function<? super S, ? extends Iterable<T>> typeConverter)
  {
    return mapX(new LinkedList<T>(), asList(array), typeConverter);
  }

  /**
   * Get the mapped elements of an iterator.
   * <p>
   * See {@link #map(java.util.Collection, Iterable, Function)} for a method which allows to define the
   * result collection.
   * <p>
   * See {@link #map(Iterable, Function, Object)} for a method which allows to discard elements.
   * <p>
   * See {@link #map(Iterable, Function)} for an easy to adapt example.
   *
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   */
  @NotNull
  public static <T, S> LinkedList<T> mapX(@NotNull Iterator<S> iterator,
                                          @NotNull Function<? super S, ? extends Iterable<T>> typeConverter)
  {
    return mapX(new LinkedList<T>(), iterator, typeConverter);
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   *
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static < T, S> LinkedList<T> mapX(@NotNull Enumeration<S> enumeration,
                                           @NotNull Function<? super S, ? extends Iterable<T>> typeConverter)
  {
    return mapX(new LinkedList<T>(), enumeration, typeConverter);
  }

  /**
   * Add the mapped elements of an array into a collection.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param target         collection where the mapped elements are added
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <C extends Collection<? super T>, T, S> C map(@NotNull C target,
                                                              @NotNull S[] array,
                                                              @NotNull Function<? super S, T> typeConverter,
                                                              @Nullable T deleteMark)
  {
    return map(target, asList(array), typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an array into a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param target         map where the mapped elements are added
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <M extends Map<? super T, ? super S>, T, S> M mapM(@NotNull M target,
                                                                   @NotNull S[] array,
                                                                   @NotNull Function<? super S, T> typeConverter,
                                                                   @Nullable T deleteMark)
  {
    return mapM(target, asList(array), typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an array into a map.
   * <p>
   * The incoming values will become keys, pointing to the associated mapped values.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param target         map where the mapped elements are added
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <M extends Map<? super S, ? super T>, T, S> M mapR(@NotNull M target,
                                                                   @NotNull S[] array,
                                                                   @NotNull Function<? super S, T> typeConverter,
                                                                   @Nullable T deleteMark)
  {
    return mapR(target, asList(array), typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an iteration into a collection.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that target and iterable are not allowed to be the same object, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         collection where the mapped elements are added
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <C extends Collection<? super T>, T, S> C map(@NotNull C target,
                                                              @NotNull Iterable<S> iterable,
                                                              @NotNull Function<? super S, T> typeConverter,
                                                              @Nullable T deleteMark)
  {
    if (target == iterable) {
      throw new IllegalArgumentException("'target' and 'iterable' may not be the same object!");
    }
    return map(target, iterable.iterator(), typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an iteration to a consumer.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param target         consumer applied to each mapped element
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <C extends Consumer<? super T>, T, S> C map(@NotNull C target,
                                                            @NotNull Iterable<S> iterable,
                                                            @NotNull Function<? super S, T> typeConverter,
                                                            @Nullable T deleteMark)
  {
    if (target == iterable) {
      throw new IllegalArgumentException("'target' and 'iterable' may not be the same object!");
    }
    return map(target, iterable.iterator(), typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an iteration into a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that target and iterable are not allowed to be the same object, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <M extends Map<? super T, ? super S>, T, S> M mapM(@NotNull M target,
                                                                   @NotNull Iterable<S> iterable,
                                                                   @NotNull Function<? super S, T> typeConverter,
                                                                   @Nullable T deleteMark)
  {
    if (target == iterable) {
      throw new IllegalArgumentException("'target' and 'iterable' may not be the same object!");
    }
    return mapM(target, iterable.iterator(), typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an iteration into a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that target and iterable are not allowed to be the same object, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <M extends Map<? super S, ? super T>, T, S> M mapR(@NotNull M target,
                                                                   @NotNull Iterable<S> iterable,
                                                                   @NotNull Function<? super S, T> typeConverter,
                                                                   @Nullable T deleteMark)
  {
    if (target == iterable) {
      throw new IllegalArgumentException("'target' and 'iterable' may not be the same object!");
    }
    return mapR(target, iterable.iterator(), typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an iterator into a collection.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         collection where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <C extends Collection<? super T>, T, S> C map(@NotNull C target,
                                                              @NotNull Iterator<S> iterator,
                                                              @NotNull Function<? super S, T> typeConverter,
                                                              @Nullable T deleteMark)
  {
    return mapImpl(target, iterator, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an iterator to a consumer.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param target         consumer to which all mapped elements are forwarded
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <C extends Consumer<? super T>, T, S> C map(@NotNull C target,
                                                            @NotNull Iterator<S> iterator,
                                                            @NotNull Function<? super S, T> typeConverter,
                                                            @Nullable T deleteMark)
  {
    return mapImpl(target, iterator, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an iterator into a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <M extends Map<? super T, ? super S>, T, S> M mapM(@NotNull M target,
                                                                   @NotNull Iterator<S> iterator,
                                                                   @NotNull Function<? super S, T> typeConverter,
                                                                   @Nullable T deleteMark)
  {
    return mapMImpl(target, iterator, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an iterator into a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <M extends Map<? super S, ? super T>, T, S> M mapR(@NotNull M target,
                                                                   @NotNull Iterator<S> iterator,
                                                                   @NotNull Function<? super S, T> typeConverter,
                                                                   @Nullable T deleteMark)
  {
    return mapRImpl(target, iterator, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         collection where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <C extends Collection<? super T>, T, S> C map(@NotNull C target,
                                                              @NotNull Enumeration<S> enumeration,
                                                              @NotNull Function<? super S, T> typeConverter,
                                                              @Nullable T deleteMark)
  {
    return mapImpl(target, enumeration, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an enumeration into a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <M extends Map<? super T, ? super S>, T, S> M mapM(@NotNull M target,
                                                                   @NotNull Enumeration<S> enumeration,
                                                                   @NotNull Function<? super S, T> typeConverter,
                                                                   @Nullable T deleteMark)
  {
    return mapMImpl(target, enumeration, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an enumeration into a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static <M extends Map<? super S, ? super T>, T, S> M mapR(@NotNull M target,
                                                                   @NotNull Enumeration<S> enumeration,
                                                                   @NotNull Function<? super S, T> typeConverter,
                                                                   @Nullable T deleteMark)
  {
    return mapRImpl(target, enumeration, typeConverter, deleteMark);
  }

  /**
   * Get the mapped elements of an array.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Example:
   * <blockquote><pre>
   *    // convert an array of strings containing double values into a collection of doubles
   *    String[] values = { "1.2", "-7.07", "3.14250255358979" };
   *    List&lt;Double&gt; result = Types.map(values,
   *                                    new Function1&lt;Double, String&gt;() {
   *                                      public Double convert(String object) {
   *                                        return Double.parseDouble(object);
   *                                      }
   *                                    });
   * </pre></blockquote>
   *
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   */
  @NotNull
  public static <T, S> LinkedList<T> map(@NotNull S[] array,
                                         @NotNull Function<? super S, T> typeConverter,
                                         @Nullable T deleteMark)
  {
    return map(new LinkedList<T>(), asList(array), typeConverter, deleteMark);
  }

  /**
   * Get the mapped elements of an array as a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with mapping results
   */
  @NotNull
  public static <T, S> HashMap<T, S> mapM(@NotNull S[] array,
                                          @NotNull Function<? super S, T> typeConverter,
                                          @Nullable T deleteMark)
  {
    return mapM(new HashMap<T, S>(), asList(array), typeConverter, deleteMark);
  }

  /**
   * Get the mapped elements of an array as a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with mapping results
   */
  @NotNull
  public static <T, S> HashMap<S, T> mapR(@NotNull S[] array,
                                          @NotNull Function<? super S, T> typeConverter,
                                          @Nullable T deleteMark)
  {
    return mapR(new HashMap<S, T>(), asList(array), typeConverter, deleteMark);
  }

  /**
   * Get the mapped elements of an iterable.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Example:
   * <blockquote><pre>
   *    // convert an array of strings containing double values into a collection of doubles
   *    String[] values = { "1.2", "-7.07", "3.14250255358979", "foobar" };
   *    List&lt;Double&gt; result = Types.map(Types.asList(values),
   *                                    new Function1&lt;Double, String&gt;() {
   *                                      public Double convert(String object) {
   *                                        try {
   *                                          return Double.parseDouble(object);
   *                                        } catch (NumberFormatException x) {
   *                                          return null; // delete mark, compare last parameter
   *                                        }
   *                                      }
   *                                    },
   *                                    null);
   * </pre></blockquote>
   *
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   */
  @NotNull
  public static <T, S> LinkedList<T> map(@NotNull Iterable<S> iterable,
                                         @NotNull Function<? super S, T> typeConverter,
                                         @Nullable T deleteMark)
  {
    return map(new LinkedList<T>(), iterable, typeConverter, deleteMark);
  }

  /**
   * Get the mapped elements of an iterable to a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with mapping results
   */
  @NotNull
  public static <T, S> HashMap<T, S> mapM(@NotNull Iterable<S> iterable,
                                          @NotNull Function<? super S, T> typeConverter,
                                          @Nullable T deleteMark)
  {
    return mapM(new HashMap<T, S>(), iterable, typeConverter, deleteMark);
  }

  /**
   * Get the mapped elements of an iterable to a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with mapping results
   */
  @NotNull
  public static <T, S> HashMap<S, T> mapR(@NotNull Iterable<S> iterable,
                                          @NotNull Function<? super S, T> typeConverter,
                                          @Nullable T deleteMark)
  {
    return mapR(new HashMap<S, T>(), iterable, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an iterator into a collection.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   */
  @NotNull
  public static <T, S> LinkedList<T> map(@NotNull Iterator<S> iterator,
                                         @NotNull Function<? super S, T> typeConverter,
                                         @Nullable T deleteMark)
  {
    return map(new LinkedList<T>(), iterator, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an iterator into a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with mapping results
   */
  @NotNull
  public static <T, S> HashMap<T, S> mapM(@NotNull Iterator<S> iterator,
                                          @NotNull Function<? super S, T> typeConverter,
                                          @Nullable T deleteMark)
  {
    return mapM(new HashMap<T, S>(), iterator, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an iterator into a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with mapping results
   */
  @NotNull
  public static <T, S> HashMap<S, T> mapR(@NotNull Iterator<S> iterator,
                                          @NotNull Function<? super S, T> typeConverter,
                                          @Nullable T deleteMark)
  {
    return mapR(new HashMap<S, T>(), iterator, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   */
  @NotNull
  public static < T, S> LinkedList<T> map(@NotNull Enumeration<S> enumeration,
                                          @NotNull Function<? super S, T> typeConverter,
                                          @Nullable T deleteMark)
  {
    return map(new LinkedList<T>(), enumeration, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an enumeration into a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return result map
   */
  @NotNull
  public static < T, S> HashMap<T, S> mapM(@NotNull Enumeration<S> enumeration,
                                           @NotNull Function<? super S, T> typeConverter,
                                           @Nullable T deleteMark)
  {
    return mapM(new HashMap<T, S>(), enumeration, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an enumeration into a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return result map
   */
  @NotNull
  public static < T, S> HashMap<S, T> mapR(@NotNull Enumeration<S> enumeration,
                                           @NotNull Function<? super S, T> typeConverter,
                                           @Nullable T deleteMark)
  {
    return mapR(new HashMap<S, T>(), enumeration, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an array into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link #map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Example:
   * <blockquote><pre>
   *    // convert an array of strings containing double values into a collection of doubles
   *    String[] values = { "1.2", "-7.07", "3.14250255358979" };
   *    List&lt;Double&gt; result = Types.map(new ArrayList&lt;Double&gt;(values.length),
   *                                    values,
   *                                    new Function1&lt;Double, String&gt;() {
   *                                      public Double convert(String object) {
   *                                        return Double.parseDouble(object);
   *                                      }
   *                                    });
   * </pre></blockquote>
   *
   * @param target         collection where the mapped elements are added
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   * @deprecated use {@link #mapFragile(Collection, Object[], FragileFunction1)} instead which
   *             does not require handling a {@link TypeConverterException}
   */
  @NotNull
  @Deprecated
  public static <C extends Collection<? super T>, T, S> C mapE(@NotNull C target,
                                                               @NotNull S[] array,
                                                               @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapE(target, asList(array), typeConverter);
  }

  /**
   * Add the mapped elements of an array into a collection by using a mapper function which might throw
   * a checked exception.
   * @param target         collection where the mapped elements are added
   * @param array          array providing the source elements
   * @param mapper         mapper function converting the source elements into the target elements
   * @return reference to target, allowing to create it directly in the call
   * @param <C>            type of returned collection
   * @param <T>            target (mapped) element type
   * @param <S>            source element type
   * @param <E>            exception type
   * @throws E mapper exception
   */
  @NotNull
  public static <C extends Collection<T>, T, S, E extends Exception> C mapFragile(@NotNull C target,
                                                                                  @NotNull S[] array,
                                                                                  @NotNull FragileFunction1<? extends T, E, ? super S> mapper)
          throws E
  {
    return mapFragile(target, asList(array), mapper);
  }

  /**
   * Add the mapped elements of an array into a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This will add all mapped elements to the target map.
   * See {@link #mapM(java.util.Map, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   *
   * @param target         map where the mapped elements are added
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <M extends Map<? super T, ? super S>, T, S> M mapME(@NotNull M target,
                                                                    @NotNull S[] array,
                                                                    @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapME(target, asList(array), typeConverter);
  }

  /**
   * Add the mapped elements of an array into a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This will add all mapped elements to the target map.
   * See {@link #mapM(java.util.Map, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   *
   * @param target         map where the mapped elements are added
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <M extends Map<? super S, ? super T>, T, S> M mapRE(@NotNull M target,
                                                                    @NotNull S[] array,
                                                                    @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapRE(target, asList(array), typeConverter);
  }

  /**
   * Add the mapped elements of an iteration into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link #map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that target and iterable are not allowed to be the same object, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         collection where the mapped elements are added
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   * @deprecated use {@link #mapFragile(Collection, Iterable, FragileFunction1)} instead which
   *             does not require handling a {@link TypeConverterException}
   */
  @NotNull
  @Deprecated
  public static <C extends Collection<? super T>, T, S> C mapE(@NotNull C target,
                                                               @NotNull Iterable<S> iterable,
                                                               @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    if (target == iterable) {
      throw new IllegalArgumentException("'target' and 'iterable' may not be the same object!");
    }
    return mapE(target, iterable.iterator(), typeConverter);
  }

  /**
   * Add the mapped elements of an iterable into a collection by using a mapper function which might throw
   * a checked exception.
   * @param target         collection where the mapped elements are added
   * @param iterable       iterable providing the source elements
   * @param mapper         mapper function converting the source elements into the target elements
   * @return reference to target, allowing to create it directly in the call
   * @param <C>            type of returned collection
   * @param <T>            target (mapped) element type
   * @param <S>            source element type
   * @param <E>            exception type
   * @throws E mapper exception
   */
  @NotNull
  public static <C extends Collection<T>, T, S, E extends Exception> C mapFragile(@NotNull C target,
                                                                                  @NotNull Iterable<S> iterable,
                                                                                  @NotNull FragileFunction1<? extends T, E, ? super S> mapper)
          throws E
  {
    if (target == iterable) {
      throw new IllegalArgumentException("'target' and 'iterable' may not be the same object!");
    }
    return mapFragile(target, iterable.iterator(), mapper);
  }

  /**
   * Add the mapped elements of an iteration into a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This will add all mapped elements to the target map.
   * See {@link #mapM(java.util.Map, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that target and iterable are not allowed to be the same object, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <M extends Map<? super T, ? super S>, T, S> M mapME(@NotNull M target,
                                                                    @NotNull Iterable<S> iterable,
                                                                    @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    if (target == iterable) {
      throw new IllegalArgumentException("'target' and 'iterable' may not be the same object!");
    }
    return mapME(target, iterable.iterator(), typeConverter);
  }

  /**
   * Add the mapped elements of an iteration into a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This will add all mapped elements to the target map.
   * See {@link #mapM(java.util.Map, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that target and iterable are not allowed to be the same object, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <M extends Map<? super S, ? super T>, T, S> M mapRE(@NotNull M target,
                                                                    @NotNull Iterable<S> iterable,
                                                                    @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    if (target == iterable) {
      throw new IllegalArgumentException("'target' and 'iterable' may not be the same object!");
    }
    return mapRE(target, iterable.iterator(), typeConverter);
  }

  /**
   * Add the mapped elements of an iterator into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link #map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   * <p>
   * See {@link #map(java.util.Collection, Iterable, Function)} for an easy to adapt example.
   *
   * @param target         collection where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   * @deprecated use {@link #mapFragile(Collection, Iterator, FragileFunction1)} instead which
   *             does not require handling a {@link TypeConverterException}
   */
  @NotNull
  @Deprecated
  public static <C extends Collection<? super T>, T, S> C mapE(@NotNull C target,
                                                               @NotNull Iterator<S> iterator,
                                                               @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapEImpl(target, iterator, typeConverter);
  }

  /**
   * Add the mapped elements of an iterator into a collection by using a mapper function which might throw
   * a checked exception.
   * @param target         collection where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param mapper         mapper function converting the source elements into the target elements
   * @return reference to target, allowing to create it directly in the call
   * @param <C>            type of returned collection
   * @param <T>            target (mapped) element type
   * @param <S>            source element type
   * @param <E>            exception type
   * @throws E mapper exception
   */
  @NotNull
  public static <C extends Collection<T>, T, S, E extends Exception> C mapFragile(@NotNull C target,
                                                                                  @NotNull Iterator<S> iterator,
                                                                                  @NotNull FragileFunction1<? extends T, E, ? super S> mapper)
          throws E
  {
    while (iterator.hasNext()) {
      target.add(mapper.apply(iterator.next()));
    }
    return target;
  }

  /**
   * Add the mapped elements of an iterator to a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This will add all mapped elements to the target map.
   * See {@link #mapM(java.util.Map, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   * <p>
   * See {@link #map(java.util.Collection, Iterable, Function)} for an easy to adapt example.
   *
   * @param target         map where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <M extends Map<? super T, ? super S>, T, S> M mapME(@NotNull M target,
                                                                    @NotNull Iterator<S> iterator,
                                                                    @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapMEImpl(target, iterator, typeConverter);
  }

  /**
   * Add the mapped elements of an iterator to a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This will add all mapped elements to the target map.
   * See {@link #mapR(java.util.Map, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <M extends Map<? super S, ? super T>, T, S> M mapRE(@NotNull M target,
                                                                    @NotNull Iterator<S> iterator,
                                                                    @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapREImpl(target, iterator, typeConverter);
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link #map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   * <p>
   * See {@link #map(java.util.Collection, Iterable, Function)} for an easy to adapt example.
   *
   * @param target         collection where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   * @deprecated use {@link #mapFragile(Collection, Enumeration, FragileFunction1)} instead which
   *             does not require handling a {@link TypeConverterException}
   */
  @NotNull
  @Deprecated
  public static <C extends Collection<? super T>, T, S> C mapE(@NotNull C target,
                                                               @NotNull Enumeration<S> enumeration,
                                                               @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapEImpl(target, enumeration, typeConverter);
  }

  /**
   * Add the mapped elements of an enumeration into a collection by using a mapper function which might throw
   * a checked exception.
   * @param target         collection where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param mapper         mapper function converting the source elements into the target elements
   * @return reference to target, allowing to create it directly in the call
   * @param <C>            type of returned collection
   * @param <T>            target (mapped) element type
   * @param <S>            source element type
   * @param <E>            exception type
   * @throws E mapper exception
   */
  @NotNull
  public static <C extends Collection<T>, T, S, E extends Exception> C mapFragile(@NotNull C target,
                                                                                  @NotNull Enumeration<S> enumeration,
                                                                                  @NotNull FragileFunction1<? extends T, E, ? super S> mapper)
          throws E
  {
    while (enumeration.hasMoreElements()) {
      target.add(mapper.apply(enumeration.nextElement()));
    }
    return target;
  }

  /**
   * Add the mapped elements of an enumeration to a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This will add all mapped elements to the target map.
   * See {@link #mapM(java.util.Map, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <M extends Map<? super T, ? super S>, T, S> M mapME(@NotNull M target,
                                                                    @NotNull Enumeration<S> enumeration,
                                                                    @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapMEImpl(target, enumeration, typeConverter);
  }

  /**
   * Add the mapped elements of an enumeration to a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This will add all mapped elements to the target map.
   * See {@link #mapM(java.util.Map, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <M extends Map<? super S, ? super T>, T, S> M mapRE(@NotNull M target,
                                                                    @NotNull Enumeration<S> enumeration,
                                                                    @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapREImpl(target, enumeration, typeConverter);
  }

  /**
   * Get the mapped elements of an iterable.
   * <p>
   * See {@link #map(java.util.Collection, Iterable, Function)} for a method which allows to define the
   * result collection.
   * <p>
   * See {@link #map(Iterable, Function, Object)} for a method which allows to discard elements.
   *
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   * @deprecated use {@link #mapFragile(Iterable, FragileFunction1)} instead which
   *             does not require handling a {@link TypeConverterException}
   */
  @NotNull
  @Deprecated
  public static <T, S> LinkedList<T> mapE(@NotNull Iterable<S> iterable,
                                          @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapE(new LinkedList<T>(), iterable, typeConverter);
  }

  /**
   * Get the mapped elements of an iterable while using a mapper which might throw a checked exception.
   *
   * @param iterable       iterable providing the source elements
   * @param mapper         mapper converting the source elements into the target elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @param <E>            exception type
   * @return linked list with mapping results
   * @throws E if the mapper throws
   */
  @NotNull
  public static <T, S, E extends Exception> LinkedList<T> mapFragile(@NotNull Iterable<S> iterable,
                                                                     @NotNull FragileFunction1<? extends T, E, ? super S> mapper)
          throws E
  {
    return mapFragile(new LinkedList<T>(), iterable, mapper);
  }

  /**
   * Get the mapped elements of an iterable as a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * See {@link #mapM(java.util.Map, Iterable, Function)} for a method which allows to define the
   * result map.
   * <p>
   * See {@link #mapM(Iterable, Function, Object)} for a method which allows to discard elements.
   *
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <T, S> HashMap<T, S> mapME(@NotNull Iterable<S> iterable,
                                           @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapME(new HashMap<T, S>(), iterable, typeConverter);
  }

  /**
   * Get the mapped elements of an iterable as a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * See {@link #mapR(java.util.Map, Iterable, Function)} for a method which allows to define the
   * result map.
   * <p>
   * See {@link #mapR(Iterable, Function, Object)} for a method which allows to discard elements.
   *
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <T, S> HashMap<S, T> mapRE(@NotNull Iterable<S> iterable,
                                           @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapRE(new HashMap<S, T>(), iterable, typeConverter);
  }

  /**
   * Get the mapped elements of an array.
   * <p>
   * See {@link #map(java.util.Collection, Object[], Function)} for a method which allows to define the
   * result collection.
   * <p>
   * See {@link #map(Object[], Function, Object)} for a method which allows to discard elements.
   *
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   * @deprecated use {@link #mapFragile(Object[], FragileFunction1)} instead which
   *             does not require handling a {@link TypeConverterException}
   */
  @NotNull
  @Deprecated
  public static <T, S> LinkedList<T> mapE(@NotNull S[] array,
                                          @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapE(new LinkedList<T>(), asList(array), typeConverter);
  }

  /**
   * Get the mapped elements of an array while using a mapper which might throw a checked exception.
   *
   * @param array          array providing the source elements
   * @param mapper         mapper converting the source elements into the target elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @param <E>            exception type
   * @return array list with mapping results
   * @throws E if the mapper throws
   */
  @NotNull
  public static <T, S, E extends Exception> ArrayList<T> mapFragile(@NotNull S[] array,
                                                                    @NotNull FragileFunction1<? extends T, E, ? super S> mapper)
          throws E
  {
    return mapFragile(asList(array), mapper);
  }

  /**
   * Get the mapped elements of a collection while using a mapper which might throw a checked exception.
   *
   * @param collection     collection providing the source elements
   * @param mapper         mapper converting the source elements into the target elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @param <E>            exception type
   * @return array list with mapping results
   * @throws E if the mapper throws
   */
  @NotNull
  public static <T, S, E extends Exception> ArrayList<T> mapFragile(@NotNull Collection<S> collection,
                                                                    @NotNull FragileFunction1<? extends T, E, ? super S> mapper)
          throws E
  {
    return mapFragile(new ArrayList<>(collection.size()), collection, mapper);
  }

  /**
   * Get the mapped elements of an array while using a mapper which might throw a checked exception.
   *
   * @param countable      countable providing the source elements
   * @param mapper         mapper converting the source elements into the target elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @param <E>            exception type
   * @return array list with mapping results
   * @throws E if the mapper throws
   */
  @NotNull
  public static <T, S, E extends Exception> ArrayList<T> mapFragile(@NotNull Countable<S> countable,
                                                                    @NotNull FragileFunction1<? extends T, E, ? super S> mapper)
          throws E
  {
    return mapFragile(new ArrayList<>(countable.size()), countable, mapper);
  }

  /**
   * Get the mapped elements of an iterator while using a mapper which might throw a checked exception.
   *
   * @param iterator       iterator providing the source elements, consumed after the call
   * @param mapper         mapper converting the source elements into the target elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @param <E>            exception type
   * @return linked list with mapping results
   * @throws E if the mapper throws
   */
  @NotNull
  public static <T, S, E extends Exception> LinkedList<T> mapFragile(@NotNull Iterator<S> iterator,
                                                                     @NotNull FragileFunction1<? extends T, E, ? super S> mapper)
          throws E
  {
    return mapFragile(new LinkedList<>(), iterator, mapper);
  }

  /**
   * Get the mapped elements of an enumeration while using a mapper which might throw a checked exception.
   *
   * @param enumeration    enumeration providing the source elements, consumed after the call
   * @param mapper         mapper converting the source elements into the target elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @param <E>            exception type
   * @return linked list with mapping results
   * @throws E if the mapper throws
   */
  @NotNull
  public static <T, S, E extends Exception> LinkedList<T> mapFragile(@NotNull Enumeration<S> enumeration,
                                                                     @NotNull FragileFunction1<? extends T, E, ? super S> mapper)
          throws E
  {
    return mapFragile(new LinkedList<>(), enumeration, mapper);
  }

  /**
   * Get the mapped elements of an array as a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * See {@link #mapM(java.util.Map, Object[], Function)} for a method which allows to define the
   * result map.
   * <p>
   * See {@link #mapM(Object[], Function, Object)} for a method which allows to discard elements.
   *
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <T, S> HashMap<T, S> mapME(@NotNull S[] array,
                                           @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapME(new HashMap<T, S>(), asList(array), typeConverter);
  }

  /**
   * Get the mapped elements of an array as a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * See {@link #mapR(java.util.Map, Object[], Function)} for a method which allows to define the
   * result map.
   * <p>
   * See {@link #mapR(Object[], Function, Object)} for a method which allows to discard elements.
   *
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <T, S> HashMap<S, T> mapRE(@NotNull S[] array,
                                           @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapRE(new HashMap<S, T>(), asList(array), typeConverter);
  }

  /**
   * Get the mapped elements of an iterator.
   * <p>
   * See {@link #map(java.util.Collection, Iterable, Function)} for a method which allows to define the
   * result collection.
   * <p>
   * See {@link #map(Iterable, Function, Object)} for a method which allows to discard elements.
   * <p>
   * See {@link #map(Iterable, Function)} for an easy to adapt example.
   *
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   * @deprecated use {@link #mapFragile(Iterator, FragileFunction1)} instead which
   *             does not require handling a {@link TypeConverterException}
   */
  @NotNull
  @Deprecated
  public static <T, S> LinkedList<T> mapE(@NotNull Iterator<S> iterator,
                                          @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapE(new LinkedList<T>(), iterator, typeConverter);
  }

  /**
   * Get the mapped elements of an iterator as a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * See {@link #mapM(java.util.Map, Iterable, Function)} for a method which allows to define the
   * result map.
   * <p>
   * See {@link #mapM(Iterable, Function, Object)} for a method which allows to discard elements.
   *
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <T, S> HashMap<T, S> mapME(@NotNull Iterator<S> iterator,
                                           @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapME(new HashMap<T, S>(), iterator, typeConverter);
  }

  /**
   * Get the mapped elements of an iterator as a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * See {@link #mapR(java.util.Map, Iterable, Function)} for a method which allows to define the
   * result map.
   * <p>
   * See {@link #mapR(Iterable, Function, Object)} for a method which allows to discard elements.
   *
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <T, S> HashMap<S, T> mapRE(@NotNull Iterator<S> iterator,
                                           @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapRE(new HashMap<S, T>(), iterator, typeConverter);
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   *
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with converted elements
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   * @deprecated use {@link #mapFragile(Enumeration, FragileFunction1)} instead which
   *             does not require handling a {@link TypeConverterException}
   */
  @NotNull
  @Deprecated
  public static <T, S> LinkedList<T> mapE(@NotNull Enumeration<S> enumeration,
                                          @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapE(new LinkedList<T>(), enumeration, typeConverter);
  }

  /**
   * Add the mapped elements of an enumeration into a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   *
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with converted elements
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <T, S> HashMap<T, S> mapME(@NotNull Enumeration<S> enumeration,
                                           @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapME(new HashMap<T, S>(), enumeration, typeConverter);
  }

  /**
   * Add the mapped elements of an enumeration into a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   *
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with converted elements
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <T, S> HashMap<S, T> mapRE(@NotNull Enumeration<S> enumeration,
                                           @NotNull FragileTypeConverter<T, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapRE(new HashMap<S, T>(), enumeration, typeConverter);
  }

  /**
   * Add the mapped elements of an array into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link #map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   *
   * @param target         collection where the mapped elements are added
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <C extends Collection<? super T>, T, S> C mapXE(@NotNull C target,
                                                                @NotNull S[] array,
                                                                @NotNull FragileTypeConverter<? extends Iterable<T>, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapXE(target, asList(array), typeConverter);
  }

  /**
   * Add the mapped elements of an iteration into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link #map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that target and iterable are not allowed to be the same object, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         collection where the mapped elements are added
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <C extends Collection<? super T>, T, S> C mapXE(@NotNull C target,
                                                                @NotNull Iterable<S> iterable,
                                                                @NotNull FragileTypeConverter<? extends Iterable<T>, ? super S> typeConverter)
          throws TypeConverterException
  {
    if (target == iterable) {
      throw new IllegalArgumentException("'target' and 'iterable' may not be the same object!");
    }
    return mapXE(target, iterable.iterator(), typeConverter);
  }

  /**
   * Add the mapped elements of an iterator into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link #map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * See {@link #map(java.util.Collection, Iterable, Function)} for an easy to adapt example.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         collection where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <C extends Collection<? super T>, T, S> C mapXE(@NotNull C target,
                                                                @NotNull Iterator<S> iterator,
                                                                @NotNull FragileTypeConverter<? extends Iterable<T>, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapXEImpl(target, iterator, typeConverter);
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   * <p>
   * This will add all mapped elements to the target collection.
   * See {@link #map(java.util.Collection, Iterable, Function, Object)} for a method
   * which allows to discard elements.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   * <p>
   * See {@link #map(java.util.Collection, Iterable, Function)} for an easy to adapt example.
   *
   * @param target         collection where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <C extends Collection<? super T>, T, S> C mapXE(@NotNull C target,
                                                                @NotNull Enumeration<S> enumeration,
                                                                @NotNull FragileTypeConverter<? extends Iterable<T>, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapXEImpl(target, enumeration, typeConverter);
  }

  /**
   * Get the mapped elements of an iterable.
   * <p>
   * See {@link #map(java.util.Collection, Iterable, Function)} for a method which allows to define the
   * result collection.
   * <p>
   * See {@link #map(Iterable, Function, Object)} for a method which allows to discard elements.
   *
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <T, S> LinkedList<T> mapXE(@NotNull Iterable<S> iterable,
                                           @NotNull FragileTypeConverter<? extends Iterable<T>, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapXE(new LinkedList<T>(), iterable, typeConverter);
  }

  /**
   * Get the mapped elements of an array.
   * <p>
   * See {@link #map(java.util.Collection, Object[], Function)} for a method which allows to define the
   * result collection.
   * <p>
   * See {@link #map(Object[], Function, Object)} for a method which allows to discard elements.
   *
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <T, S> LinkedList<T> mapXE(@NotNull S[] array,
                                           @NotNull FragileTypeConverter<? extends Iterable<T>, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapXE(new LinkedList<T>(), asList(array), typeConverter);
  }

  /**
   * Get the mapped elements of an iterator.
   * <p>
   * See {@link #map(java.util.Collection, Iterable, Function)} for a method which allows to define the
   * result collection.
   * <p>
   * See {@link #map(Iterable, Function, Object)} for a method which allows to discard elements.
   * <p>
   * See {@link #map(Iterable, Function)} for an easy to adapt example.
   *
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <T, S> LinkedList<T> mapXE(@NotNull Iterator<S> iterator,
                                           @NotNull FragileTypeConverter<? extends Iterable<T>, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapXE(new LinkedList<T>(), iterator, typeConverter);
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   *
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static < T, S> LinkedList<T> mapXE(@NotNull Enumeration<S> enumeration,
                                            @NotNull FragileTypeConverter<? extends Iterable<T>, ? super S> typeConverter)
          throws TypeConverterException
  {
    return mapXE(new LinkedList<T>(), enumeration, typeConverter);
  }

  /**
   * Add the mapped elements of an array into a collection.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param target         collection where the mapped elements are added
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   * @deprecated use {@link #mapFragile(Collection, Object[], FragileFunction1, Object)} instead which
   *             does not require handling a {@link TypeConverterException}
   */
  @NotNull
  @Deprecated
  public static <C extends Collection<? super T>, T, S> C mapE(@NotNull C target,
                                                               @NotNull S[] array,
                                                               @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                                               @Nullable T deleteMark)
          throws TypeConverterException
  {
    return mapE(target, asList(array), typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an array into a collection while using a mapper which might throw an exception.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param target         collection where the mapped elements are added
   * @param array          array providing the source elements
   * @param mapper         fragile mapper for converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @param <E>            exception type
   * @return reference to target, allowing to create it directly in the call
   * @throws E             if the mapper throws
   */
  @NotNull
  public static <C extends Collection<T>, T, S, E extends Exception> C mapFragile(@NotNull C target,
                                                                                  @NotNull S[] array,
                                                                                  @NotNull FragileFunction1<? extends T, E, ? super S> mapper,
                                                                                  @Nullable T deleteMark)
          throws E
  {
    return mapFragile(target, asList(array), mapper, deleteMark);
  }

  /**
   * Add the mapped elements of an array into a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param target         map where the mapped elements are added
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <M extends Map<? super T, ? super S>, T, S> M mapME(@NotNull M target,
                                                                    @NotNull S[] array,
                                                                    @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                                                    @Nullable T deleteMark)
          throws TypeConverterException
  {
    return mapME(target, asList(array), typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an array into a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param target         map where the mapped elements are added
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <M extends Map<? super S, ? super T>, T, S> M mapRE(@NotNull M target,
                                                                    @NotNull S[] array,
                                                                    @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                                                    @Nullable T deleteMark)
          throws TypeConverterException
  {
    return mapRE(target, asList(array), typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an iteration into a collection.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that target and iterable are not allowed to be the same object, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         collection where the mapped elements are added
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   * @deprecated use {@link #mapFragile(Collection, Iterable, FragileFunction1, Object)} instead which
   *             does not require handling a {@link TypeConverterException}
   */
  @NotNull
  @Deprecated
  public static <C extends Collection<? super T>, T, S> C mapE(@NotNull C target,
                                                               @NotNull Iterable<S> iterable,
                                                               @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                                               @Nullable T deleteMark)
          throws TypeConverterException
  {
    if (target == iterable) {
      throw new IllegalArgumentException("'target' and 'iterable' may not be the same object!");
    }
    return mapE(target, iterable.iterator(), typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an iterable into a collection while using a mapper which might throw an exception.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param target         collection where the mapped elements are added
   * @param iterable       iterable providing the source elements
   * @param mapper         fragile mapper for converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @param <E>            exception type
   * @return reference to target, allowing to create it directly in the call
   * @throws E             if the mapper throws
   */
  @NotNull
  public static <C extends Collection<T>, T, S, E extends Exception> C mapFragile(@NotNull C target,
                                                                                  @NotNull Iterable<S> iterable,
                                                                                  @NotNull FragileFunction1<? extends T, E, ? super S> mapper,
                                                                                  @Nullable T deleteMark)
          throws E
  {
    if (target == iterable) {
      throw new IllegalArgumentException("'target' and 'iterable' may not be the same object!");
    }
    return mapFragile(target, iterable.iterator(), mapper, deleteMark);
  }

  /**
   * Add the mapped elements of an iteration into a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that target and iterable are not allowed to be the same object, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (converted) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <M extends Map<? super T, ? super S>, T, S> M mapME(@NotNull M target,
                                                                    @NotNull Iterable<S> iterable,
                                                                    @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                                                    @Nullable T deleteMark)
          throws TypeConverterException
  {
    if (target == iterable) {
      throw new IllegalArgumentException("'target' and 'iterable' may not be the same object!");
    }
    return mapME(target, iterable.iterator(), typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an iteration into a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that target and iterable are not allowed to be the same object, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (converted) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <M extends Map<? super S, ? super T>, T, S> M mapRE(@NotNull M target,
                                                                    @NotNull Iterable<S> iterable,
                                                                    @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                                                    @Nullable T deleteMark)
          throws TypeConverterException
  {
    if (target == iterable) {
      throw new IllegalArgumentException("'target' and 'iterable' may not be the same object!");
    }
    return mapRE(target, iterable.iterator(), typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an iterator into a collection.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         collection where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   * @deprecated use {@link #mapFragile(Collection, Iterator, FragileFunction1, Object)} instead which
   *             does not require handling a {@link TypeConverterException}
   */
  @NotNull
  @Deprecated
  public static <C extends Collection<? super T>, T, S> C mapE(@NotNull C target,
                                                               @NotNull Iterator<S> iterator,
                                                               @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                                               @Nullable T deleteMark)
          throws TypeConverterException
  {
    return mapEImpl(target, iterator, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an iterator into a collection while using a mapper which might throw an exception.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param target         collection where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param mapper         fragile mapper for converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @param <E>            exception type
   * @return reference to target, allowing to create it directly in the call
   * @throws E             if the mapper throws
   */
  @NotNull
  public static <C extends Collection<T>, T, S, E extends Exception> C mapFragile(@NotNull C target,
                                                                                  @NotNull Iterator<S> iterator,
                                                                                  @NotNull FragileFunction1<? extends T, E, ? super S> mapper,
                                                                                  @Nullable T deleteMark)
          throws E
  {
    while (iterator.hasNext()) {
      final T result = mapper.apply(iterator.next());
      if (!Objects.equals(result, deleteMark)) {
        target.add(result);
      }
    }
    return target;
  }

  /**
   * Add the mapped elements of an iterator into a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <M extends Map<? super T, ? super S>, T, S> M mapME(@NotNull M target,
                                                                    @NotNull Iterator<S> iterator,
                                                                    @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                                                    @Nullable T deleteMark)
          throws TypeConverterException
  {
    return mapMEImpl(target, iterator, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an iterator into a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that iterator is not allowed to be an iterator of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <M extends Map<? super S, ? super T>, T, S> M mapRE(@NotNull M target,
                                                                    @NotNull Iterator<S> iterator,
                                                                    @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                                                    @Nullable T deleteMark)
          throws TypeConverterException
  {
    return mapREImpl(target, iterator, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         collection where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   * @deprecated use {@link #mapFragile(Collection, Enumeration, FragileFunction1, Object)} instead which
   *             does not require handling a {@link TypeConverterException}
   */
  @NotNull
  @Deprecated
  public static <C extends Collection<? super T>, T, S> C mapE(@NotNull C target,
                                                               @NotNull Enumeration<S> enumeration,
                                                               @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                                               @Nullable T deleteMark)
          throws TypeConverterException
  {
    return mapEImpl(target, enumeration, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an enumeration into a collection while using a mapper which might throw an exception.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param target         collection where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param mapper         fragile mapper for converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <C>            collection type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @param <E>            exception type
   * @return reference to target, allowing to create it directly in the call
   * @throws E             if the mapper throws
   */
  @NotNull
  public static <C extends Collection<T>, T, S, E extends Exception> C mapFragile(@NotNull C target,
                                                                                  @NotNull Enumeration<S> enumeration,
                                                                                  @NotNull FragileFunction1<? extends T, E, ? super S> mapper,
                                                                                  @Nullable T deleteMark)
          throws E
  {
    while (enumeration.hasMoreElements()) {
      final T result = mapper.apply(enumeration.nextElement());
      if (!Objects.equals(result, deleteMark)) {
        target.add(result);
      }
    }
    return target;
  }

  /**
   * Add the mapped elements of an enumeration to a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <M extends Map<? super T, ? super S>, T, S> M mapME(@NotNull M target,
                                                                    @NotNull Enumeration<S> enumeration,
                                                                    @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                                                    @Nullable T deleteMark)
          throws TypeConverterException
  {
    return mapMEImpl(target, enumeration, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an enumeration to a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This method allows to discard elements during the mapping.
   * <p>
   * Note that enumeration is not allowed to be an enumeration of the target collection, otherwise you'll
   * get either a {@link java.util.ConcurrentModificationException}, or another exception
   * because you are running into an endless loop.
   *
   * @param target         map where the mapped elements are added
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <M>            map type
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return reference to target, allowing to create it directly in the call
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <M extends Map<? super S, ? super T>, T, S> M mapRE(@NotNull M target,
                                                                    @NotNull Enumeration<S> enumeration,
                                                                    @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                                                    @Nullable T deleteMark)
          throws TypeConverterException
  {
    return mapREImpl(target, enumeration, typeConverter, deleteMark);
  }

  /**
   * Get the mapped elements of an array.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   * @deprecated use {@link #mapFragile(Object[], FragileFunction1, Object)} instead which
   *             does not require handling a {@link TypeConverterException}
   */
  @NotNull
  @Deprecated
  public static <T, S> LinkedList<T> mapE(@NotNull S[] array,
                                          @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                          @Nullable T deleteMark)
          throws TypeConverterException
  {
    return mapE(new LinkedList<T>(), asList(array), typeConverter, deleteMark);
  }

  /**
   * Get the mapped elements of an array while using a mapper which might throw an exception.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param array          array providing the source elements
   * @param mapper         fragile mapper for converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @param <E>            exception type
   * @return reference to target, allowing to create it directly in the call
   * @throws E             if the mapper throws
   */
  @NotNull
  public static <T, S, E extends Exception> ArrayList<T> mapFragile(@NotNull S[] array,
                                                                    @NotNull FragileFunction1<? extends T, E, ? super S> mapper,
                                                                    @Nullable T deleteMark)
          throws E
  {
    return mapFragile(new ArrayList<>(array.length), array, mapper, deleteMark);
  }

  /**
   * Get the mapped elements of an array as a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <T, S> HashMap<T, S> mapME(@NotNull S[] array,
                                           @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                           @Nullable T deleteMark)
          throws TypeConverterException
  {
    return mapME(new HashMap<T, S>(), asList(array), typeConverter, deleteMark);
  }

  /**
   * Get the mapped elements of an array as a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param array          array providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <T, S> HashMap<S, T> mapRE(@NotNull S[] array,
                                           @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                           @Nullable T deleteMark)
          throws TypeConverterException
  {
    return mapRE(new HashMap<S, T>(), asList(array), typeConverter, deleteMark);
  }

  /**
   * Get the mapped elements of an iterable.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   * @deprecated use {@link #mapFragile(Iterable, FragileFunction1, Object)} instead which
   *             does not require handling a {@link TypeConverterException}
   */
  @NotNull
  @Deprecated
  public static <T, S> LinkedList<T> mapE(@NotNull Iterable<S> iterable,
                                          @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                          @Nullable T deleteMark)
          throws TypeConverterException
  {
    return mapE(new LinkedList<T>(), iterable, typeConverter, deleteMark);
  }

  /**
   * Get the mapped elements of an iterable while using a mapper which might throw an exception.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param iterable       iterable providing the source elements
   * @param mapper         fragile mapper for converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @param <E>            exception type
   * @return mapped and filtered results
   * @throws E             if the mapper throws
   */
  @NotNull
  public static <T, S, E extends Exception> LinkedList<T> mapFragile(@NotNull Iterable<S> iterable,
                                                                     @NotNull FragileFunction1<? extends T, E, ? super S> mapper,
                                                                     @Nullable T deleteMark)
          throws E
  {
    return mapFragile(new LinkedList<>(), iterable, mapper, deleteMark);
  }

  /**
   * Get the mapped elements of an iterable as a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <T, S> HashMap<T, S> mapME(@NotNull Iterable<S> iterable,
                                           @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                           @Nullable T deleteMark)
          throws TypeConverterException
  {
    return mapME(new HashMap<T, S>(), iterable, typeConverter, deleteMark);
  }

  /**
   * Get the mapped elements of an iterable as a map.
   * <p>
   * The incoming values will become keys, pointing to the associated convereted values.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param iterable       iterable providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <T, S> HashMap<S, T> mapRE(@NotNull Iterable<S> iterable,
                                           @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                           @Nullable T deleteMark)
          throws TypeConverterException
  {
    return mapRE(new HashMap<S, T>(), iterable, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an iterator into a collection.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   * @deprecated use {@link #mapFragile(Iterator, FragileFunction1, Object)} instead which
   *             does not require handling a {@link TypeConverterException}
   */
  @NotNull
  @Deprecated
  public static <T, S> LinkedList<T> mapE(@NotNull Iterator<S> iterator,
                                          @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                          @Nullable T deleteMark)
          throws TypeConverterException
  {
    return mapE(new LinkedList<T>(), iterator, typeConverter, deleteMark);
  }

  /**
   * Get the mapped elements of an iterator while using a mapper which might throw an exception.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param iterator       iterator providing the source elements
   * @param mapper         fragile mapper for converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @param <E>            exception type
   * @return mapped and filtered results
   * @throws E             if the mapper throws
   */
  @NotNull
  public static <T, S, E extends Exception> LinkedList<T> mapFragile(@NotNull Iterator<S> iterator,
                                                                     @NotNull FragileFunction1<? extends T, E, ? super S> mapper,
                                                                     @Nullable T deleteMark)
          throws E
  {
    return mapFragile(new LinkedList<>(), iterator, mapper, deleteMark);
  }

  /**
   * Add the mapped elements of an iterator into a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <T, S> HashMap<T, S> mapME(@NotNull Iterator<S> iterator,
                                           @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                           @Nullable T deleteMark)
          throws TypeConverterException
  {
    return mapME(new HashMap<T, S>(), iterator, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an iterator into a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param iterator       iterator providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static <T, S> HashMap<S, T> mapRE(@NotNull Iterator<S> iterator,
                                           @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                           @Nullable T deleteMark)
          throws TypeConverterException
  {
    return mapRE(new HashMap<S, T>(), iterator, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an enumeration into a collection.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return linked list with mapped elements
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   * @deprecated use {@link #mapFragile(Enumeration, FragileFunction1, Object)} instead which
   *             does not require handling a {@link TypeConverterException}
   */
  @NotNull
  @Deprecated
  public static < T, S> LinkedList<T> mapE(@NotNull Enumeration<S> enumeration,
                                           @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                           @Nullable T deleteMark)
          throws TypeConverterException
  {
    return mapE(new LinkedList<T>(), enumeration, typeConverter, deleteMark);
  }

  /**
   * Get the mapped elements of an enumeration while using a mapper which might throw an exception.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param enumeration    enumeration providing the source elements
   * @param mapper         fragile mapper for converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @param <E>            exception type
   * @return mapped and filtered results
   * @throws E             if the mapper throws
   */
  @NotNull
  public static <T, S, E extends Exception> LinkedList<T> mapFragile(@NotNull Enumeration<S> enumeration,
                                                                     @NotNull FragileFunction1<? extends T, E, ? super S> mapper,
                                                                     @Nullable T deleteMark)
          throws E
  {
    return mapFragile(new LinkedList<>(), enumeration, mapper, deleteMark);
  }

  /**
   * Add the mapped elements of an enumeration to a map.
   * <p>
   * The converted values will become keys, pointing to the associated incoming values.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static < T, S> HashMap<T, S> mapME(@NotNull Enumeration<S> enumeration,
                                            @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                            @Nullable T deleteMark)
          throws TypeConverterException
  {
    return mapME(new HashMap<T, S>(), enumeration, typeConverter, deleteMark);
  }

  /**
   * Add the mapped elements of an enumeration to a map.
   * <p>
   * The incoming values will become keys, pointing to the associated converted values.
   * <p>
   * This method allows to discard elements during the mapping.
   *
   * @param enumeration    enumeration providing the source elements
   * @param typeConverter  type converter converting the source elements into the mapped elements
   * @param deleteMark     special return value of the type converter which means not to copy
   *                       the result into the target collection. It is compared with reference
   *                       equality. Often {@code null} is a useful choice, only in cases
   *                       where the target collection may contain {@code null} values and the
   *                       type converter may produce them another mark is necessary.
   * @param <T>            target (mapped) type
   * @param <S>            source type
   * @return map with mapping results
   * @throws TypeConverterException if the conversion fails with an exception,
   *                                the thrown exception will provide this original exception
   *                                as {@link de.caff.generics.TypeConverterException#getException()}
   */
  @NotNull
  public static < T, S> HashMap<S, T> mapRE(@NotNull Enumeration<S> enumeration,
                                            @NotNull FragileTypeConverter<T, ? super S> typeConverter,
                                            @Nullable T deleteMark)
          throws TypeConverterException
  {
    return mapRE(new HashMap<S, T>(), enumeration, typeConverter, deleteMark);
  }

  /**
   * Call a handler for each item in an array.
   * @param array array to loop over
   * @param handler  item handler which is called for each item
   * @param <T> item type
   */
  public static <T> void forEach(@NotNull T[] array,
                                 @NotNull LoopItemHandler<? super T> handler)
  {
    forEach(asList(array), handler);
  }

  /**
   * Call a handler for each item in an array.
   * @param array array to loop over
   * @param handler  item handler which is called for each item
   * @param <T> item type
   * @param <E> exception type
   * @throws E exception thrown by handler
   */
  public static <T, E extends Exception>
  void forEach(@NotNull T[] array,
               @NotNull FragileLoopItemHandler<? super T, E> handler) throws E
  {
    forEach(asList(array), handler);
  }

  /**
   * Call a handler for each item in an iterable.
   * @param iterable iterable to loop over
   * @param handler  item handler which is called for each item
   * @param <T> item type
   */
  public static <T> void forEach(@NotNull Iterable<T> iterable,
                                 @NotNull LoopItemHandler<? super T> handler)
  {
    forEach(iterable.iterator(), handler);
  }

  /**
   * Call a handler for each item in an iterable.
   * @param iterable iterable to loop over
   * @param handler  item handler which is called for each item
   * @param <T> item type
   * @param <E> exception thrown by handler
   * @throws E exception thrown by handler
   */
  public static <T, E extends Exception>
  void forEach(@NotNull Iterable<T> iterable,
               @NotNull FragileLoopItemHandler<? super T, E> handler) throws E
  {
    forEach(iterable.iterator(), handler);
  }

  /**
   * Call a handler for each item in an iterator.
   * @param iterator iterator to loop over
   * @param handler  item handler which is called for each item
   * @param <T> item type
   */
  public static <T> void forEach(@NotNull Iterator<T> iterator,
                                 @NotNull LoopItemHandler<? super T> handler)
  {
    handler.beforeStart();
    try {
      while (iterator.hasNext()) {
        handler.handle(iterator.next());
      }
    } finally {
      handler.afterFinish();
    }
  }

  /**
   * Call a handler for each item in an iterator
   * @param iterator iterator to loop over
   * @param handler  item handler which is called for each item
   * @param <T> item type
   * @param <E> exception thrown by handler
   * @throws E exception thrown by handler
   */
  public static <T, E extends Exception>
  void forEach(@NotNull Iterator<T> iterator,
               @NotNull FragileLoopItemHandler<? super T, E> handler) throws E
  {
    handler.beforeStart();
    try {
      while (iterator.hasNext()) {
        handler.handle(iterator.next());
      }
    } finally {
      handler.afterFinish();
    }
  }

  /**
   * Call a handler for each item in an enumeration.
   * @param enumeration enumeration to loop over
   * @param handler  item handler which is called for each item
   * @param <T> item type
   */
  public static <T> void forEach(@NotNull Enumeration<T> enumeration,
                                 @NotNull LoopItemHandler<? super T> handler)
  {
    forEach(toIterator(enumeration), handler);
  }

  /**
   * Call a handler for each item in an enumeration
   * @param enumeration enumeration to loop over
   * @param handler  item handler which is called for each item
   * @param <T> item type
   * @param <E> exception thrown by handler
   * @throws E exception thrown by handler
   */
  public static <T, E extends Exception>
  void forEach(@NotNull Enumeration<T> enumeration,
               @NotNull FragileLoopItemHandler<? super T, E> handler) throws E
  {
    forEach(toIterator(enumeration), handler);
  }

  /**
   * Foreach implementation which can be used with consumers which might throw a checked exception.
   * Use this instead of {@link Iterable#forEach(Consumer)} when the consumer is defined with
   * a checked exception.
   * @param iterable iterable to iterate over
   * @param consumer consumer called for each iterated element
   * @param <T> element type
   * @param <E> exception type
   * @throws E forwarded exception of the consumer
   */
  public static <T, E extends Exception>
  void forEachFragile(@NotNull Iterable<T> iterable,
                      @NotNull FragileProcedure1<E, ? super T> consumer)
          throws E
  {
    for (T elem : iterable) {
      consumer.apply(elem);
    }
  }

  /**
   * Foreach remaining implementation which can be used with consumers which might throw a checked exception.
   * Use this instead of {@link Iterator#forEachRemaining(Consumer)} when the consumer is defined with
   * a checked exception.
   * @param iterator iterator for which all remaining elements are consumed
   * @param consumer consumer called for each remaining element
   * @param <T> element type
   * @param <E> exception type
   * @throws E forwarded exception of the consumer
   */
  public static <T, E extends Exception>
  void forEachRemainingFragile(@NotNull Iterator<T> iterator,
                               @NotNull FragileProcedure1<E, ? super T> consumer)
          throws E
  {
    while (iterator.hasNext()) {
      consumer.apply(iterator.next());
    }
  }

  /**
   * Wrap a careful loop item handler as a simple loop item handler.
   * @param handler handler to wrap
   * @param <T> item type
   * @param <E> exception type
   * @param <L> careful loop item handler type
   * @return careless loop item handler wrapper
   */
  @NotNull
  public static <T, E extends Exception, L extends FragileLoopItemHandler<T, E>> CarelessLoopItemHandler<T, E, L> toLoopItemHandler(L handler)
  {
    return new CarelessLoopItemHandler<>(handler);
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list returned from this method,
   *  so the safest use is as a temporary object.
   *
   * <blockquote><pre>
   *    public void method(Collection&lt;Double&gt; input) {
   *      // ...
   *    }
   *
   *    // ...
   *    method(Types.asList(0, 7.1, -42, Math.PI));
   * </pre></blockquote>
   *  @param array the array to wrap into a list view
   *  @param <T> element type of array and list
   *  @return unmodifiable list accessing the array
   */
  @NotNull
  @SafeVarargs
  @SuppressWarnings("varargs")
  public static <T> List<T> asList(@NotNull T ... array)
  {
    return new UnmodifiableListArrayWrapper<>(array);
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method,
   *  so the safest use is as a temporary object.
   * <p>
   *  See {@link #asList(Object[])} for an easy to adapt example.
   *
   *  @param array the array to wrap into a list view
   *  @param length the number of elements to be used from the array
   *  @param <T> element type of array and list
   *  @return unmodifiable list accessing the array
   */
  @NotNull
  public static <T> List<T> asList(@NotNull T[] array, int length)
  {
    return new UnmodifiableListArrayWrapper<>(array, length);
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method.
   * <p>
   *  See {@link #asList(Object[])} for an easy to adapt example.
   *
   *  @param array the array to wrap into a list view
   *  @param start index of first used element of the array
   *  @param length the number of elements to be used from the array
   *  @param <T> element type of array and list
   *  @return unmodifiable list accessing the array
   */
  @NotNull
  public static <T> List<T> asList(@NotNull T[] array, int start, int length)
  {
    return new UnmodifiableListArrayWrapper<>(array, start, length);
  }

  /**
   * Convert a collection of {@link Boolean}s to a raw array of booleans.
   * @param c collection of Booleans
   * @return array of boolean
   * @see BooleanIndexable#viewList(List)
   */
  @NotNull
  public static boolean[] toBooleanArray(@NotNull Collection<Boolean> c)
  {
    return toBooleanArray(Countable.viewCollection(c));
  }

  /**
   * Convert a countable of {@link Boolean}s to a raw array of booleans.
   * @param c collection of Booleans
   * @return array of boolean
   */
  public static boolean[] toBooleanArray(@NotNull Countable<Boolean> c)
  {
    boolean[] array = new boolean[c.size()];
    int i = 0;
    for (Boolean v : c) {
      array[i++] = v;
    }
    return array;
  }

  /**
   * Collect an enumeration into a linked list.
   * This will propagate over the complete enumeration,
   * so it can no longer be enumerated after calling this method.
   * @param enumeration enumeration to convert
   * @param <T> type of enumeration and result list elements
   * @return linked list with all elements of the enumeration
   * @see #addAll(Collection, Enumeration)
   */
  @NotNull
  public static <T> LinkedList<T> toList(@NotNull Enumeration<T> enumeration)
  {
    return addAll(new LinkedList<T>(), enumeration);
  }

  /**
   * Collect an iterator into a linked list.
   * This will propagate over the complete iterator,
   * so it can no longer be iterated over after calling this method.
   * @param iterator iterator to convert
   * @param <T> type of iterator and result list elements
   * @return linked list with all elements of the iterator
   * @see #addAll(Collection, Iterator)
   */
  @NotNull
  public static <T> LinkedList<T> toList(@NotNull Iterator<T> iterator)
  {
    return addAll(new LinkedList<T>(), iterator);
  }

  /**
   * Collect the elements of an iterable into a linked list.
   * @param iterable iterable to convert
   * @param <T> type of iterable and result list elements
   * @return list with all elements of the iterable
   * @see #addAll(Collection, Iterable)
   */
  @NotNull
  public static <T> LinkedList<T> toList(@NotNull Iterable<T> iterable)
  {
    return addAll(new LinkedList<T>(), iterable);
  }

  /**
   * Collect an enumeration into a linked list of strings.
   * This will propagate over the complete enumeration and call @link{@link #toString()},
   * on its elements, so it can no longer be enumerated after calling this method.
   * @param enumeration enumeration to convert
   * @return linked list with all stringified elements of the enumeration
   */
  @NotNull
  public static LinkedList<String> toStringList(@NotNull Enumeration<?> enumeration)
  {
    return map(enumeration, TO_STRING);
  }

  /**
   * Collect an iterator into a linked list of strings.
   * This will propagate over the complete iterator and call @link{@link #toString()},
   * on its elements, so it can no longer be iterated over after calling this method.
   * @param iterator iterator to convert
   * @return linked list with all stringified elements of the iterator
   */
  @NotNull
  public static LinkedList<String> toStringList(@NotNull Iterator<?> iterator)
  {
    return map(iterator, TO_STRING);
  }

  /**
   * Collect the elements of an iterable into a linked list.
   * This will propagate over the complete iterable and
   * call @link{@link #toString()} on its elements
   * @param iterable iterable to convert
   * @return list with all stringified elements of the iterable
   */
  @NotNull
  public static LinkedList<String> toStringList(@NotNull Iterable<?> iterable)
  {
    return map(iterable, TO_STRING);
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method,
   *  so the safest use is as a temporary object.
   *
   *  <blockquote><pre>
   *    public void method(Collection&lt;Double&gt; input) {
   *      // ...
   *    }
   *
   *    // ...
   *    method(Types.toList(0, 7.1, -42, Math.PI));
   *  </pre></blockquote>
   *  @param array the array to wrap into a list view
   *  @return unmodifiable list accessing the array
   */
  @NotNull
  public static List<Boolean> toList(@NotNull boolean ... array)
  {
    return new UnmodifiableListBooleanArrayWrapper(array);
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method,
   *  so the safest use is as a temporary object.
   * <p>
   *  See {@link #asList(Object[])} for an easy to adapt example.
   *
   *  @param array the array to wrap into a list view
   *  @param length the number of elements to be used from the array
   *  @return unmodifiable list accessing the array
   */
  @NotNull
  public static List<Boolean> toList(@NotNull boolean[] array, int length)
  {
    return new UnmodifiableListBooleanArrayWrapper(array, length);
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method.
   * <p>
   *  See {@link #asList(Object[])} for an easy to adapt example.
   *
   *  @param array the array to wrap into a list view
   *  @param start index of first used element of the array
   *  @param length the number of elements to be used from the array
   *  @return unmodifiable list accessing the array
   */
  @NotNull
  public static List<Boolean> toList(@NotNull boolean[] array, int start, int length)
  {
    return new UnmodifiableListBooleanArrayWrapper(array, start, length);
  }

  /**
   * Convert a collection of {@link Number}s to a raw array of bytes.
   * @param c collection of Numbers
   * @return array of bytes
   * @see ByteIndexable#viewList(List)
   */
  @NotNull
  public static byte[] toByteArray(@NotNull Collection<? extends Number> c)
  {
    return toByteArray(Countable.viewCollection(c));
  }

  /**
   * Convert a collection of {@link Number}s to a raw array of bytes.
   * @param c collection of Numbers
   * @return array of bytes
   */
  @NotNull
  public static byte[] toByteArray(@NotNull Countable<? extends Number> c)
  {
    if (c.isEmpty()) {
      return Empty.BYTE_ARRAY;
    }
    final byte[] array = new byte[c.size()];
    int i = 0;
    for (Number v : c) {
      array[i++] = v.byteValue();
    }
    return array;
  }

  /**
   * Convert a number of bytes into a byte array.
   * This is a convenience method allowing to use integer values.
   * @param bytes values which will be converted into bytes
   * @return byte array
   */
  public static byte[] toByteArray(int ... bytes)
  {
    if (bytes.length == 0) {
      return Empty.BYTE_ARRAY;
    }
    final byte[] array = new byte[bytes.length];
    for (int i = bytes.length - EMPTY_HASH; i >= 0; --i) {
      array[i] = (byte)bytes[i];
    }
    return array;
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method,
   *  so the safest use is as a temporary object.
   *
   *  @param array the array to wrap into a list view
   *  @return unmodifiable list accessing the array
   */
  @NotNull
  public static List<Byte> toList(@NotNull byte ... array)
  {
    return new UnmodifiableListByteArrayWrapper(array);
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method,
   *  so the safest use is as a temporary object.
   * <p>
   *  See {@link #asList(Object[])} for an easy to adapt example.
   *
   *  @param array the array to wrap into a list view
   *  @param length the number of elements to be used from the array
   *  @return unmodifiable list accessing the array
   */
  @NotNull
  public static List<Byte> toList(@NotNull byte[] array, int length)
  {
    return new UnmodifiableListByteArrayWrapper(array, length);
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method.
   * <p>
   *  See {@link #asList(Object[])} for an easy to adapt example.
   *
   *  @param array the array to wrap into a list view
   *  @param start index of first used element of the array
   *  @param length the number of elements to be used from the array
   *  @return unmodifiable list accessing the array
   */
  @NotNull
  public static List<Byte> toList(@NotNull byte[] array, int start, int length)
  {
    return new UnmodifiableListByteArrayWrapper(array, start, length);
  }

  /**
   * Convert a collection of {@link Number}s to a raw array of shorts.
   * @param c collection of Numbers
   * @return array of shorts
   * @see ShortIndexable#viewList(List)
   */
  @NotNull
  public static short[] toShortArray(@NotNull Collection<? extends Number> c)
  {
    return toShortArray(Countable.viewCollection(c));
  }

  /**
   * Convert a countable of {@link Number}s to a raw array of shorts.
   * @param c countable of Numbers
   * @return array of shorts
   */
  public static short[] toShortArray(@NotNull Countable<? extends Number> c)
  {
    if (c.isEmpty()) {
      return Empty.SHORT_ARRAY;
    }
    final short[] array = new short[c.size()];
    int i = 0;
    for (Number v : c) {
      array[i++] = v.shortValue();
    }
    return array;
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method,
   *  so the safest use is as a temporary object.
   *
   *  @param array the array to wrap into a list view
   *  @return unmodifiable list accessing the array
   * @see ShortIndexable#viewArray(short...)
   */
  @NotNull
  public static List<Short> toList(@NotNull short ... array)
  {
    return new UnmodifiableListShortArrayWrapper(array);
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method,
   *  so the safest use is as a temporary object.
   * <p>
   *  See {@link #asList(Object[])} for an easy to adapt example.
   *
   *  @param array the array to wrap into a list view
   *  @param length the number of elements to be used from the array
   *  @return unmodifiable list accessing the array
   */
  @NotNull
  public static List<Short> toList(@NotNull short[] array, int length)
  {
    return new UnmodifiableListShortArrayWrapper(array, length);
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method.
   * <p>
   *  See {@link #asList(Object[])} for an easy to adapt example.
   *
   *  @param array the array to wrap into a list view
   *  @param start index of first used element of the array
   *  @param length the number of elements to be used from the array
   *  @return unmodifiable list accessing the array
   * @see ShortIndexable#viewArray(short[], int, int)
   */
  @NotNull
  public static List<Short> toList(@NotNull short[] array, int start, int length)
  {
    return new UnmodifiableListShortArrayWrapper(array, start, length);
  }

  /**
   * Convert a collection of {@link Number}s to a raw array of ints.
   * @param c collection of Numbers
   * @return array of ints
   * @see IntIndexable#viewList(List)
   */
  public static int[] toIntArray(@NotNull Collection<? extends Number> c)
  {
    return toIntArray(Countable.viewCollection(c));
  }

  /**
   * Convert a countable of {@link Number}s to a raw array of ints.
   * @param c countable Numbers
   * @return array of ints
   */
  public static int[] toIntArray(@NotNull Countable<? extends Number> c)
  {
    if (c.isEmpty()) {
      return Empty.INT_ARRAY;
    }
    final int[] array = new int[c.size()];
    int i = 0;
    for (Number v : c) {
      array[i++] = v.intValue();
    }
    return array;
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method,
   *  so the safest use is as a temporary object.
   *
   *  <blockquote><pre>
   *    public void method(Collection&lt;Integer&gt; input) {
   *      // ...
   *    }
   *
   *    // ...
   *    method(Types.asList(0, 7, -42, 3));
   *  </pre></blockquote>
   *  @param array the array to wrap into a list view
   *  @return unmodifiable list accessing the array
   * @see IntIndexable#viewArray(int...)
   */
  @NotNull
  public static List<Integer> toList(@NotNull int ... array)
  {
    return new UnmodifiableListIntArrayWrapper(array);
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method,
   *  so the safest use is as a temporary object.
   * <p>
   *  See {@link #asList(Object[])} for an easy to adapt example.
   *
   *  @param array the array to wrap into a list view
   *  @param length the number of elements to be used from the array
   *  @return unmodifiable list accessing the array
   */
  @NotNull
  public static List<Integer> toList(@NotNull int[] array, int length)
  {
    return new UnmodifiableListIntArrayWrapper(array, length);
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method.
   * <p>
   *  See {@link #asList(Object[])} for an easy to adapt example.
   *
   *  @param array the array to wrap into a list view
   *  @param start index of first used element of the array
   *  @param length the number of elements to be used from the array
   *  @return unmodifiable list accessing the array
   * @see IntIndexable#viewArray(int[], int, int)
   */
  @NotNull
  public static List<Integer> toList(@NotNull int[] array, int start, int length)
  {
    return new UnmodifiableListIntArrayWrapper(array, start, length);
  }

  /**
   * Convert a collection of {@link Number}s to a raw array of long ints.
   * @param c collection of Numbers
   * @return array of long ints
   */
  public static long[] toLongArray(@NotNull Collection<? extends Number> c)
  {
    return toLongArray(Countable.viewCollection(c));
  }

  /**
   * Convert a countable of {@link Number}s to a raw array of long ints.
   * @param c countable Numbers
   * @return array of long ints
   */
  public static long[] toLongArray(@NotNull Countable<? extends Number> c)
  {
    if (c.isEmpty()) {
      return Empty.LONG_ARRAY;
    }
    final long[] array = new long[c.size()];
    int i = 0;
    for (Number v : c) {
      array[i++] = v.longValue();
    }
    return array;
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method,
   *  so the safest use is as a temporary object.
   *
   *  <blockquote><pre>
   *    public void method(Collection(&lt;Long&gt; input) {
   *      // ...
   *    }
   *
   *    // ...
   *    method(Types.toList(0L, 7L, -42L, 3L));
   *  </pre></blockquote>
   *  @param array the array to wrap into a list view
   *  @return unmodifiable list accessing the array
   * @see LongIndexable#viewArray(long...)
   */
  @NotNull
  public static List<Long> toList(@NotNull long ... array)
  {
    return new UnmodifiableListLongArrayWrapper(array);
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method,
   *  so the safest use is as a temporary object.
   * <p>
   *  See {@link #asList(Object[])} for an easy to adapt example.
   *
   *  @param array the array to wrap into a list view
   *  @param length the number of elements to be used from the array
   *  @return unmodifiable list accessing the array
   */
  @NotNull
  public static List<Long> toList(@NotNull long[] array, int length)
  {
    return new UnmodifiableListLongArrayWrapper(array, length);
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method.
   * <p>
   *  See {@link #asList(Object[])} for an easy to adapt example.
   *
   *  @param array the array to wrap into a list view
   *  @param start index of first used element of the array
   *  @param length the number of elements to be used from the array
   *  @return unmodifiable list accessing the array
   * @see LongIndexable#viewArray(long[], int, int)
   */
  @NotNull
  public static List<Long> toList(@NotNull long[] array, int start, int length)
  {
    return new UnmodifiableListLongArrayWrapper(array, start, length);
  }

  /**
   * Convert a collection of {@link Number}s to a raw array of doubles.
   * @param c collection of Numbers
   * @return array of doubles
   */
  @NotNull
  public static double[] toDoubleArray(@NotNull Collection<? extends Number> c)
  {
    return toDoubleArray(Countable.viewCollection(c));
  }

  /**
   * Convert a countable of {@link Number}s to a raw array of doubles.
   * @param c countable Numbers
   * @return array of doubles
   */
  @NotNull
  public static double[] toDoubleArray(@NotNull Countable<? extends Number> c)
  {
    if (c.isEmpty()) {
      return Empty.DOUBLE_ARRAY;
    }
    final double[] array = new double[c.size()];
    int i = 0;
    for (Number v : c) {
      array[i++] = v.doubleValue();
    }
    return array;
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method,
   *  so the safest use is as a temporary object.
   *
   *  <blockquote><pre>
   *    public void method(Collection&lt;Double&gt; input) {
   *      // ...
   *    }
   *
   *    // ...
   *    method(Types.toList(0.0, 7.1, -42.0, Math.PI));
   *  </pre></blockquote>
   *  @param array the array to wrap into a list view
   *  @return unmodifiable list accessing the array
   *  @see DoubleIndexable#viewArray(double...)
   */
  @NotNull
  public static List<Double> toList(@NotNull double ... array)
  {
    return new UnmodifiableListDoubleArrayWrapper(array);
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method,
   *  so the safest use is as a temporary object.
   * <p>
   *  See {@link #asList(Object[])} for an easy to adapt example.
   *
   *  @param array the array to wrap into a list view
   *  @param length the number of elements to be used from the array
   *  @return unmodifiable list accessing the array
   */
  @NotNull
  public static List<Double> toList(@NotNull double[] array, int length)
  {
    return new UnmodifiableListDoubleArrayWrapper(array, length);
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method.
   * <p>
   *  See {@link #asList(Object[])} for an easy to adapt example.
   *
   *  @param array the array to wrap into a list view
   *  @param start index of first used element of the array
   *  @param length the number of elements to be used from the array
   *  @return unmodifiable list accessing the array
   * @see DoubleIndexable#viewArray(double[], int, int)
   */
  @NotNull
  public static List<Double> toList(@NotNull double[] array, int start, int length)
  {
    return new UnmodifiableListDoubleArrayWrapper(array, start, length);
  }

  /**
   * Convert a collection of {@link Number}s to a raw array of floats.
   * @param c collection of Numbers
   * @return array of floats
   */
  @NotNull
  public static float[] toFloatArray(@NotNull Collection<? extends Number> c)
  {
    return toFloatArray(Countable.viewCollection(c));
  }

  /**
   * Convert a countable of {@link Number}s to a raw array of floats.
   * @param c countable Numbers
   * @return array of floats
   */
  @NotNull
  public static float[] toFloatArray(@NotNull Countable<? extends Number> c)
  {
    if (c.isEmpty()) {
      return Empty.FLOAT_ARRAY;
    }
    final float[] array = new float[c.size()];
    int i = 0;
    for (Number v : c) {
      array[i++] = v.floatValue();
    }
    return array;
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method,
   *  so the safest use is as a temporary object.
   *
   *  @param array the array to wrap into a list view
   *  @return unmodifiable list accessing the array
   * @see FloatIndexable#viewArray(float...)
   */
  @NotNull
  public static List<Float> toList(@NotNull float ... array)
  {
    return new UnmodifiableListFloatArrayWrapper(array);
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method,
   *  so the safest use is as a temporary object.
   * <p>
   *  See {@link #asList(Object[])} for an easy to adapt example.
   *
   *  @param array the array to wrap into a list view
   *  @param length the number of elements to be used from the array
   *  @return unmodifiable list accessing the array
   */
  @NotNull
  public static List<Float> toList(@NotNull float[] array, int length)
  {
    return new UnmodifiableListFloatArrayWrapper(array, length);
  }

  /**
   *  A more effective implementation of {@code Arrays.asList()},
   *  useful if only an unmodifiable list is needed.
   * <p>
   *  Changes of the array will be reflected in the list return from this method.
   * <p>
   *  See {@link #asList(Object[])} for an easy to adapt example.
   *
   *  @param array the array to wrap into a list view
   *  @param start index of first used element of the array
   *  @param length the number of elements to be used from the array
   *  @return unmodifiable list accessing the array
   * @see FloatIndexable#viewArray(float[], int, int)
   */
  @NotNull
  public static List<Float> toList(@NotNull float[] array, int start, int length)
  {
    return new UnmodifiableListFloatArrayWrapper(array, start, length);
  }

  /**
   * Convert some items into an enumeration.
   * @param item items
   * @param <T> enumeration type
   * @return enumeration
   */
  @NotNull
  @SafeVarargs
  @SuppressWarnings("varargs")
  public static <T> Enumeration<T> toEnumeration(@NotNull T ... item)
  {
    return toEnumeration(asList(item));
  }

  /**
   * Convert an iterable into an enumeration.
   * @param iterable the iterable
   * @param <T> iterator type
   * @return enumeration
   */
  @NotNull
  public static <T> Enumeration<T> toEnumeration(@NotNull Iterable<T> iterable)
  {
    return new IteratorEnumeration<>(iterable.iterator());
  }

  /**
   * Convert an iterator into an enumeration.
   * @param iterator the iterator
   * @param <T> iterator type
   * @return enumeration
   */
  @NotNull
  public static <T> Enumeration<T> toEnumeration(@NotNull Iterator<T> iterator)
  {
    return new IteratorEnumeration<>(iterator);
  }

  /**
   * Convert an enumeration into an iterator.
   * @param enumeration the enumeration
   * @param <T> enumeration type
   * @return iterator
   */
  @NotNull
  public static <T> Iterator<T> toIterator(@NotNull Enumeration<T> enumeration)
  {
    return new EnumerationIterator<>(enumeration);
  }

  /**
   * Convert some items into an iterator.
   * @param item items
   * @param <T> item type
   * @return iterator
   */
  @NotNull
  @SafeVarargs
  @SuppressWarnings("varargs")
  public static <T> Iterator<T> toIterator(@NotNull T ... item)
  {
    return asList(item).iterator();
  }

  /**
   * Get a one shot iterable from an enumeration.
   * The preferred usage of this is to wrap the enumeration so it can be used in a for each loop.
   *
   * @param enumeration enumeration to wrap
   * @param <T> item type
   * @return iterable which may only be used once, eg in a for each loop
   */
  @NotNull
  public static <T> Iterable<T> toIterable(@NotNull final Enumeration<T> enumeration)
  {
    return () -> toIterator(enumeration);
  }

  /**
   * Test whether a collection contains an element which is equal to another element.
   * The standard Collection {@code contains()} method is sometimes erroneously checking for reference equality,
   * while this method uses the {@code equals()} method.
   * As this method has to use an iterator its efficiency is determined by the size of the
   * collection and the time for equality checking of the element.
   * @param collection collection
   * @param element    element which is looked up
   * @param <T>        generic parameter
   * @return {@code true} if an element which is equal to the given one is contained in the collection<br>
   *         {@code false} otherwise
   */
  public static <T> boolean containsEquals(@NotNull Iterable<T> collection, @NotNull T element)
  {
    for (T elem : collection) {
      if (element.equals(elem)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Find the first element which fulfills the given check.
   * @param iterable iterable with possibly matching elements
   * @param check    check to perform
   * @param <T>      generic parameter
   * @return first matching element or {@code null} if no matching element was found
   */
  @Nullable
  public static <T> T findFirst(@NotNull Iterable<T> iterable, @NotNull Predicate<? super T> check)
  {
    return findNext(iterable.iterator(), check);
  }

  /**
   * Find the nect element which fulfils the given check.
   * @param iterator iterator which is advanced until the given element is found
   * @param check    check to perform
   * @param <T>      generic parameter
   * @return first matching element or {@code null} if no matching element was found
   */
  @Nullable
  public static <T> T findNext(@NotNull Iterator<T> iterator, @NotNull Predicate<? super T> check)
  {
    while (iterator.hasNext()) {
      final T elem = iterator.next();
      if (check.test(elem)) {
        return elem;
      }
    }
    return null;
  }

  /**
   * Find the nect element which fulfils the given check.
   * @param enumeration enumeration which is advanced until the given element is found
   * @param check    check to perform
   * @param <T>      generic parameter
   * @return first matching element or {@code null} if no matching element was found
   */
  @Nullable
  public static <T> T findNext(@NotNull Enumeration<T> enumeration, @NotNull Predicate<? super T> check)
  {
    while (enumeration.hasMoreElements()) {
      final T elem = enumeration.nextElement();
      if (check.test(elem)) {
        return elem;
      }
    }
    return null;
  }

  /**
   * Find the first element in an collection which is equal to another element.
   * The standard Collection {@code indexOf()} method is sometimes erroneously checking for reference equality,
   * while this method uses the {@code equals()} method.
   * As this method has to use an iterator its efficiency is determined by the size of the
   * collection and the time for equality checking of the element.
   * @param collection collection
   * @param element    element which is looked up
   * @param <T>        generic parameter
   * @return {@code true} if an element which is equal to the given one is contained in the collection<br>
   *         {@code false} otherwise
   */
  public static <T> int findFirstEqual(@NotNull Iterable<T> collection, @NotNull T element)
  {
    int i = 0;
    for (T elem : collection) {
      if (element.equals(elem)) {
        return i;
      }
      ++i;
    }
    return -EMPTY_HASH;
  }

  /**
   * Find the first element in a collection which is matching another element.
   * As this method has to use an iterator its efficiency is determined by the size of the
   * collection and the time for match checking of the element.
   * @param collection collection
   * @param element    element which is looked up
   * @param matcher    matcher which matches the elements in the iteration to the given element
   * @param <T>        generic parameter for iterator
   * @param <E>        generic parameter for looked up element
   * @return {@code true} if an element which is equal to the given one is contained in the collection<br>
   *         {@code false} otherwise
   */
  public static <T, E> int findFirstMatch(@NotNull Iterable<T> collection,
                                          @NotNull E element,
                                          @NotNull BiPredicate<? super T, ? super E> matcher)
  {
    int i = 0;
    for (T elem : collection) {
      if (matcher.test(elem, element)) {
        return i;
      }
      ++i;
    }
    return -1;
  }

  /**
   * Add all elements to a collection.
   * @param collection collection to add to, has to be modifiable
   * @param elements   elements to add
   * @param <C> collection type
   * @param <T> element type
   * @return the collection, allowing to create it on the fly when calling this method
   * @see Collections#addAll(Collection, Object[])
   */
  @NotNull
  @SafeVarargs
  @SuppressWarnings("varargs")
  public static <C extends Collection<? super T>, T> C addAll(@NotNull C collection,
                                                              T ... elements)
  {
    Collections.addAll(collection, elements);
    return collection;
  }

  /**
   * Add all elements of an iterable to a collection.
   * @param collection collection to add to, has to be modifiable
   * @param iterable   iterable to add
   * @param <C> collection type
   * @param <T> collection/iterator type
   * @return the collection, allowing to create it on the fly when calling this method
   */
  @NotNull
  public static <C extends Collection<T>, T> C addAll(@NotNull C collection,
                                                      @NotNull Iterable<? extends T> iterable)
  {
    return addAll(collection, iterable.iterator());
  }

  /**
   * Add an iterable to a collection.
   * @param collection collection to add to, has to be modifiable
   * @param iterable   iterable to add
   * @param <C> collection type
   * @param <T> collection/iterator type
   * @return the collection, allowing to create it on the fly when calling this method
   * @deprecated Use {@link #addAll(Collection, Iterable)} instead
   */
  @Deprecated
  @NotNull
  public static <C extends Collection<T>, T> C addIterable(@NotNull C collection,
                                                           @NotNull Iterable<? extends T> iterable)
  {
    return addAll(collection, iterable);
  }

  /**
   * Add an iterator to a collection.
   * This will propagate over the complete iteration,
   * so it is after it's end after calling this method.
   * @param collection collection to add to, has to be modifiable
   * @param iterator   iterator to add
   * @param <C> collection type
   * @param <T> collection/iterator type
   * @return the collection, allowing to create it on the fly when calling this method
   * @deprecated Use {@link #addAll(Collection, Iterator)} instead
   */
  @NotNull
  @Deprecated
  public static <C extends Collection<T>, T> C addIterator(@NotNull C collection,
                                                           @NotNull Iterator<? extends T> iterator)
  {
    return addAll(collection, iterator);
  }

  /**
   * Add all elements of an iterator to a collection.
   * This will propagate over the complete iteration,
   * so it is after it's end after calling this method.
   * @param collection collection to add to, has to be modifiable
   * @param iterator   iterator to add
   * @param <C> collection type
   * @param <T> collection/iterator type
   * @return the collection, allowing to create it on the fly when calling this method
   */
  @NotNull
  public static <C extends Collection<T>, T> C addAll(@NotNull C collection,
                                                      @NotNull Iterator<? extends T> iterator)
  {
    while (iterator.hasNext()) {
      collection.add(iterator.next());
    }
    return collection;
  }

  /**
   * Add an enumeration to a collection.
   * This will propagate over the complete enumeration,
   * so it can no longer be enumerated after calling this method.
   * @param collection collection to add to, has to be modifiable
   * @param enumeration  enumeration to add
   * @param <C> collection type
   * @param <T> collection/iterator type
   * @return the collection, allowing to create it on the fly when calling this method
   * @deprecated Use {@link #addAll(Collection, Enumeration)} instead
   */
  @NotNull
  @Deprecated
  public static <C extends Collection<T>, T> C addEnumeration(@NotNull C collection,
                                                              @NotNull Enumeration<? extends T> enumeration)
  {
    while (enumeration.hasMoreElements()) {
      collection.add(enumeration.nextElement());
    }
    return collection;
  }

  /**
   * Add all elements of an enumeration to a collection.
   * This will propagate over the complete enumeration,
   * so it can no longer be enumerated after calling this method.
   * @param collection collection to add to, has to be modifiable
   * @param enumeration  enumeration to add
   * @param <C> collection type
   * @param <T> collection/iterator type
   * @return the collection, allowing to create it on the fly when calling this method
   */
  @NotNull
  public static <C extends Collection<T>, T> C addAll(@NotNull C collection,
                                                      @NotNull Enumeration<? extends T> enumeration)
  {
    while (enumeration.hasMoreElements()) {
      collection.add(enumeration.nextElement());
    }
    return collection;
  }

  /**
   * Create a list from an iterable.
   * This copies all elements of the iterable to a linked list.
   * @param iterable iterable
   * @param <T>      iterable/list type
   * @return linked list containing the elements of the iterable
   */
  @NotNull
  public static <T> LinkedList<T> makeList(@NotNull Iterable<T> iterable)
  {
    return addAll(new LinkedList<>(), iterable.iterator());
  }

  /**
   * Create a list from an iterator.
   * This copies all elements of the iterator to a linked list.
   * @param iterator iterator
   * @param <T>      iterator/list type
   * @return linked list containing the elements of the iterator
   */
  @NotNull
  public static <T> LinkedList<T> makeList(@NotNull Iterator<T> iterator)
  {
    return addAll(new LinkedList<>(), iterator);
  }

  /**
   * Create a list from an enumeration.
   * This copies all elements of the enumeration to a linked list.
   * @param enumeration enumeration
   * @param <T>      enumeration/list type
   * @return linked list containing the elements of the enumeration
   */
  @NotNull
  public static <T> LinkedList<T> makeList(@NotNull Enumeration<T> enumeration)
  {
    return addAll(new LinkedList<>(), enumeration);
  }

  /**
   * Get all entries of a map as a list of ordered pairs.
   * The map keys become the first pair value, the map values the second pair value.
   *
   * @param map map to convert
   * @param <K> map key / first pair value type
   * @param <V> map value / second pair value type
   * @return array list with ordered pair values
   */
  @NotNull
  public static <K, V> ArrayList<OrderedPair<K, V>> toPairs(@NotNull Map<K, V> map)
  {
    return toPairs(new ArrayList<>(map.size()),
                   map);
  }

  /**
   * Get all entries of a map as a collection of ordered pairs.
   * The map keys become the first pair value, the map values the second pair value.
   *
   * @param target collection where the pairs are added
   * @param map map to convert
   * @param <K> map key / first pair value type
   * @param <V> map value / second pair value type
   * @param <C> collection type
   * @return the target collection
   */
  @NotNull
  public static <K, V, C extends Collection<OrderedPair<K, V>>> C toPairs(@NotNull C target,
                                                                          @NotNull Map<K, V> map)
  {
    return map(target, map.entrySet(), OrderedPair::create);
  }

  /**
   * Get a synchronized collection from a basic collection which is optimized for being often iterated, but rarely changed.
   * <p>
   * To make the synchronization work it is adamant that the basic collection is never accessed directly, only via
   * the returned collection. Different from {@link java.util.Collections#synchronizedCollection(java.util.Collection)}
   * special handling for iterations is not necessary.
   *
   * @param collection basic collection
   * @param <T> element type
   * @return synchronized collection
   * @see SynchronizedCollection
   */
  @NotNull
  public static <T> Collection<T> synchronizedCollection(@NotNull Collection<T> collection)
  {
    return new SynchronizedCollection<>(collection);
  }

  /**
   * Get a synchronized list from a basic collection which is optimized for being often iterated, but rarely changed.
   * <p>
   * To make the synchronization work it is adamant that the basic list is never accessed directly, only via
   * the returned list. Different from {@link java.util.Collections#synchronizedList(List)}
   * special handling for iterations is not necessary.
   *
   * @param list basic list
   * @param <T> element type
   * @return synchronized list
   * @see SynchronizedList
   */
  public static <T>  List<T> synchronizedList(@NotNull List<T> list)
  {
    return new SynchronizedList<>(list);
  }

  /**
   * Extract all elements of a given type.
   * <p>
   * Internally this is using reflection, namely the {@link Class#isInstance(Object)} method,
   * which is not considered fast. If you need this method often, rethink your design.
   * Compare {@link #extract(Iterable, Class)} for an easily adaptable example.
   *
   * @param array        array with arbitrary elements
   * @param targetType   type of elements which shall be extracted
   * @param <T>          target type
   * @return a linked list containing only the elements of the target type (including extending types)
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T> LinkedList<T> extract(@NotNull Object[] array,
                                          @NotNull Class<T> targetType)
  {
    return (LinkedList<T>)filter(array, Check.isType(targetType));
  }

  /**
   * Extract all elements of a given type.
   * <p>
   * Internally this is using reflection, namely the {@link Class#isInstance(Object)} method,
   * which is not considered fast. If you need this method often, rethink your design.
   * Compare {@link #extract(Iterable, Class)} for an easily adaptable example.
   *
   * @param target       target collection
   * @param array        array with arbitrary elements
   * @param targetType   type of elements which shall be extracted
   * @param <T>          target type
   * @param <C>          target collection type
   * @return a linked list containing only the elements of the target type (including extending types)
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T, C extends Collection<? super T>> C extract(@NotNull C target,
                                                               @NotNull Object[] array,
                                                               @NotNull Class<T> targetType)
  {
    return filter(target, (T[])array, Check.<T>isType(targetType));
  }

  /**
   * Extract all elements of a given type.
   * <p>
   * Internally this is using reflection, namely the {@link Class#isInstance(Object)} method,
   * which is not considered fast. If you need this method often, rethink your design.
   * <p>
   * Example:
   * <blockquote><pre>
   * public Collection&lt;String&gt; getStrings(Collection&lt;?&gt; collection) {
   *   return Types.extract(collection, String.class);
   * }
   * </pre></blockquote>
   *
   * @param iterable     iterable with arbitrary elements
   * @param targetType   type of elements which shall be extracted
   * @param <T>          target type
   * @return a linked list containing only the elements of the target type (including extending types)
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T> LinkedList<T> extract(@NotNull Iterable<?> iterable,
                                          @NotNull Class<T> targetType)
  {
    return (LinkedList<T>)filter(iterable, Check.isType(targetType));
  }

  /**
   * Extract all elements of a given type.
   * <p>
   * Internally this is using reflection, namely the {@link Class#isInstance(Object)} method,
   * which is not considered fast. If you need this method often, rethink your design.
   * Compare {@link #extract(Iterable, Class)} for an easily adaptable example.
   *
   * @param target       target collection
   * @param iterable     iterable with arbitrary elements
   * @param targetType   type of elements which shall be extracted
   * @param <T>          target type
   * @param <C>          target collection type
   * @return a linked list containing only the elements of the target type (including extending types)
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T, C extends Collection<? super T>> C extract(@NotNull C target,
                                                               @NotNull Iterable<?> iterable,
                                                               @NotNull Class<T> targetType)
  {
    return filter(target, (Iterable<T>)iterable, Check.<T>isType(targetType));
  }

  /**
   * Extract all elements of a given type.
   * <p>
   * Internally this is using reflection, namely the {@link Class#isInstance(Object)} method,
   * which is not considered fast. If you need this method often, rethink your design.
   * Compare {@link #extract(Iterable, Class)} for an easily adaptable example.
   *
   * @param iterator     iterator with arbitrary elements
   * @param targetType   type of elements which shall be extracted
   * @param <T>          target type
   * @return a linked list containing only the elements of the target type (including extending types)
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T> LinkedList<T> extract(@NotNull Iterator<?> iterator,
                                          @NotNull Class<T> targetType)
  {
    return (LinkedList<T>)filter(iterator, Check.isType(targetType));
  }

  /**
   * Extract all elements of a given type.
   * <p>
   * Internally this is using reflection, namely the {@link Class#isInstance(Object)} method,
   * which is not considered fast. If you need this method often, rethink your design.
   * Compare {@link #extract(Iterable, Class)} for an easily adaptable example.
   *
   * @param target       target collection
   * @param iterator     iterator with arbitrary elements
   * @param targetType   type of elements which shall be extracted
   * @param <T>          target type
   * @param <C>          target collection type
   * @return a linked list containing only the elements of the target type (including extending types)
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T, C extends Collection<? super T>> C extract(@NotNull C target,
                                                               @NotNull Iterator<?> iterator,
                                                               @NotNull Class<T> targetType)
  {
    return filter(target, (Iterator<T>)iterator, Check.<T>isType(targetType));
  }

  /**
   * Extract all elements of a given type.
   * <p>
   * Internally this is using reflection, namely the {@link Class#isInstance(Object)} method,
   * which is not considered fast. If you need this method often, rethink your design.
   * Compare {@link #extract(Iterable, Class)} for an easily adaptable example.
   *
   * @param enumeration  enumeration with arbitrary elements
   * @param targetType   type of elements which shall be extracted
   * @param <T>          target type
   * @return a linked list containing only the elements of the target type (including extending types)
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T> LinkedList<T> extract(@NotNull Enumeration<?> enumeration,
                                          @NotNull Class<T> targetType)
  {
    return (LinkedList<T>)filter(enumeration, Check.isType(targetType));
  }

  /**
   * Extract all elements of a given type.
   * <p>
   * Internally this is using reflection, namely the {@link Class#isInstance(Object)} method,
   * which is not considered fast. If you need this method often, rethink your design.
   * Compare {@link #extract(Iterable, Class)} for an easily adaptable example.
   *
   * @param target       target collection
   * @param enumeration  enumeration with arbitrary elements
   * @param targetType   type of elements which shall be extracted
   * @param <T>          target type
   * @param <C>          target collection type
   * @return a linked list containing only the elements of the target type (including extending types)
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T, C extends Collection<? super T>> C extract(@NotNull C target,
                                                               @NotNull Enumeration<?> enumeration,
                                                               @NotNull Class<T> targetType)
  {
    return filter(target, (Enumeration<T>)enumeration, Check.<T>isType(targetType));
  }

  /**
   * Are two collections equal?
   * <p>
   * This uses the {@link Object#equals(Object) equals()} method of objects of the first collection
   * for equality checking.
   * Both collections are iterated over in order, so if both contain the same elements, but in different
   * orders, they are considered different.
   *
   * @param c1 first collection
   * @param c2 second collection
   * @param <T1> type of first collection
   * @param <T2> type of second collection
   * @return {@code true} if both collections are equal<br>
   *         {@code false} if they are different
   */
  public static <T1, T2> boolean areEqual(@NotNull Collection<T1> c1,
                                          @NotNull Collection<T2> c2)
  {
    return areEqual(c1, c2, Match.DEFAULT_MATCHER);
  }

  /**
   * Are two collections equal?
   * <p>
   * Both collections are iterated over in order, so if both contain the same elements, but in different
   * orders, they are considered different.
   *
   * @param c1 first collection
   * @param c2 second collection
   * @param matcher matcher for equality checking
   * @param <T1> type of first collection
   * @param <T2> type of second collection
   * @return {@code true} if both collection are equal<br>
   *         {@code false} if they are different
   */
  public static <T1, T2> boolean areEqual(@NotNull Collection<T1> c1,
                                          @NotNull Collection<T2> c2,
                                          @NotNull BiPredicate<? super T1, ? super T2> matcher)
  {
    return c1 == c2 ||
           (c1.size() == c2.size() && areEqual(c1.iterator(), c2.iterator(), matcher));
  }

  /**
   * Are two countables equal?
   * <p>
   * This uses the {@link Object#equals(Object) equals()} method of objects of the first countabel
   * for equality checking.
   * Both countables are iterated over in order, so if both contain the same elements, but in different
   * orders, they are considered different.
   *
   * @param c1 first countable
   * @param c2 second countable
   * @param <T1> type of first countable
   * @param <T2> type of second countable
   * @return {@code true} if both countables are equal<br>
   *         {@code false} if they are different
   */
  public static <T1, T2> boolean areEqual(@NotNull Countable<T1> c1,
                                          @NotNull Countable<T2> c2)
  {
    return areEqual(c1, c2, Match.DEFAULT_MATCHER);
  }

  /**
   * Are two countables equal?
   * <p>
   * Both countables are iterated over in order, so if both contain the same elements, but in different
   * orders, they are considered different.
   *
   * @param c1 first countable
   * @param c2 second countable
   * @param matcher matcher for equality checking
   * @param <T1> type of first countable
   * @param <T2> type of second countable
   * @return {@code true} if both countable are equal<br>
   *         {@code false} if they are different
   */
  public static <T1, T2> boolean areEqual(@NotNull Countable<T1> c1,
                                          @NotNull Countable<T2> c2,
                                          @NotNull BiPredicate<? super T1, ? super T2> matcher)
  {
    return c1 == c2 ||
           (c1.size() == c2.size() && areEqual(c1.iterator(), c2.iterator(), matcher));
  }

  /**
   * Are two arrays equal?
   * <p>
   * This uses the {@link Object#equals(Object) equals()} method of objects of the first array
   * for equality checking.
   * Both arrays are iterated over in order, so if both contain the same elements, but in different
   * orders, they are considered different.
   *
   * @param arr1 first array
   * @param arr2 second array
   * @param <T1> type of first array
   * @param <T2> type of second array
   * @return {@code true} if both arrays are equal<br>
   *         {@code false} if they are different
   */
  public static <T1, T2> boolean areEqual(@NotNull T1[] arr1,
                                          @NotNull T2[] arr2)
  {
    return areEqual(arr1, arr2, Match.DEFAULT_MATCHER);
  }

  /**
   * Are two arrays equal?
   * <p>
   * Both arrays are iterated over in order, so if both contain the same elements, but in different
   * orders, they are considered different.
   *
   * @param arr1 first array
   * @param arr2 second array
   * @param matcher matcher for equality checking
   * @param <T1> type of first array
   * @param <T2> type of second array
   * @return {@code true} if both arrays are equal<br>
   *         {@code false} if they are different
   */
  public static <T1, T2> boolean areEqual(@NotNull T1[] arr1,
                                          @NotNull T2[] arr2,
                                          @NotNull BiPredicate<? super T1, ? super T2> matcher)
  {
    return arr1 == arr2 ||
           (arr1.length == arr2.length  &&  areEqual(asList(arr1), asList(arr2), matcher));
  }

  /**
   * Are two iterables equal?
   * <p>
   * This uses the {@link Object#equals(Object) equals()} method of objects of the first iterable
   * for equality checking.
   * Both iterables are iterated over in order, so if both contain the same elements, but in different
   * orders, they are considered different.
   *
   * @param it1 first iterable
   * @param it2 second iterable
   * @param <T1> type of first iterable
   * @param <T2> type of second iterable
   * @return {@code true} if both iterables are equal<br>
   *         {@code false} if they are different
   */
  public static <T1, T2> boolean areEqual(@NotNull Iterable<T1> it1,
                                          @NotNull Iterable<T2> it2)
  {
    return areEqual(it1, it2, Match.DEFAULT_MATCHER);
  }

  /**
   * Are two iterables equal?
   * <p>
   * Both iterables are iterated over in order, so if both contain the same elements, but in different
   * orders, they are considered different.
   *
   * @param it1 first iterable
   * @param it2 second iterable
   * @param matcher matcher for equality checking
   * @param <T1> type of first iterable
   * @param <T2> type of second iterable
   * @return {@code true} if both iterables are equal<br>
   *         {@code false} if they are different
   */
  public static <T1, T2> boolean areEqual(@NotNull Iterable<T1> it1,
                                          @NotNull Iterable<T2> it2,
                                          @NotNull BiPredicate<? super T1, ? super T2> matcher)
  {
    return it1 == it2 || areEqual(it1.iterator(), it2.iterator(), matcher);
  }

  /**
   * Are two iterations equal?
   * <p>
   * This uses the {@link Object#equals(Object) equals()} method of objects of the first iterator
   * for equality checking.
   * Both iterators are iterated over in order, so if both contain the same elements, but in different
   * orders, they are considered different.
   *
   * @param it1 first iterator
   * @param it2 second iterator
   * @param <T1> type of first iterator
   * @param <T2> type of second iterator
   * @return {@code true} if both iterators are equal<br>
   *         {@code false} if they are different
   */
  public static <T1, T2> boolean areEqual(@NotNull Iterator<T1> it1,
                                          @NotNull Iterator<T2> it2)
  {
    return areEqual(it1, it2, Match.DEFAULT_MATCHER);
  }

  /**
   * Are two iterations equal?
   * <p>
   * Both iterators are iterated over in order, so if both contain the same elements, but in different
   * orders, they are considered different.
   *
   * @param it1 first iterator
   * @param it2 second iterator
   * @param matcher matcher for equality checking
   * @param <T1> type of first iterator
   * @param <T2> type of second iterator
   * @return {@code true} if both iterators are equal<br>
   *         {@code false} if they are different
   */
  public static <T1, T2> boolean areEqual(@NotNull Iterator<T1> it1,
                                          @NotNull Iterator<T2> it2,
                                          @NotNull BiPredicate<? super T1, ? super T2> matcher)
  {
    while (it1.hasNext() && it2.hasNext()) {
      if (!matcher.test(it1.next(), it2.next())) {
        return false;
      }
    }
    return !it1.hasNext() && !it2.hasNext();
  }

  /**
   * Are two enumerators equal?
   * <p>
   * This uses the {@link Object#equals(Object) equals()} method of objects of the first enumeration
   * for equality checking.
   * Both iterators are iterated over in order, so if both contain the same elements, but in different
   * orders, they are considered different.
   *
   * @param en1 first enumeration
   * @param en2 second enumeration
   * @param <T1> type of first enumeration
   * @param <T2> type of second enumeration
   * @return {@code true} if both enumerations are equal<br>
   *         {@code false} if they are different
   */
  public static <T1, T2> boolean areEqual(@NotNull Enumeration<T1> en1,
                                          @NotNull Enumeration<T2> en2)
  {
    return areEqual(en1, en2, Match.DEFAULT_MATCHER);
  }

  /**
   * Are two enumerators equal?
   * <p>
   * This uses the {@link Object#equals(Object) equals()} method of objects of the first enumeration
   * for equality checking.
   * Both iterators are iterated over in order, so if both contain the same elements, but in different
   * orders, they are considered different.
   *
   * @param en1 first enumeration
   * @param en2 second enumeration
   * @param matcher matcher for equality checking
   * @param <T1> type of first enumeration
   * @param <T2> type of second enumeration
   * @return {@code true} if both enumerations are equal<br>
   *         {@code false} if they are different
   */
  public static <T1, T2> boolean areEqual(@NotNull Enumeration<T1> en1,
                                          @NotNull Enumeration<T2> en2,
                                          @NotNull BiPredicate<? super T1, ? super T2> matcher)
  {
    while (en1.hasMoreElements() && en2.hasMoreElements()) {
      if (!matcher.test(en1.nextElement(), en2.nextElement())) {
        return false;
      }
    }
    return !en1.hasMoreElements() && !en2.hasMoreElements();
  }

  /**
   * Are the given sets equal.
   * This method takes care of the fact that sets don't always have a defined order.
   * It also assumes that the set-internal idea of equality is correct,
   * by use of the set's {@link Collection#containsAll(Collection)} method.
   * @param set1 first set of objects
   * @param set2 second set of objects
   * @return {@code true} if both sets contain the same
   * @param <T> element type of the sets
   */
  public static <T> boolean areEqualSets(@NotNull Collection<T> set1,
                                         @NotNull Collection<T> set2)
  {
    return set1.containsAll(set2) && set2.containsAll(set1);
  }

  /**
   * Get the nth element of an iterable.
   * @param iterable iterable
   * @param index    index of requested element
   * @param <T>      type of iterable
   * @return element with requested index
   * @throws java.lang.IndexOutOfBoundsException if index is invalid
   */
  public static <T> T get(@NotNull Iterable<T> iterable, final int index)
  {
    if (index < 0) {
      throw new IndexOutOfBoundsException("Negative index not allowed!");
    }
    int i = 0;
    for (T elem : iterable) {
      if (i++ == index) {
        return elem;
      }
    }
    throw new IndexOutOfBoundsException(String.format("%d >= %d !", index, i));
  }

  /**
   * Get the first index where a given element appears.
   * @param array    array to iterate over
   * @param element  element to look for, whether {@code null} is allowed depends on the matcher
   * @param matcher  matcher to use for equality checking
   * @param <T>      type of iterable
   * @param <E>      type of element
   * @return {@code -1} if no element was found, otherwise the index of the element (i.e. the number of
   *         iterations minus one)
   */
  public static <T, E> int indexOf(@NotNull T[] array,
                                   E element,
                                   @NotNull BiPredicate<? super T, ? super E> matcher)
  {
    return indexOf(asList(array), element, matcher);
  }

  /**
   * Get the first index where a given element appears.
   * @param array    array to iterate over
   * @param element  element to look for, whether {@code null} is allowed depends on the matcher
   * @param matcher  matcher to use for equality checking
   * @param start    index where to start the search
   * @param <T>      type of iterable
   * @param <E>      type of element
   * @return {@code -1} if no element was found, otherwise the index of the element (i.e. the number of
   *         iterations minus one)
   */
  public static <T, E> int indexOf(@NotNull T[] array,
                                   E element,
                                   @NotNull BiPredicate<? super T, ? super E> matcher,
                                   int start)
  {
    return indexOf(asList(array), element, matcher, start);
  }

  /**
   * Get the first index where a given element appears.
   * @param iterable iterable to iterate over
   * @param element  element to look for, whether {@code null} is allowed depends on the matcher
   * @param matcher  matcher to use for equality checking
   * @param <T>      type of iterable
   * @param <E>      type of element
   * @return {@code -1} if no element was found, otherwise the index of the element (i.e. the number of
   *         iterations minus one)
   */
  public static <T, E> int indexOf(@NotNull Iterable<T> iterable,
                                   E element,
                                   @NotNull BiPredicate<? super T, ? super E> matcher)
  {
    return indexOf(iterable.iterator(), element, matcher);
  }

  /**
   * Get the first index where a given element appears.
   * @param iterable iterable to iterate over
   * @param element  element to look for, whether {@code null} is allowed depends on the matcher
   * @param matcher  matcher to use for equality checking
   * @param start    index where to start the search
   * @param <T>      type of iterable
   * @param <E>      type of element
   * @return {@code -1} if no element was found, otherwise the index of the element (i.e. the number of
   *         iterations minus one)
   */
  public static <T, E> int indexOf(@NotNull Iterable<T> iterable,
                                   E element,
                                   @NotNull BiPredicate<? super T, ? super E> matcher,
                                   int start)
  {
    return indexOf(iterable.iterator(), element, matcher, start);
  }

  /**
   * Get the first index where a given element appears.
   * @param iterator iterator to iterate over
   * @param element  element to look for, whether {@code null} is allowed depends on the matcher
   * @param matcher  matcher to use for equality checking
   * @param <T>      type of iterator
   * @param <E>      type of element
   * @return {@code -1} if no element was found, otherwise the index of the element
   */
  public static <T, E> int indexOf(@NotNull Iterator<T> iterator,
                                   E element,
                                   @NotNull BiPredicate<? super T, ? super E> matcher)
  {
    int index = 0;
    while (iterator.hasNext()) {
      if (matcher.test(iterator.next(), element)) {
        return index;
      }
      ++index;
    }
    return -EMPTY_HASH;
  }

  /**
   * Get the first index where a given element appears.
   * @param iterator iterator to iterate over
   * @param element  element to look for, whether {@code null} is allowed depends on the matcher
   * @param matcher  matcher to use for equality checking
   * @param start    index where to start the search
   * @param <T>      type of iterator
   * @param <E>      type of element
   * @return {@code -1} if no element was found, otherwise the index of the element
   */
  public static <T, E> int indexOf(@NotNull Iterator<T> iterator,
                                   E element,
                                   @NotNull BiPredicate<? super T, ? super E> matcher,
                                   int start)
  {
    int index = 0;
    while (index < start && iterator.hasNext()) {
      ++index;
    }
    while (iterator.hasNext()) {
      if (matcher.test(iterator.next(), element)) {
        return index;
      }
      ++index;
    }
    return -EMPTY_HASH;
  }

  /**
   * Get the first index where a given element appears.
   * @param enumeration enumeration to iterate over
   * @param element  element to look for, whether {@code null} is allowed depends on the matcher
   * @param matcher  matcher to use for equality checking
   * @param <T>      type of enumeration
   * @param <E>      type of element
   * @return {@code -1} if no element was found, otherwise the index of the element
   */
  public static <T, E> int indexOf(@NotNull Enumeration<T> enumeration,
                                   E element,
                                   @NotNull BiPredicate<? super T, ? super E> matcher)
  {
    int index = 0;
    while (enumeration.hasMoreElements()) {
      if (matcher.test(enumeration.nextElement(), element)) {
        return index;
      }
      ++index;
    }
    return -EMPTY_HASH;
  }

  /**
   * Get the first index where a given element appears.
   * @param enumeration enumeration to iterate over
   * @param element  element to look for, whether {@code null} is allowed depends on the matcher
   * @param matcher  matcher to use for equality checking
   * @param start    index where to start the search
   * @param <T>      type of enumeration
   * @param <E>      type of element
   * @return {@code -1} if no element was found, otherwise the index of the element
   */
  public static <T, E> int indexOf(@NotNull Enumeration<T> enumeration,
                                   E element,
                                   @NotNull BiPredicate<? super T, ? super E> matcher,
                                   int start)
  {
    int index = 0;
    while (index < start && enumeration.hasMoreElements()) {
      ++index;
    }
    while (enumeration.hasMoreElements()) {
      if (matcher.test(enumeration.nextElement(), element)) {
        return index;
      }
      ++index;
    }
    return -EMPTY_HASH;
  }

  /**
   * Get the first index where a given element appears.
   * @param list     list to search through
   * @param element  element to look for, whether {@code null} is allowed depends on the matcher
   * @param matcher  matcher to use for equality checking
   * @param start    index where to start the search
   * @param <T>      type of list
   * @param <E>      type of element
   * @return {@code -1} if no element was found, otherwise the index of the element
   */
  public static <T, E> int indexOf(@NotNull List<T> list,
                                   E element,
                                   @NotNull BiPredicate<? super T, ? super E> matcher,
                                   int start)
  {
    if (start >= list.size()) {
      return -EMPTY_HASH;
    }
    int index = start;
    for (ListIterator<T> it = list.listIterator(start);  it.hasNext();  ) {
      if (matcher.test(it.next(), element)) {
        return index;
      }
      ++index;
    }
    return -EMPTY_HASH;
  }

  /**
   * Get the last index where a given element appears.
   * @param list     list to search through
   * @param element  element to look for, whether {@code null} is allowed depends on the matcher
   * @param matcher  matcher to use for equality checking
   * @param start    index where to start the search
   * @param <T>      type of list
   * @param <E>      type of element
   * @return {@code -1} if no element was found, otherwise the index of the element
   */
  public static <T, E> int lastIndexOf(@NotNull List<T> list,
                                       E element,
                                       @NotNull BiPredicate<? super T, ? super E> matcher,
                                       int start)
  {
    if (start < 0) {
      return -EMPTY_HASH;
    }
    int index = start;
    for (ListIterator<T> it = list.listIterator(start);  it.hasPrevious();  ) {
      if (matcher.test(it.previous(), element)) {
        return index;
      }
      --index;
    }
    return -EMPTY_HASH;
  }

  /**
   * Get the last index where a given element appears.
   * @param list     list to search through
   * @param element  element to look for, whether {@code null} is allowed depends on the matcher
   * @param matcher  matcher to use for equality checking
   * @param <T>      type of list
   * @param <E>      type of element
   * @return {@code -1} if no element was found, otherwise the index of the element
   */
  public static <T, E> int lastIndexOf(@NotNull List<T> list,
                                       E element,
                                       @NotNull BiPredicate<? super T, ? super E> matcher)
  {
    return lastIndexOf(list, element, matcher, list.size());
  }

  /**
   * Get the last matching element from a list.
   * @param list     list to search through
   * @param element  element to look for, whether {@code null} is allowed depends on the matcher
   * @param matcher  matcher to use for equality checking
   * @param start    index where to start the search
   * @param <T>      type of list
   * @param <E>      type of element
   * @return {@code null} if no element was found, otherwise the matching element
   */
  public static <T, E> T lastMatch(@NotNull List<T> list,
                                   E element,
                                   @NotNull BiPredicate<? super T, ? super E> matcher,
                                   int start)
  {
    if (start < 0) {
      return null;
    }
    for (ListIterator<T> it = list.listIterator(start);  it.hasPrevious();  ) {
      final T current = it.previous();
      if (matcher.test(current, element)) {
        return current;
      }
    }
    return null;
  }

  /**
   * Get the last matching element where a given element appears.
   * @param list     list to search through
   * @param element  element to look for, whether {@code null} is allowed depends on the matcher
   * @param matcher  matcher to use for equality checking
   * @param <T>      type of list
   * @param <E>      type of element
   * @return {@code null} if no element was found, otherwise the matching element
   */
  @Nullable
  public static <T, E> T lastMatch(@NotNull List<T> list,
                                   E element,
                                   @NotNull BiPredicate<? super T, ? super E> matcher)
  {
    return lastMatch(list, element, matcher, list.size());
  }

  /**
   * Is an element contained in an array?
   * @param array    array which possibly contains an element
   * @param element  element to look for, whether {@code null} is allowed depends on the matcher
   * @param matcher  matcher for equality checking
   * @param <T>      type of iterable
   * @param <E>      type of element
   * @return {@code true} if the element is contained (i.e. the matcher finds a match)<br>
   *         {@code false} otherwise
   */
  public static <T, E> boolean contains(@NotNull T[] array,
                                        E element,
                                        @NotNull BiPredicate<? super T, ? super E> matcher)
  {
    return indexOf(array, element, matcher) >= 0;
  }

  /**
   * Is an element contained in an iterable?
   * @param iterable iterable which possibly contains an element
   * @param element  element to look for, whether {@code null} is allowed depends on the matcher
   * @param matcher  matcher for equality checking
   * @param <T>      type of iterable
   * @param <E>      type of element
   * @return {@code true} if the element is contained (i.e. the matcher finds a match)<br>
   *         {@code false} otherwise
   */
  public static <T, E> boolean contains(@NotNull Iterable<T> iterable,
                                        E element,
                                        @NotNull BiPredicate<? super T, ? super E> matcher)
  {
    return indexOf(iterable, element, matcher) >= 0;
  }

  /**
   * Is an element contained in an iterator?
   * @param iterator iterator which possibly contains an element
   * @param element  element to look for, whether {@code null} is allowed depends on the matcher
   * @param matcher  matcher for equality checking
   * @param <T>      type of iterator
   * @param <E>      type of element
   * @return {@code true} if the element is contained (i.e. the matcher finds a match)<br>
   *         {@code false} otherwise
   */
  public static <T, E> boolean contains(@NotNull Iterator<T> iterator,
                                        E element,
                                        @NotNull BiPredicate<? super T, ? super E> matcher)
  {
    return indexOf(iterator, element, matcher) >= 0;
  }

  /**
   * Is an element contained in an enumeration?
   * @param enumeration enumeration which possibly contains an element
   * @param element  element to look for, whether {@code null} is allowed depends on the matcher
   * @param matcher  matcher for equality checking
   * @param <T>      type of iterator
   * @param <E>      type of element
   * @return {@code true} if the element is contained (i.e. the matcher finds a match)<br>
   *         {@code false} otherwise
   */
  public static <T, E> boolean contains(@NotNull Enumeration<T> enumeration,
                                        E element,
                                        @NotNull BiPredicate<? super T, ? super E> matcher)
  {
    return indexOf(enumeration, element, matcher) >= 0;
  }

  /**
   * Get a copy of an array, while copying the content.
   * @param array array
   * @param <T> content type
   * @return array list with copied elements
   * @see #copy(Collection, Copyable[])
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T extends Copyable<T>> ArrayList<T> copy(@NotNull T[] array)
  {
    return map(new ArrayList<>(array.length), array, (CopyConverter<T>)COPY_CONVERTER);
  }

  /**
   * Get a copy of an array, while copying the content.
   * @param target target collection where the copies are stored
   * @param array array
   * @param <T> content type
   * @param <C> target collection type
   * @return collection copied elements
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T extends Copyable<T>, C extends Collection<T>> C copy(@NotNull C target,
                                                                        @NotNull T[] array)
  {
    return map(target, array, (CopyConverter<T>)COPY_CONVERTER);
  }

  /**
   * Get a copy of an iterable, while copying the content.
   * @param iterable iterable
   * @param <T> content type
   * @return linked list with copied elements
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T extends Copyable<T>> LinkedList<T> copy(@NotNull Iterable<T> iterable)
  {
    return map(iterable, (CopyConverter<T>)COPY_CONVERTER);
  }

  /**
   * Get a copy of a indexable, while copying the content.
   * @param indexable indexable collection
   * @param <T> content type
   * @return array list with copied elements
   */
  @NotNull
  public static <T extends Copyable<T>> ArrayList<T> copy(@NotNull Indexable<T> indexable)
  {
    return copy(new ArrayList<>(indexable.size()), indexable);
  }

  /**
   * Get a copy of a collection, while copying the content.
   * @param collection collection
   * @param <T> content type
   * @return array list with copied elements
   */
  @NotNull
  public static <T extends Copyable<T>> ArrayList<T> copy(@NotNull Collection<T> collection)
  {
    return copy(new ArrayList<>(collection.size()), collection);
  }

  /**
   * Get a copy of a countable, while copying the content.
   * @param countable countable
   * @param <T> content type
   * @return array list with copied elements
   */
  @NotNull
  public static <T extends Copyable<T>> ArrayList<T> copy(@NotNull Countable<T> countable)
  {
    return copy(new ArrayList<>(countable.size()), countable);
  }

  /**
   * Get a copy of an iterable, while copying the content.
   * @param target target collection where the copies are stored
   * @param iterable iterable
   * @param <T> content type
   * @param <C> target collection type
   * @return collection copied elements
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T extends Copyable<T>, C extends Collection<T>> C copy(@NotNull C target,
                                                                        @NotNull Iterable<T> iterable)
  {
    return map(target, iterable, (CopyConverter<T>)COPY_CONVERTER);
  }

  /**
   * Get a copy of an iterator, while copying the content.
   * @param iterator iterator
   * @param <T> content type
   * @return linked list with copied elements
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T extends Copyable<T>> LinkedList<T> copy(@NotNull Iterator<T> iterator)
  {
    return map(iterator, (CopyConverter<T>)COPY_CONVERTER);
  }

  /**
   * Get a copy of an iterator while copying the content.
   * @param target target collection where the copies are stored
   * @param iterator iterator
   * @param <T> content type
   * @param <C> target collection type
   * @return collection copied elements
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T extends Copyable<T>, C extends Collection<T>> C copy(@NotNull C target,
                                                                        @NotNull Iterator<T> iterator)
  {
    return map(target, iterator, (CopyConverter<T>)COPY_CONVERTER);
  }

  /**
   * Get a copy of an enumeration, while copying the content.
   * @param enumeration enumeration
   * @param <T> content type
   * @return linked list with copied elements
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T extends Copyable<T>> LinkedList<T> copy(@NotNull Enumeration<T> enumeration)
  {
    return map(enumeration, (CopyConverter<T>)COPY_CONVERTER);
  }

  /**
   * Get a copy of an enumeration while copying the content.
   * @param target target collection where the copies are stored
   * @param enumeration enumeration
   * @param <T> content type
   * @param <C> target collection type
   * @return collection copied elements
   */
  @NotNull
  @SuppressWarnings("unchecked")
  public static <T extends Copyable<T>, C extends Collection<T>> C copy(@NotNull C target,
                                                                        @NotNull Enumeration<T> enumeration)
  {
    return map(target, enumeration, (CopyConverter<T>)COPY_CONVERTER);
  }

  /**
   * Join an array of strings into one.
   * @param glue    string to use as separator
   * @param strings strings to join
   * @return joined string
   */
  @NotNull
  public static String join(@NotNull String glue,
                            @NotNull String ... strings)
  {
    return join(glue, asList(strings));
  }

  /**
   * Join an array of strings into one.
   * @param glue    character to use as separator
   * @param strings strings to join
   * @return joined string
   */
  @NotNull
  public static String join(char glue,
                            @NotNull String ... strings)
  {
    return join(glue, asList(strings));
  }

  /**
   * Join a bunch of strings into one.
   * @param glue    string to use as separator
   * @param strings strings to join
   * @return joined string
   */
  @NotNull
  public static String join(@NotNull String glue,
                            @NotNull Iterable<String> strings)
  {
    return join(glue, strings.iterator());
  }

  /**
   * Join a bunch of strings into one.
   * @param glue    character to use as separator
   * @param strings strings to join
   * @return joined string
   */
  @NotNull
  public static String join(char glue,
                            @NotNull Iterable<String> strings)
  {
    return join(glue, strings.iterator());
  }

  /**
   * Join a bunch of strings into one.
   * @param glue    scharacter to use as separator
   * @param strings strings to join
   * @return joined string
   */
  @NotNull
  public static String join(char glue,
                            @NotNull Iterator<String> strings)
  {
    return join(Character.toString(glue),
                strings);
  }

  /**
   * Join a bunch of strings into one.
   * @param glue    string to use as separator
   * @param strings strings to join
   * @return joined string
   */
  @NotNull
  public static String join(@NotNull String glue,
                            @NotNull Iterator<String> strings)
  {
    StringBuilder sb = new StringBuilder();
    boolean first = true;
    while(strings.hasNext()) {
      if (first) {
        first = false;
      }
      else {
        sb.append(glue);
      }
      sb.append(strings.next());
    }
    return sb.toString();
  }

  /**
   * Join a bunch of strings into one.
   * @param glue    string to use as separator
   * @param strings strings to join
   * @return joined string
   */
  @NotNull
  public static String join(@NotNull String glue,
                            @NotNull Enumeration<String> strings)
  {
    StringBuilder sb = new StringBuilder();
    boolean first = true;
    while(strings.hasMoreElements()) {
      if (first) {
        first = false;
      }
      else {
        sb.append(glue);
      }
      sb.append(strings.nextElement());
    }
    return sb.toString();
  }

  /**
   * View an iterator as if it belongs to another type.
   * Instead of {@link #map(java.util.Iterator, Function)} this
   * will work transparently on the fly without creating an intermediate list.
   * @param sourceIterator source iterator
   * @param mapper         mapper function which converts the source to the target type
   * @param <T>            target type
   * @param <S>            source type
   * @return mapped iterator
   */
  @NotNull
  public static <T,S> Iterator<T> view(@NotNull Iterator<S> sourceIterator,
                                       @NotNull Function<? super S, ? extends T> mapper)
  {
    return new IteratorConverter<>(sourceIterator, mapper);
  }

  /**
   * View an iterable
   * Instead of {@link #map(java.lang.Iterable, Function)} this
   * will work transparently on the fly.
   * @param sourceIterable source iterable
   * @param mapper         mapper function which converts the source to the target type
   * @param <T>            target type
   * @param <S>            source type
   * @return mapped iterable
   */
  @NotNull
  public static <T,S> Iterable<T> view(@NotNull Iterable<S> sourceIterable,
                                       @NotNull Function<? super S, ? extends T> mapper)
  {
    return new IterableConverter<>(sourceIterable, mapper);
  }

  /**
   * View a generic iterator as if it is an iterator for primitive double values.
   * @param sourceIterator source iterator
   * @param doubleMapper   mapper which maps the elements of the source iterator to {@code double}
   * @return primitive double iterator
   * @param <S> source element type
   */
  @NotNull
  public static <S> PrimitiveIterator.OfDouble doubleView(@NotNull Iterator<S> sourceIterator,
                                                          @NotNull ToDoubleFunction<? super S> doubleMapper)
  {
    return new PrimitiveIterator.OfDouble()
    {
      @Override
      public double nextDouble()
      {
        return doubleMapper.applyAsDouble(sourceIterator.next());
      }

      @Override
      public boolean hasNext()
      {
        return sourceIterator.hasNext();
      }
    };
  }

  /**
   * View a generic iterable as if it is an iterable of primitive {@code double} values.
   * <p>
   * Because for-each loops will silently introduce boxing/unboxing it is recommended to
   * iterate over the returned double iterable with its looping functions:
   * {@link PrimitiveDoubleIterable#forEachDouble(DoubleConsumer)} or 
   * {@link PrimitiveDoubleIterable#forEachDoubleFragile(FragileDoubleConsumer)} (if code throws
   * checked exceptions).
   * @param sourceIterable source iterable
   * @param doubleMapper   mapper from the incoming elements to double
   * @return primitive double iterable
   * @param <S> source element type
   */
  @NotNull
  public static <S> PrimitiveDoubleIterable doubleView(@NotNull Iterable<S> sourceIterable,
                                                       @NotNull ToDoubleFunction<? super S> doubleMapper)
  {
    return () -> doubleView(sourceIterable.iterator(), doubleMapper);
  }

  /**
   * View a generic iterator as if it is an iterator for primitive float values.
   * @param sourceIterator source iterator
   * @param floatMapper   mapper which maps the elements of the source iterator to {@code float}
   * @return primitive float iterator
   * @param <S> source element type
   */
  @NotNull
  public static <S> PrimitiveFloatIterator floatView(@NotNull Iterator<S> sourceIterator,
                                                     @NotNull ToFloatFunction<? super S> floatMapper)
  {
    return new PrimitiveFloatIterator()
    {
      @Override
      public float nextFloat()
      {
        return floatMapper.applyAsFloat(sourceIterator.next());
      }

      @Override
      public boolean hasNext()
      {
        return sourceIterator.hasNext();
      }
    };
  }

  /**
   * View a generic iterable as if it is an iterable of primitive {@code float} values.
   * <p>
   * Because for-each loops will silently introduce boxing/unboxing it is recommended to
   * iterate over the returned float iterable with its looping functions:
   * {@link PrimitiveFloatIterable#forEachFloat(FloatConsumer)} or 
   * {@link PrimitiveFloatIterable#forEachFloatFragile(FragileFloatConsumer)} (if code throws
   * checked exceptions).
   * @param sourceIterable source iterable
   * @param floatMapper   mapper from the incoming elements to float
   * @return primitive float iterable
   * @param <S> source element type
   */
  @NotNull
  public static <S> PrimitiveFloatIterable floatView(@NotNull Iterable<S> sourceIterable,
                                                     @NotNull ToFloatFunction<? super S> floatMapper)
  {
    return () -> floatView(sourceIterable.iterator(), floatMapper);
  }

  /**
   * View a generic iterator as if it is an iterator for primitive long values.
   * @param sourceIterator source iterator
   * @param longMapper   mapper which maps the elements of the source iterator to {@code long}
   * @return primitive long iterator
   * @param <S> source element type
   */
  @NotNull
  public static <S> PrimitiveIterator.OfLong longView(@NotNull Iterator<S> sourceIterator,
                                                      @NotNull ToLongFunction<? super S> longMapper)
  {
    return new PrimitiveIterator.OfLong()
    {
      @Override
      public long nextLong()
      {
        return longMapper.applyAsLong(sourceIterator.next());
      }

      @Override
      public boolean hasNext()
      {
        return sourceIterator.hasNext();
      }
    };
  }

  /**
   * View a generic iterable as if it is an iterable of primitive {@code long} values.
   * <p>
   * Because for-each loops will silently introduce boxing/unboxing it is recommended to
   * iterate over the returned long iterable with its looping functions:
   * {@link PrimitiveLongIterable#forEachLong(LongConsumer)} or 
   * {@link PrimitiveLongIterable#forEachLongFragile(FragileLongConsumer)} (if code throws
   * checked exceptions).
   * @param sourceIterable source iterable
   * @param longMapper   mapper from the incoming elements to long
   * @return primitive long iterable
   * @param <S> source element type
   */
  @NotNull
  public static <S> PrimitiveLongIterable longView(@NotNull Iterable<S> sourceIterable,
                                                   @NotNull ToLongFunction<? super S> longMapper)
  {
    return () -> longView(sourceIterable.iterator(), longMapper);
  }

  /**
   * View a generic iterator as if it is an iterator for primitive int values.
   * @param sourceIterator source iterator
   * @param intMapper   mapper which maps the elements of the source iterator to {@code int}
   * @return primitive int iterator
   * @param <S> source element type
   */
  @NotNull
  public static <S> PrimitiveIterator.OfInt intView(@NotNull Iterator<S> sourceIterator,
                                                    @NotNull ToIntFunction<? super S> intMapper)
  {
    return new PrimitiveIterator.OfInt()
    {
      @Override
      public int nextInt()
      {
        return intMapper.applyAsInt(sourceIterator.next());
      }

      @Override
      public boolean hasNext()
      {
        return sourceIterator.hasNext();
      }
    };
  }

  /**
   * View a generic iterable as if it is an iterable of primitive {@code int} values.
   * <p>
   * Because for-each loops will silently introduce boxing/unboxing it is recommended to
   * iterate over the returned int iterable with its looping functions:
   * {@link PrimitiveIntIterable#forEachInt(IntConsumer)} or 
   * {@link PrimitiveIntIterable#forEachIntFragile(FragileIntConsumer)} (if code throws
   * checked exceptions).
   * @param sourceIterable source iterable
   * @param intMapper   mapper from the incoming elements to int
   * @return primitive int iterable
   * @param <S> source element type
   */
  @NotNull
  public static <S> PrimitiveIntIterable intView(@NotNull Iterable<S> sourceIterable,
                                                 @NotNull ToIntFunction<? super S> intMapper)
  {
    return () -> intView(sourceIterable.iterator(), intMapper);
  }

  /**
   * View a generic iterator as if it is an iterator for primitive short values.
   * @param sourceIterator source iterator
   * @param shortMapper   mapper which maps the elements of the source iterator to {@code short}
   * @return primitive short iterator
   * @param <S> source element type
   */
  @NotNull
  public static <S> PrimitiveShortIterator shortView(@NotNull Iterator<S> sourceIterator,
                                                     @NotNull ToShortFunction<? super S> shortMapper)
  {
    return new PrimitiveShortIterator()
    {
      @Override
      public short nextShort()
      {
        return shortMapper.applyAsShort(sourceIterator.next());
      }

      @Override
      public boolean hasNext()
      {
        return sourceIterator.hasNext();
      }
    };
  }

  /**
   * View a generic iterable as if it is an iterable of primitive {@code short} values.
   * <p>
   * Because for-each loops will silently introduce boxing/unboxing it is recommended to
   * iterate over the returned short iterable with its looping functions:
   * {@link PrimitiveShortIterable#forEachShort(ShortConsumer)} or 
   * {@link PrimitiveShortIterable#forEachShortFragile(FragileShortConsumer)} (if code throws
   * checked exceptions).
   * @param sourceIterable source iterable
   * @param shortMapper   mapper from the incoming elements to short
   * @return primitive short iterable
   * @param <S> source element type
   */
  @NotNull
  public static <S> PrimitiveShortIterable shortView(@NotNull Iterable<S> sourceIterable,
                                                     @NotNull ToShortFunction<? super S> shortMapper)
  {
    return () -> shortView(sourceIterable.iterator(), shortMapper);
  }

  /**
   * View a generic iterator as if it is an iterator for primitive char values.
   * @param sourceIterator source iterator
   * @param charMapper   mapper which maps the elements of the source iterator to {@code char}
   * @return primitive char iterator
   * @param <S> source element type
   */
  @NotNull
  public static <S> PrimitiveCharIterator charView(@NotNull Iterator<S> sourceIterator,
                                                   @NotNull ToCharFunction<? super S> charMapper)
  {
    return new PrimitiveCharIterator()
    {
      @Override
      public char nextChar()
      {
        return charMapper.applyAsChar(sourceIterator.next());
      }

      @Override
      public boolean hasNext()
      {
        return sourceIterator.hasNext();
      }
    };
  }

  /**
   * View a generic iterable as if it is an iterable of primitive {@code char} values.
   * <p>
   * Because for-each loops will silently introduce boxing/unboxing it is recommended to
   * iterate over the returned char iterable with its looping functions:
   * {@link PrimitiveCharIterable#forEachChar(CharConsumer)} or 
   * {@link PrimitiveCharIterable#forEachCharFragile(FragileCharConsumer)} (if code throws
   * checked exceptions).
   * @param sourceIterable source iterable
   * @param charMapper   mapper from the incoming elements to char
   * @return primitive char iterable
   * @param <S> source element type
   */
  @NotNull
  public static <S> PrimitiveCharIterable charView(@NotNull Iterable<S> sourceIterable,
                                                   @NotNull ToCharFunction<? super S> charMapper)
  {
    return () -> charView(sourceIterable.iterator(), charMapper);
  }

  /**
   * View a generic iterator as if it is an iterator for primitive byte values.
   * @param sourceIterator source iterator
   * @param byteMapper   mapper which maps the elements of the source iterator to {@code byte}
   * @return primitive byte iterator
   * @param <S> source element type
   */
  @NotNull
  public static <S> PrimitiveByteIterator byteView(@NotNull Iterator<S> sourceIterator,
                                                   @NotNull ToByteFunction<? super S> byteMapper)
  {
    return new PrimitiveByteIterator()
    {
      @Override
      public byte nextByte()
      {
        return byteMapper.applyAsByte(sourceIterator.next());
      }

      @Override
      public boolean hasNext()
      {
        return sourceIterator.hasNext();
      }
    };
  }

  /**
   * View a generic iterable as if it is an iterable of primitive {@code byte} values.
   * <p>
   * Because for-each loops will silently introduce boxing/unboxing it is recommended to
   * iterate over the returned byte iterable with its looping functions:
   * {@link PrimitiveByteIterable#forEachByte(ByteConsumer)} or 
   * {@link PrimitiveByteIterable#forEachByteFragile(FragileByteConsumer)} (if code throws
   * checked exceptions).
   * @param sourceIterable source iterable
   * @param byteMapper   mapper from the incoming elements to byte
   * @return primitive byte iterable
   * @param <S> source element type
   */
  @NotNull
  public static <S> PrimitiveByteIterable byteView(@NotNull Iterable<S> sourceIterable,
                                                   @NotNull ToByteFunction<? super S> byteMapper)
  {
    return () -> byteView(sourceIterable.iterator(), byteMapper);
  }

  /**
   * View a generic iterator as if it is an iterator for primitive boolean values.
   * @param sourceIterator source iterator
   * @param booleanMapper   mapper which maps the elements of the source iterator to {@code boolean}
   * @return primitive boolean iterator
   * @param <S> source element type
   */
  @NotNull
  public static <S> PrimitiveBooleanIterator booleanView(@NotNull Iterator<S> sourceIterator,
                                                         @NotNull Predicate<? super S> booleanMapper)
  {
    return new PrimitiveBooleanIterator()
    {
      @Override
      public boolean nextBool()
      {
        return booleanMapper.test(sourceIterator.next());
      }

      @Override
      public boolean hasNext()
      {
        return sourceIterator.hasNext();
      }
    };
  }

  /**
   * View a generic iterable as if it is an iterable of primitive {@code boolean} values.
   * <p>
   * Because for-each loops will silently introduce boxing/unboxing it is recommended to
   * iterate over the returned boolean iterable with its looping functions:
   * {@link PrimitiveBooleanIterable#forEachBool(BooleanConsumer)} or
   * {@link PrimitiveBooleanIterable#forEachBoolFragile(FragileBooleanConsumer)} (if code throws
   * checked exceptions).
   * @param sourceIterable source iterable
   * @param booleanMapper   mapper from the incoming elements to boolean
   * @return primitive boolean iterable
   * @param <S> source element type
   */
  @NotNull
  public static <S> PrimitiveBooleanIterable booleanView(@NotNull Iterable<S> sourceIterable,
                                                         @NotNull Predicate<? super S> booleanMapper)
  {
    return () -> booleanView(sourceIterable.iterator(), booleanMapper);
  }

  /**
   * View an indexable.
   * Instead of {@link #map(java.lang.Iterable, Function)} this
   * will work transparently on the fly.
   * @param sourceIndexable source indesxable
   * @param mapper          mapper function which converts the source to the target type
   * @param <T>             target type
   * @param <S>             source type
   * @return mapped iterable
   * @see Indexable#view(Function)
   */
  @NotNull
  public static <T,S> Indexable<T> view(@NotNull Indexable<S> sourceIndexable,
                                        @NotNull Function<? super S, T> mapper)
  {
    return sourceIndexable.view(mapper);
  }

  /**
   * View a list iterator as if it belongs to another type.
   * Instead of {@link #map(java.util.Iterator, Function)} this
   * will work transparently on the fly without creating an intermediate list.
   * @param sourceIterator source iterator
   * @param mapper         mapper function which converts the source to the target type
   * @param <T>            target type
   * @param <S>            source type
   * @return mapped iterator
   */
  @NotNull
  public static <T,S> ListIterator<T> view(@NotNull ListIterator<S> sourceIterator,
                                           @NotNull Function<? super S, T> mapper)
  {
    return new ListIteratorConverter<>(sourceIterator, mapper);
  }

  /**
   * View an array as if it belongs to another iterable type.
   * <p>
   * Changes to the array during the use of the resulting iterable will
   * lead to undefined behavior.
   * @param mapper mapper function which converts the source to the target type
   * @param array  array to be view differently
   * @param <T>    target type
   * @param <S>    source type
   * @return iterable over array providing a different type
   * @see Indexable#viewArray(Object[])
   */
  @NotNull
  @SafeVarargs
  @SuppressWarnings("varargs")
  public static <T,S> Iterable<T> view(@NotNull  Function<? super S, T> mapper,
                                       @NotNull S ... array)
  {
    return view(asList(array), mapper);
  }

  /**
   * View a collection as if it has of another element type.
   * This will return an unmodifiable collection with a different element type.
   * It is considered useful in cases when you are in need of a readable collection,
   * but only have one with a slightly different type. Compared to {@link #map(Collection, Function)}
   * this does not copy the collection, instead it applies the mapper on each access. So
   * in general it is most useful when elements are rarely read, eg for iterating over them.
   * <p>
   *   Possible restrictions on the source collection (eg a set may contain elements only once,
   *   or it may be ordered) are no longer guaranteed by the returned collection. Use
   *   {@link #map(Collection, Iterable, Function)} where you can provide the required
   *   target collection in cases where these restrictions are required.
   * </p>
   * @param collection original collection
   * @param mapper     mapper which creates the element type of the result collection
   * @param <T>        target type
   * @param <S>        source type
   * @return unmodifiable collection with adapted element typee
   */
  @NotNull
  public static <T, S> Collection<T> viewCollection(@NotNull Collection<S> collection,
                                                    @NotNull Function<? super S, T> mapper)
  {
    return new AbstractCollection<T>()
    {
      @Override
      public int size()
      {
        return collection.size();
      }

      @Override
      public boolean isEmpty()
      {
        return collection.isEmpty();
      }

      @NotNull
      @Override
      public Iterator<T> iterator()
      {
        return Types.view(collection.iterator(), mapper);
      }
    };
  }

  /**
   * View a list as if it has another element type.
   * This will return an unmodifiable list with a different element type.
   * It is considered useful in cases when you are in need of a readable list,
   * but only have one with a slightly different type. Compared to {@link #map(Collection, Function)}
   * this does not copy the list, instead it applies the mapper on each access. So
   * in general it is most useful when elements are rarely read, eg for iterating over them.
   * @param list       original list
   * @param mapper     mapper which creates the element type of the result collection
   * @param <T>        target type
   * @param <S>        source type
   * @return unmodifiable collection with adapted element typee
   */
  @NotNull
  public static <T, S> List<T> viewList(@NotNull List<S> list,
                                        @NotNull Function<? super S, T> mapper)
  {
    return new AbstractList<T>()
    {
      @Override
      public int size()
      {
        return list.size();
      }

      @Override
      public boolean isEmpty()
      {
        return list.isEmpty();
      }

      @NotNull
      @Override
      public Iterator<T> iterator()
      {
        return Types.view(list.iterator(), mapper);
      }

      @NotNull
      @Override
      public ListIterator<T> listIterator()
      {
        return Types.view(list.listIterator(), mapper);
      }

      @Override
      public T get(int index)
      {
        return mapper.apply(list.get(index));
      }
    };
  }

  /**
   * Wrap an iterator to keep it from modifying the underlying collection.
   * <p>
   * The {@code Iterator} interface exports a method allowing to delete elements.
   * The wrapper returned from this method will throw an
   * {@link UnsupportedOperationException} for this method.
   *
   * @param iterator iterator to wrap
   * @param <T>      element type of iterator
   * @return iterator which cannot modify the underlying collection
   */
  public static <T> Iterator<T> unmodifiable(@NotNull final Iterator<T> iterator)
  {
    return new Iterator<T>()
    {
      @Override
      public boolean hasNext()
      {
        return iterator.hasNext();
      }

      @Override
      public T next()
      {
        return iterator.next();
      }

      @Override
      public void remove()
      {
        throw new UnsupportedOperationException("Not modifiable!");
      }
    };
  }

  /**
   * Wrap a list iterator to keep it from modifying the underlying collection.
   * <p>
   * The {@code ListIterator} interface exports methods allowing to modify the
   * underlying list. The wrapper returned from this method will throw an
   * {@link UnsupportedOperationException} for these methods.
   *
   * @param iterator iterator to wrap
   * @param <T>      element type of iterator
   * @return iterator which cannot modify the underlying collection
   */
  public static <T> ListIterator<T> unmodifiable(@NotNull final ListIterator<T> iterator)
  {
    return new ListIterator<T>()
    {
      @Override
      public boolean hasNext()
      {
        return iterator.hasNext();
      }

      @Override
      public T next()
      {
        return iterator.next();
      }

      @Override
      public boolean hasPrevious()
      {
        return iterator.hasPrevious();
      }

      @Override
      public T previous()
      {
        return iterator.previous();
      }

      @Override
      public int nextIndex()
      {
        return iterator.nextIndex();
      }

      @Override
      public int previousIndex()
      {
        return iterator.previousIndex();
      }

      @Override
      public void remove()
      {
        throw new UnsupportedOperationException("Not modifiable!");
      }

      @Override
      public void set(T t)
      {
        throw new UnsupportedOperationException("Not modifiable!");
      }

      @Override
      public void add(T t)
      {
        throw new UnsupportedOperationException("Not modifiable!");
      }
    };
  }

  /**
   * Create a factory from a type.
   * <p>
   * The factory will only create new objects if there is an argumentless public default constructor
   * for the given type. This method is thought for highly dynamic situations. In most cases it is
   * recommended to create a factory as an anonymous class:
   * <blockquote><pre>
   * // this is BAD style
   * Factory&lt;XY&gt; factory1 = Types.makeFactory(XY.class);
   *
   * // this is GOOD style
   * Factory&lt;XY&gt; factory2 = new Factory&lt;XY&gt;() {
   *   public XY create() { return new XY(); }
   * };
   * </pre></blockquote>
   * @param type type info
   * @param <T>  type
   * @return factory using reflection to create new elements
   */
  public static <T> Factory<T> makeFactory(final Class<T> type)
  {
    return () -> {
      try {
        return type.getDeclaredConstructor().newInstance();
      } catch (InstantiationException | IllegalAccessException | NoSuchMethodException | InvocationTargetException e) {
        throw new FactoryError(e);
      }
    };
  }

  /**
   * Create an array factory from a type.
   * <p>
   * This method is thought for highly dynamic situations. In most cases it is
   * recommended to create a factory as an anonymous class:
   * <blockquote><pre>
   * // this is BAD style
   * ArrayFactory&lt;XY&gt; factory1 = Types.makeArrayFactory(XY.class);
   *
   * // this is GOOD style
   * ArrayFactory&lt;XY&gt; factory2 = new ArrayFactory&lt;XY&gt;() {
   *   public XY[] create(int length) { return new XY[length]; }
   * };
   * </pre></blockquote>
   * @param type type info
   * @param <T>  type
   * @return factory using reflection to create new elements
   */
  @SuppressWarnings("unchecked")
  public static <T> ArrayFactory<T> makeArrayFactory(final Class<T> type)
  {
    return length -> (T[])Array.newInstance(type, length);
  }

  /**
   * Use a fragile function with 1 argument as a type converter.
   * @param function function to use
   * @param <T> target type
   * @param <S> source type
   * @return fragile type converter
   */
  @NotNull
  public static <T,S> FragileTypeConverter<T, S> asTypeConverter(@NotNull final FragileFunction1<T, ? extends Exception, S> function)
  {
    return object -> {
      try {
        return function.apply(object);
      } catch (Exception e) {
        throw new TypeConverterException(e);
      }
    };
  }

  /**
   * Return an empty iterator which does not iterate.
   * This is a low-cost method always returning the same object.
   * @param <T> iteration type (ignored as no iteration is done)
   * @return non-iterating iterator
   */
  @SuppressWarnings("unchecked") // as type is never used
  @NotNull
  public static <T> Iterator<T> emptyIterator()
  {
    return (Iterator<T>)EMPTY_ITERATOR;
  }

  /**
   * Return an empty list iterator which does not iterate.
   * This is a low-cost method always returning the same object.
   * @param <T> iteration element type (ignored as no iteration is done)
   * @return non-iterating list iterator
   */
  @NotNull
  @SuppressWarnings("unchecked") // as type is never used
  public static <T> ListIterator<T> emptyListIterator()
  {
    return (ListIterator<T>) EMPTY_LIST_ITERATOR;
  }

  /**
   * Return an empty enumeration which does not iterate.
   * This is a low-cost method returning always the same object.
   * @param <T> enumeration type (ignored as no iteration is done)
   * @return non-iterating enumeration
   */
  @SuppressWarnings("unchecked")
  @NotNull
  public static <T> Enumeration<T> emptyEnumeration()
  {
    return (Enumeration<T>)EMPTY_ENUMERATION;
  }

  /**
   * Return an empty iterable which does not iterate.
   * This is a low-cost method returning always the same object.
   * @param <T> iterable type (ignored as no iteration is done)
   * @return non-iterating iterable
   */
  @SuppressWarnings("unchecked")
  @NotNull
  public static <T> Iterable<T> emptyIterable()
  {
    return (Iterable<T>)EMPTY_ITERABLE;
  }

  /**
   * Get a copier function which creates an copy of its argument.
   * @param elemClass element class
   * @param <T> class type, has to implement {@link Copyable}
   * @return copier function
   */
  @SuppressWarnings("unchecked")
  @NotNull
  public static <T extends Copyable<T>> Function<T, T> getCopier(@NotNull Class<T> elemClass)
  {
    return (Function<T, T>)COPY_CONVERTER;
  }

  /**
   * Clone a collection.
   * @param sourceCollection source collection
   * @param cloner     cloner which clones one list element
   * @param <T>        list type
   * @return array list with cloned elements
   */
  @NotNull
  public static <T> ArrayList<T> clone(@NotNull Collection<T> sourceCollection,
                                       @NotNull Function<T, T> cloner)
  {
    return map(new ArrayList<T>(sourceCollection.size()),
               sourceCollection,
               cloner);
  }

  /**
   * Clone a countable.
   * @param sourceCountable source countable
   * @param cloner     cloner which clones one list element
   * @param <T>        list type
   * @return array list with cloned elements
   */
  @NotNull
  public static <T> ArrayList<T> clone(@NotNull Countable<T> sourceCountable,
                                       @NotNull Function<T, T> cloner)
  {
    return map(new ArrayList<T>(sourceCountable.size()),
               sourceCountable,
               cloner);
  }

  /**
   * Concatenate some iterables to a combined iterable.
   *
   * @param iterables  iterables to concatenate
   * @param <T> iteration element type
   * @return combined iterable which iterates over all the basic iterables one after the other
   */
  @SafeVarargs
  @NotNull
  @SuppressWarnings("varargs")
  public static <T> Iterable<T> concat(@NotNull Iterable<? extends T> ... iterables)
  {
    return new ConcatenatedIterables<T>(iterables);
  }

  /**
   * Concatenate some iterables to a combined iterable.
   *
   * @param iterables  iterables to concatenate
   * @param <T> iteration element type
   * @return combined iterable which iterates over all the basic iterables one after the other
   */
  @NotNull
  @SuppressWarnings("varargs")
  public static <T> Iterable<T> concatIterables(@NotNull Iterable<? extends Iterable<? extends T>> iterables)
  {
    return new ConcatenatedIterables<T>(iterables);
  }

  /**
   * Concatenate some iterators.
   *
   * @param iterators iterators to concatenate
   * @param <T> iteration element type
   * @return combined iterator which iterates over all the basic iterators one after the other
   */
  @SafeVarargs
  @NotNull
  @SuppressWarnings("varargs")
  public static <T> Iterator<T> concat(@NotNull Iterator<? extends T> ... iterators)
  {
    return new ConcatenatedIterators<>(iterators);
  }

  /**
   * Concatenate some iterators.
   * @param iterators iterators to concatenate
   * @return combined iterator which iterates over all the basic iterators one after the other
   * @param <T> iteration element type
   */
  @NotNull
  public static <T> Iterator<T> concatIterators(@NotNull Iterable<? extends Iterator<? extends T>> iterators)
  {
    return new ConcatenatedIterators<>(iterators);
  }

  /**
   * Make sure that something is not {@code null}.
   * Use a default value if necessary.
   * @param item         item which may be {@code null}
   * @param defaultValue default value substituted if item is {@code null}
   * @param <T> item type
   * @param <D> default value type
   * @return {@code item} if it is not {@code null},
   *         {@code defaultValue} otherwise
   */
  @NotNull
  public static <T, D extends T> T notNull(@Nullable T item, @NotNull D defaultValue)
  {
    return item == null
            ? defaultValue
            : item;
  }

  /**
   * Make sure a string is not {@code null}.
   * Use the empty string if necessary.
   * @param str string which might be {@code null}
   * @return the string if it is not {@code null}, the empty string otherwise
   */
  @NotNull
  public static String notNull(@Nullable String str)
  {
    return notNull(str, Empty.STRING);
  }

  /**
   * Make sure that something is not {@code null},
   * but postpone the fallback creation.
   * This is similar to {@link #notNull(Object, Object)},
   * but is creating the second {@code defaultObject} parameter
   * only when required. This is preferable if creating the object
   * is an expensive process.
   * @param item     item which is tested for being {@code null}
   * @param provider supplier asked to provide a fallback when {@code item} is {@code null},
   *                 required to return non-null object!
   * @param <T> type of nullable value
   * @return non-null value, either {@code item} if it is not {@code null},
   *         or a fallback created by {@code provider}
   * @throws IllegalArgumentException if provider creates {@code null}
   * @see #notNullOrN(Object, Supplier)
   */
  @NotNull
  public static <T> T notNullOr(@Nullable T item,
                                @NotNull Supplier<? extends T> provider)
  {
    if (item != null) {
      return item;
    }
    final T fallback = provider.get();
    if (fallback == null) {
      throw new IllegalArgumentException("provider for non-null object returned null!");
    }
    return fallback;
  }

  /**
   * Try to determine a fallback when a value is {@code null}.
   * <p>
   * This is useful in cases where the fallback creation is expensive.
   * In difference to {@link #notNullOr(Object, Supplier)} this accepts
   * the fallback provider to return {@code null}. Thus, it is necessary
   * to check the result of this method for being {@code null}.
   *
   * @param item     item which is tested for being {@code null}
   * @param provider supplier asked to provide a fallback when {@code item} is {@code null},
   *                 allowed to reutn
   * @param <T> type of nullable value
   * @return either {@code item} if it is not {@code null},
   *         or a fallback created by {@code provider}
   */
  @Nullable
  public static <T> T notNullOrN(@Nullable T item,
                                 @NotNull Supplier<? extends T> provider)
  {
    return item != null
            ? item
            : provider.get();
  }

  /**
   * Make sure a string is neither {@code null} nor empty.
   * @param str      string which might be {@code null} or empty
   * @param fallback fallback return when string is {@code null} or empty,
   *                 should be not empty although this is not checked here
   * @return {@code fallback} when {@code str} is {@code null} or empty,
   *         or {@code str} otherwise
   */
  @NotNull
  public static String notEmpty(@Nullable String str, @NotNull String fallback)
  {
    return str == null  ||  str.isEmpty()
            ? fallback
            : str;
  }

  /**
   * Check whether a string is empty or {@code null}.
   * @param str the string to check
   * @return {@code true}: if the string is empty or null<br>
   *         {@code false}: otherwise
   */
  public static boolean isEmptyOrNull(@Nullable String str)
  {
    return str == null  ||  str.isEmpty();
  }

  /**
   * Check whether a string is empty, contains only whitespace, or is {@code null}.
   * @param str the string to check
   * @return {@code true}: if the string contains only whitespace, is empty, or null<br>
   *         {@code false}: otherwise
   */
  public static boolean isWhiteOrNull(@Nullable String str)
  {
    return str == null  ||  str.trim().isEmpty();
  }

  /**
   * Require an item to be not null.
   * @param item     item to check
   * @param thrower  procedure called when item is {@code null},
   *                 expected to throw an appropriate unchecked exception
   * @param <T> item type
   * @return non-null item
   * @throws NullPointerException if thrower does not throw anything
   * @throws RuntimeException anything thrower will throw
   */
  @NotNull
  public static <T> T requireNonNull(@Nullable T item,
                                     @NotNull Procedure0 thrower)
  {
    if (item == null) {
      thrower.apply();
      // as the above line is expected to throw something we should not get here
      throw new NullPointerException();
    }
    return item;
  }

  /** True predicate for optimized chaining. */
  private static final Predicate<Object> PREDICATE_TRUE =
          new Predicate<Object>()
          {
            @Override
            public boolean test(Object o)
            {
              return true;
            }

            @Override
            public Predicate<Object> and(@NotNull Predicate<? super Object> other)
            {
              return other;
            }

            @Override
            public Predicate<Object> negate()
            {
              return PREDICATE_FALSE;
            }

            @Override
            public Predicate<Object> or(@NotNull Predicate<? super Object> other)
            {
              return this;
            }
          };
  /** False predicate for optimized chaining. */
  private static final Predicate<Object> PREDICATE_FALSE =
          new Predicate<Object>()
          {
            @Override
            public boolean test(Object o)
            {
              return false;
            }

            @Override
            public Predicate<Object> and(@NotNull Predicate<? super Object> other)
            {
              return this;
            }

            @Override
            public Predicate<Object> negate()
            {
              return PREDICATE_TRUE;
            }

            @Override
            public Predicate<Object> or(@NotNull Predicate<? super Object> other)
            {
              return other;
            }
          };
  /** True bi predicate for optimized chaining. */
  private static final BiPredicate<Object, Object> BI_PREDICATE_TRUE =
          new BiPredicate<Object, Object>()
          {
            @Override
            public boolean test(Object o, Object o2)
            {
              return true;
            }

            @Override
            public BiPredicate<Object, Object> and(BiPredicate<? super Object, ? super Object> other)
            {
              return other;
            }

            @Override
            public BiPredicate<Object, Object> negate()
            {
              return BI_PREDICATE_FALSE;
            }

            @Override
            public BiPredicate<Object, Object> or(BiPredicate<? super Object, ? super Object> other)
            {
              return this;
            }
          };
  /** False bi predicate for optimized chaining. */
  private static final BiPredicate<Object, Object> BI_PREDICATE_FALSE =
          new BiPredicate<Object, Object>()
          {
            @Override
            public boolean test(Object o, Object o2)
            {
              return false;
            }

            @Override
            public BiPredicate<Object, Object> and(BiPredicate<? super Object, ? super Object> other)
            {
              return this;
            }

            @Override
            public BiPredicate<Object, Object> negate()
            {
              return BI_PREDICATE_TRUE;
            }

            @Override
            public BiPredicate<Object, Object> or(BiPredicate<? super Object, ? super Object> other)
            {
              return other;
            }
          };


  /**
   * Return a predicate which returns always {@code true}.
   * The returned predicate optimizes predicate chaining.
   * @param <T> type of predicate parameter
   * @return always true predicate
   */
  @NotNull
  @SuppressWarnings("unchecked") // as the returned predicate completely ignores the argument
  public static <T> Predicate<T> truePredicate()
  {
    return (Predicate<T>)PREDICATE_TRUE;
  }

  /**
   * Return a predicate which returns always {@code false}.
   * The returned predicate optimizes predicate chaining.
   * @param <T> type of predicate parameter
   * @return always true predicate
   */
  @NotNull
  @SuppressWarnings("unchecked") // as the returned predicate completely ignores the argument
  public static <T> Predicate<T> falsePredicate()
  {
    return (Predicate<T>)PREDICATE_FALSE;
  }

  /**
   * Return a bi-predicate which returns always {@code true}.
   * The returned predicate optimizes predicate chaining.
   * @param <T1> type of first parameter
   * @param <T2> type of second parameter
   * @return always true bi-predicate
   */
  @NotNull
  @SuppressWarnings("unchecked") // as the returned predicate completely ignores the argument
  public static <T1, T2> BiPredicate<T1, T2> trueBiPredicate()
  {
    return (BiPredicate<T1, T2>)BI_PREDICATE_TRUE;
  }

  /**
   * Return a bi-predicate which returns always {@code false}.
   * The returned predicate optimizes predicate chaining.
   * @param <T1> type of first parameter
   * @param <T2> type of second parameter
   * @return always false bi-predicate
   */
  @NotNull
  @SuppressWarnings("unchecked") // as the returned predicate completely ignores the argument
  public static <T1, T2> BiPredicate<T1, T2> falseBiPredicate()
  {
    return (BiPredicate<T1, T2>)BI_PREDICATE_FALSE;
  }

  /**
   * Return an indexable from a primitive integer array.
   * @param array array on which the indexable is based
   * @return indexable from array
   * @see IntIndexable
   */
  @NotNull
  public static Indexable<Integer> indexable(@NotNull int ... array)
  {
    return new Indexable.Base<Integer>()
    {
      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public Integer get(int index)
      {
        return array[index];
      }
    };
  }

  /**
   * Return an indexable from a primitive short integer array.
   * @param array array on which the indexable is based
   * @return indexable from array
   * @see ShortIndexable
   */
  @NotNull
  public static Indexable<Short> indexable(@NotNull short ... array)
  {
    return new Indexable.Base<Short>()
    {
      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public Short get(int index)
      {
        return array[index];
      }
    };
  }

  /**
   * Return an indexable from a primitive long integer array.
   * @param array array on which the indexable is based
   * @return indexable from array
   * @see LongIndexable
   */
  @NotNull
  public static Indexable<Long> indexable(@NotNull long ... array)
  {
    return new Indexable.Base<Long>()
    {
      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public Long get(int index)
      {
        return array[index];
      }
    };
  }

  /**
   * Return an indexable from a primitive byte integer array.
   * @param array array on which the indexable is based
   * @return indexable from array
   * @see ByteIndexable
   */
  @NotNull
  public static Indexable<Byte> indexable(@NotNull byte ... array)
  {
    return new Indexable.Base<Byte>()
    {
      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public Byte get(int index)
      {
        return array[index];
      }
    };
  }

  /**
   * Return an indexable from a primitive double array.
   * @param array array on which the indexable is based
   * @return indexable from array
   * @see DoubleIndexable
   */
  @NotNull
  public static Indexable<Double> indexable(@NotNull double ... array)
  {
    return new Indexable.Base<Double>()
    {
      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public Double get(int index)
      {
        return array[index];
      }
    };
  }

  /**
   * Return an indexable from a primitive float array.
   * @param array array on which the indexable is based
   * @return indexable from array
   * @see FloatIndexable
   */
  @NotNull
  public static Indexable<Float> indexable(@NotNull float ... array)
  {
    return new Indexable.Base<Float>()
    {
      @Override
      public int size()
      {
        return array.length;
      }

      @Override
      public Float get(int index)
      {
        return array[index];
      }
    };
  }

  /**
   * Get the maximal value.
   * @param values values to check, at least 1
   * @return maximal value
   * @throws IllegalArgumentException if there are no values
   */
  public static double maxDouble(@NotNull double ... values)
  {
    if (values.length == 0) {
      throw new IllegalArgumentException("Need at least 1 value!");
    }
    double maxValue = values[0];
    for (int v = values.length - EMPTY_HASH; v > 0; --v) {
      maxValue = Math.max(maxValue, values[v]);
    }
    return maxValue;
  }

  /**
   * Get the maximal double value from a double view of an iterable.
   * @param iterable  iterable (must not be empty!)
   * @param toDouble  converter from the element type of the iterable to double
   * @param <T> iterable element type
   * @return maximal double value
   * @throws IllegalArgumentException if iterable is empty
   */
  public static <T> double maxDouble(@NotNull Iterable<T> iterable,
                                     @NotNull ToDoubleFunction<? super T> toDouble)
  {
    final Iterator<T> it = iterable.iterator();
    if (!it.hasNext()) {
      throw new IllegalArgumentException("Need at least 1 value, but iterable is empty!");
    }
    double maxValue = toDouble.applyAsDouble(it.next());
    while (it.hasNext()) {
      maxValue = Math.max(maxValue, toDouble.applyAsDouble(it.next()));
    }
    return maxValue;
  }

  /**
   * Get the maximal double value for an iterable of numbers.
   * @param iterable  iterable (must not be empty!)
   * @return maximal double value
   * @throws IllegalArgumentException if iterable is empty
   */
  public static double maxDouble(@NotNull Iterable<? extends Number> iterable)
  {
    final Iterator<? extends Number> it = iterable.iterator();
    if (!it.hasNext()) {
      throw new IllegalArgumentException("Need at least 1 value, but iterable is empty!");
    }
    double maxValue = it.next().doubleValue();
    while (it.hasNext()) {
      maxValue = Math.max(maxValue, it.next().doubleValue());
    }
    return maxValue;
  }

  /**
   * Get the minimal value.
   * @param values values to check, at least 1
   * @return minimal value
   * @throws IllegalArgumentException if there are no values
   */
  public static double minDouble(@NotNull double ... values)
  {
    if (values.length == 0) {
      throw new IllegalArgumentException("Need at least 1 value!");
    }
    double minValue = values[0];
    for (int v = values.length - EMPTY_HASH; v > 0; --v) {
      minValue = Math.min(minValue, values[v]);
    }
    return minValue;
  }

  /**
   * Get the minimal double value from a double view of an iterable.
   * @param iterable  iterable (must not be empty!)
   * @param toDouble  converter from the element type of the iterable to double
   * @param <T> iterable element type
   * @return minimal double value
   * @throws IllegalArgumentException if iterable is empty
   */
  public static <T> double minDouble(@NotNull Iterable<T> iterable,
                                     @NotNull ToDoubleFunction<? super T> toDouble)
  {
    final Iterator<T> it = iterable.iterator();
    if (!it.hasNext()) {
      throw new IllegalArgumentException("Need at least 1 value, but iterable is empty!");
    }
    double minValue = toDouble.applyAsDouble(it.next());
    while (it.hasNext()) {
      minValue = Math.min(minValue, toDouble.applyAsDouble(it.next()));
    }
    return minValue;
  }

  /**
   * Get the minimal double value for an iterable of numbers.
   * @param iterable  iterable (must not be empty!)
   * @return minimal double value
   * @throws IllegalArgumentException if iterable is empty
   */
  public static double minDouble(@NotNull Iterable<? extends Number> iterable)
  {
    final Iterator<? extends Number> it = iterable.iterator();
    if (!it.hasNext()) {
      throw new IllegalArgumentException("Need at least 1 value, but iterable is empty!");
    }
    double minValue = it.next().doubleValue();
    while (it.hasNext()) {
      minValue = Math.min(minValue, it.next().doubleValue());
    }
    return minValue;
  }

  /**
   * Get the maximal value.
   * @param values values to check, at least 1
   * @return maximal value
   * @throws IllegalArgumentException if there are no values
   */
  public static float maxFloat(@NotNull float ... values)
  {
    if (values.length == 0) {
      throw new IllegalArgumentException("Need at least 1 value!");
    }
    float maxValue = values[0];
    for (int v = values.length - EMPTY_HASH; v > 0; --v) {
      maxValue = Math.max(maxValue, values[v]);
    }
    return maxValue;
  }

  /**
   * Get the minimal value.
   * @param values values to check, at least 1
   * @return minimal value
   * @throws IllegalArgumentException if there are no values
   */
  public static float minFloat(@NotNull float ... values)
  {
    if (values.length == 0) {
      throw new IllegalArgumentException("Need at least 1 value!");
    }
    float minValue = values[0];
    for (int v = values.length - EMPTY_HASH; v > 0; --v) {
      minValue = Math.min(minValue, values[v]);
    }
    return minValue;
  }

  /**
   * Get the maximal value.
   * @param values values to check, at least 1
   * @return maximal value
   * @throws IllegalArgumentException if there are no values
   */
  public static long maxLong(@NotNull long ... values)
  {
    if (values.length == 0) {
      throw new IllegalArgumentException("Need at least 1 value!");
    }
    long maxValue = values[0];
    for (int v = values.length - EMPTY_HASH; v > 0; --v) {
      maxValue = Math.max(maxValue, values[v]);
    }
    return maxValue;
  }

  /**
   * Get the minimal value.
   * @param values values to check, at least 1
   * @return minimal value
   * @throws IllegalArgumentException if there are no values
   */
  public static long minLong(@NotNull long ... values)
  {
    if (values.length == 0) {
      throw new IllegalArgumentException("Need at least 1 value!");
    }
    long minValue = values[0];
    for (int v = values.length - EMPTY_HASH; v > 0; --v) {
      minValue = Math.min(minValue, values[v]);
    }
    return minValue;
  }

  /**
   * Get the maximal value.
   * @param values values to check, at least 1
   * @return maximal value
   * @throws IllegalArgumentException if there are no values
   */
  public static int maxInt(@NotNull int ... values)
  {
    if (values.length == 0) {
      throw new IllegalArgumentException("Need at least 1 value!");
    }
    int maxValue = values[0];
    for (int v = values.length - EMPTY_HASH; v > 0; --v) {
      maxValue = Math.max(maxValue, values[v]);
    }
    return maxValue;
  }

  /**
   * Get the minimal value.
   * @param values values to check, at least 1
   * @return minimal value
   * @throws IllegalArgumentException if there are no values
   */
  public static int minInt(@NotNull int ... values)
  {
    if (values.length == 0) {
      throw new IllegalArgumentException("Need at least 1 value!");
    }
    int minValue = values[0];
    for (int v = values.length - EMPTY_HASH; v > 0; --v) {
      minValue = Math.min(minValue, values[v]);
    }
    return minValue;
  }

  /**
   * Get the maximal value.
   * @param values values to check, at least 1
   * @return maximal value
   * @throws IllegalArgumentException if there are no values
   */
  public static short maxShort(@NotNull short ... values)
  {
    if (values.length == 0) {
      throw new IllegalArgumentException("Need at least 1 value!");
    }
    int maxValue = values[0];
    for (int v = values.length - EMPTY_HASH; v > 0; --v) {
      maxValue = Math.max(maxValue, values[v]);
    }
    return (short)maxValue;
  }

  /**
   * Get the minimal value.
   * @param values values to check, at least 1
   * @return minimal value
   * @throws IllegalArgumentException if there are no values
   */
  public static short minShort(@NotNull short ... values)
  {
    if (values.length == 0) {
      throw new IllegalArgumentException("Need at least 1 value!");
    }
    int minValue = values[0];
    for (int v = values.length - EMPTY_HASH; v > 0; --v) {
      minValue = Math.min(minValue, values[v]);
    }
    return (short)minValue;
  }

  /**
   * Get the maximal value.
   * @param values values to check, at least 1
   * @return maximal value
   * @throws IllegalArgumentException if there are no values
   */
  public static byte maxByte(@NotNull byte ... values)
  {
    if (values.length == 0) {
      throw new IllegalArgumentException("Need at least 1 value!");
    }
    int maxValue = values[0];
    for (int v = values.length - EMPTY_HASH; v > 0; --v) {
      maxValue = Math.max(maxValue, values[v]);
    }
    return (byte)maxValue;
  }

  /**
   * Get the minimal value.
   * @param values values to check, at least 1
   * @return minimal value
   * @throws IllegalArgumentException if there are no values
   */
  public static byte minByte(@NotNull byte ... values)
  {
    if (values.length == 0) {
      throw new IllegalArgumentException("Need at least 1 value!");
    }
    int minValue = values[0];
    for (int v = values.length - EMPTY_HASH; v > 0; --v) {
      minValue = Math.min(minValue, values[v]);
    }
    return (byte)minValue;
  }

  /**
   * Get the maximal value.
   * @param values values to check, at least 1
   * @return maximal value
   * @throws IllegalArgumentException if there are no values
   */
  public static char maxChar(@NotNull char ... values)
  {
    if (values.length == 0) {
      throw new IllegalArgumentException("Need at least 1 value!");
    }
    int maxValue = values[0];
    for (int v = values.length - EMPTY_HASH; v > 0; --v) {
      maxValue = Math.max(maxValue, values[v]);
    }
    return (char)maxValue;
  }

  /**
   * Get the minimal value.
   * @param values values to check, at least 1
   * @return minimal value
   * @throws IllegalArgumentException if there are no values
   */
  public static char minChar(@NotNull char ... values)
  {
    if (values.length == 0) {
      throw new IllegalArgumentException("Need at least 1 value!");
    }
    int minValue = values[0];
    for (int v = values.length - EMPTY_HASH; v > 0; --v) {
      minValue = Math.min(minValue, values[v]);
    }
    return (char)minValue;
  }

  /**
   * Get the minimal value from an array of values.
   * @param comparator comparator defining how values are ordered
   * @param values     values, at least 1
   * @param <T>        value type
   * @return minimal value
   */
  @NotNull
  @SafeVarargs
  @SuppressWarnings("varargs")
  public static <T> T min(@NotNull Comparator<? super T> comparator,
                          @NotNull T ... values)
  {
    if (values.length == 0) {
      throw new IllegalArgumentException("Need at least 1 value!");
    }
    return min(comparator, asList(values));
  }

  /**
   * Get the minimal value from an array of comparable values.
   * @param values     values, at least 1
   * @param <T>        value type
   * @return minimal value
   */
  @NotNull
  @SafeVarargs
  @SuppressWarnings("varargs")
  public static <T extends Comparable<? super T>> T min(@NotNull T ... values)
  {
    if (values.length == 0) {
      throw new IllegalArgumentException("Need at least 1 value!");
    }
    return min(Comparator.naturalOrder(), asList(values));
  }

  /**
   * Get the minimal value from some values.
   * @param comparator comparator defining how values are ordered
   * @param values     values, at least 1
   * @param <T>        value type
   * @return minimal value
   */
  @NotNull
  public static <T> T min(@NotNull Comparator<? super T> comparator,
                          @NotNull Iterable<T> values)
  {
    T minValue = null;
    for (T value : values) {
      if (minValue != null) {
        if (comparator.compare(minValue, value) > 0)  {
          minValue = value;
        }
      }
      else {
        minValue = value;
      }
    }
    if (minValue == null) {
      throw new IllegalArgumentException("Need at least 1 value!");
    }
    return minValue;
  }

  /**
   * Get the minimal value from some comparable values.
   * @param values     values, at least 1
   * @param <T>        value type
   * @return minimal value
   */
  @NotNull
  public static <T extends Comparable<? super T>> T min(@NotNull Iterable<T> values)
  {
    return min(Comparator.naturalOrder(), values);
  }

  /**
   * Get the maximal value from an array of values.
   * @param comparator comparator defining how values are ordered
   * @param values     values, at least 1
   * @param <T>        value type
   * @return maximal value
   */
  @NotNull
  @SafeVarargs
  @SuppressWarnings("varargs")
  public static <T> T max(@NotNull Comparator<? super T> comparator,
                          T ... values)
  {
    if (values.length == 0) {
      throw new IllegalArgumentException("Need at least 1 value!");
    }
    return max(comparator, asList(values));
  }

  /**
   * Get the maximal value from an array of comparable values.
   * @param values     values, at least 1
   * @param <T>        value type
   * @return maximal value
   */
  @NotNull
  @SafeVarargs
  @SuppressWarnings("varargs")
  public static <T extends Comparable<? super T>> T max(T ... values)
  {
    if (values.length == 0) {
      throw new IllegalArgumentException("Need at least 1 value!");
    }
    return max(Comparator.naturalOrder(), asList(values));
  }

  /**
   * Get the maximal value from some values.
   * @param comparator comparator defining how values are ordered
   * @param values     values, at least 1
   * @param <T>        value type
   * @return maximal value
   */
  @NotNull
  public static <T> T max(@NotNull Comparator<? super T> comparator,
                          @NotNull Iterable<T> values)
  {
    T maxValue = null;
    for (T value : values) {
      if (maxValue != null) {
        if (comparator.compare(maxValue, value) < 0)  {
          maxValue = value;
        }
      }
      else {
        maxValue = value;
      }
    }
    if (maxValue == null) {
      throw new IllegalArgumentException("Need at least 1 value!");
    }
    return maxValue;
  }

  /**
   * Get the maximal value from some comparable values.
   * @param values     values, at least 1
   * @param <T>        value type
   * @return maximal value
   */
  @NotNull
  public static <T extends Comparable<? super T>> T max(@NotNull Iterable<T> values)
  {
    return max(Comparator.naturalOrder(), values);
  }

  /**
   * Fill an array list with a number of elements.
   * After the call the list will have lost possible previous elements,
   * and be filled with the mapped incoming elements.
   * The list will also be {@link ArrayList#trimToSize()}.
   * @param list     list to be filled
   * @param elements elements to fill in
   * @param mapper   mapper from incoming elements to list elements
   * @param <T> element type of result list
   * @param <S> element type of incoming elements
   */
  public static <T,S> void fillTight(@NotNull ArrayList<T> list,
                                     @NotNull Iterable<S> elements,
                                     @NotNull Function<? super S, ? extends T> mapper)
  {
    list.clear();
    Types.map(list, elements, mapper);
    list.trimToSize();
  }

  /**
   * Fill an array list with a number of alements.
   * After the call the list will have lost possible previous elements,
   * and be filled with the incoming elements.
   * The list will also be {@link ArrayList#trimToSize()}.
   * @param list     list to be filled
   * @param elements elements to fill in
   * @param <T> element type of result list, incoming iterable may extend this
   */
  public static <T> void fillTight(@NotNull ArrayList<T> list,
                                   @NotNull Iterable<? extends T> elements)
  {
    fillTight(list, elements, e -> e);
  }

  /**
   * Fill an array list with a number of elements.
   * After the call the list will have lost possible previous elements,
   * and be filled with the mapped incoming elements.
   * The list will also be {@link ArrayList#trimToSize()}.
   * @param list     list to be filled
   * @param elements elements to fill in
   * @param mapper   mapper from incoming elements to list elements
   * @param <T> element type of result list
   * @param <S> element type of incoming elements
   */
  public static <T,S> void fillTight(@NotNull ArrayList<T> list,
                                     @NotNull Collection<S> elements,
                                     @NotNull Function<? super S, ? extends T> mapper)
  {
    list.clear();
    list.ensureCapacity(elements.size());
    Types.map(list, elements, mapper);
    list.trimToSize();
  }

  /**
   * Fill an array list with a number of elements.
   * After the call the list will have lost possible previous elements,
   * and be filled with the mapped incoming elements.
   * The list will also be {@link ArrayList#trimToSize()}.
   * @param list     list to be filled
   * @param elements elements to fill in
   * @param mapper   mapper from incoming elements to list elements
   * @param <T> element type of result list
   * @param <S> element type of incoming elements
   */
  public static <T,S> void fillTight(@NotNull ArrayList<T> list,
                                     @NotNull Countable<S> elements,
                                     @NotNull Function<? super S, ? extends T> mapper)
  {
    list.clear();
    list.ensureCapacity(elements.size());
    Types.map(list, elements, mapper);
    list.trimToSize();
  }

  /**
   * Fill an array list with a number of alements.
   * After the call the list will have lost possible previous elements,
   * and be filled with the incoming elements.
   * The list will also be {@link ArrayList#trimToSize()}.
   * @param list     list to be filled
   * @param elements elements to fill in
   * @param <T> element type of result list, incoming iterable may extend this
   */
  public static <T> void fillTight(@NotNull ArrayList<T> list,
                                   @NotNull Collection<? extends T> elements)
  {
    fillTight(list, elements, e -> e);
  }

  /**
   * Fill an array list with a number of alements.
   * After the call the list will have lost possible previous elements,
   * and be filled with the incoming elements.
   * The list will also be {@link ArrayList#trimToSize()}.
   * @param list     list to be filled
   * @param elements elements to fill in
   * @param <T> element type of result list, incoming iterable may extend this
   */
  public static <T> void fillTight(@NotNull ArrayList<T> list,
                                   @NotNull Countable<? extends T> elements)
  {
    fillTight(list, elements, e -> e);
  }

  /**
   * Fill part of an array with elements provided by a supplier.
   * @param array    array to fill
   * @param start    first index to be filled
   * @param length   number of elements filled from {@code start}
   * @param supplier supplier called for each element to be filled
   * @param <T> array type
   */
  public static <T> void fillArray(@NotNull T[] array,
                                   int start, int length,
                                   @NotNull Supplier<? extends T> supplier)
  {
    if (length < 0) {
      throw new InvalidParameterException("length has to be non-negative!");
    }
    if (start < 0  ||  start > array.length - length) {
      throw new ArrayIndexOutOfBoundsException();
    }

    final int end = start + length;
    for (int i = start;  i < end;  ++i) {
      array[i] = supplier.get();
    }
  }

  /**
   * Fill an array with elements provided by a supplier.
   * @param array    array to fill
   * @param supplier supplier called for each element to be filled
   * @param <T> array type
   */
  public static <T> void fillArray(@NotNull T[] array,
                                   @NotNull Supplier<? extends T> supplier)
  {
    fillArray(array, 0, array.length, supplier);
  }

  /**
   * Compare two iterables in lexical order.
   * This means that the first difference between the two iterables
   * determines the order. If one is a prefix of the other
   * it is considered first in order, and only if both
   * iterables contain the same number of elements which
   * are equal (as given by order) {@code 0} is returned.
   * @param it1         first iterable
   * @param it2         second iterable
   * @param comparator  comparator for the elements of the iterables
   * @param <T>         common element type
   * @return less than {@code 0} if {@code it1 < it2}, greater than
   *         {@code 0} if {@code it1 > it2}, and {@code 0} if
   *         both sequences are considered equal
   */
  public static <T> int lexicalCompare(@NotNull Iterable<? extends T> it1,
                                       @NotNull Iterable<? extends T> it2,
                                       @NotNull Comparator<T> comparator)
  {
    return lexicalCompare(it1.iterator(), it2.iterator(), comparator);
  }

  /**
   * Compare two iterables in lexical natural order.
   * This means that the first difference between the two iterables
   * determines the order. If one is a prefix of the other
   * it is considered first in order, and only if both
   * iterables contain the same number of elements which
   * are equal (as given by order) {@code 0} is returned.
   * @param it1  first iterable of comparable elements
   * @param it2  second iterable of comparable elements
   * @param <T>  comparable element type
   * @return less than {@code 0} if {@code it1 < it2}, greater than
   *         {@code 0} if {@code it1 > it2}, and {@code 0} if
   *         both sequences are considered equal
   */
  public static <T extends Comparable<T>> int lexicalCompare(@NotNull Iterable<? extends T> it1,
                                                             @NotNull Iterable<? extends T> it2)
  {
    return lexicalCompare(it1, it2, naturalOrder());
  }

  /**
   * Compare two iterators in lexical order.
   * This means that the first difference between the two iterables
   * determines the order. If one is a prefix of the other
   * it is considered first in order, and only if both
   * iterables contain the same number of elements which
   * are equal (as given by order) {@code 0} is returned.
   * @param it1         first iterator, placed after the first difference on return
   * @param it2         second iterator, placed after the first difference on return
   * @param comparator  comparator for the elements of the iterators
   * @param <T>         common element type
   * @return less than {@code 0} if {@code it1 < it2}, greater than
   *         {@code 0} if {@code it1 > it2}, and {@code 0} if
   *         both sequences are considered equal
   */
  public static <T> int lexicalCompare(@NotNull Iterator<? extends T> it1,
                                       @NotNull Iterator<? extends T> it2,
                                       @NotNull Comparator<T> comparator)
  {
    while (it1.hasNext()  &&  it2.hasNext()) {
      final int cmp = comparator.compare(it1.next(), it2.next());
      if (cmp != 0) {
        return cmp;
      }
    }
    // now look for the shorter
    return it1.hasNext()
            ? 1              // it2 is shorter
            : (it2.hasNext()
                       ? -1  // it1 is shorter
                       : 0); // both have equal length
  }

  /**
   * Compare two iterators in lexical natural order.
   * This means that the first difference between the two iterators
   * determines the order. If one is a prefix of the other
   * it is considered first in order, and only if both
   * iterables contain the same number of elements which
   * are equal (as given by order) {@code 0} is returned.
   * @param it1         first iterator, placed after the first difference on return
   * @param it2         second iterator, placed after the first difference on return
   * @param <T>  comparable element type
   * @return less than {@code 0} if {@code it1 < it2}, greater than
   *         {@code 0} if {@code it1 > it2}, and {@code 0} if
   *         both sequences are considered equal
   */
  public static <T extends Comparable<T>> int lexicalCompare(@NotNull Iterator<? extends T> it1,
                                                             @NotNull Iterator<? extends T> it2)
  {
    return lexicalCompare(it1, it2, naturalOrder());
  }

  /**
   * Calculate a hash code for a generic iterable.
   * This methods sums up the hash codes of all elements of the iterable.
   * This implementation follows {@link Arrays#hashCode(Object[])},
   * so for an array with with the same elements as the given iterable
   * the results are the same.
   * @param iterable iterable for which the hash code is required
   * @return hash code
   */
  public static int hash(@NotNull Iterable<?> iterable)
  {
    // this follows Arrays.hashCode()
    int result = EMPTY_HASH;

    for (Object elem : iterable) {
      result = 31 * result + Objects.hash(elem);
    }

    return result;
  }

  /**
   * Find the first key in a map which the given value checker accepts.
   * @param map          map to search
   * @param valueChecker value checker
   * @param <K>          map key type
   * @param <V>          map value type
   * @return first matching key, or {@code null} if no value matches
   */
  @Nullable
  public static <K, V> K findAnyKey(@NotNull Map<K, V> map,
                                    @NotNull Predicate<? super V> valueChecker)
  {
    final Map.Entry<K, V> entry = findAnyEntry(map,
                                               (k, v) -> valueChecker.test(v));
    return entry != null
            ? entry.getKey()
            : null;
  }

  /**
   * Find the first entry in a map which the given entry checker accepts.
   * @param map          map to search
   * @param entryChecker entry checker
   * @param <K>          map key type
   * @param <V>          map value type
   * @return first matching key, or {@code null} if no value matches
   */
  @Nullable
  public static <K, V> Map.Entry<K, V> findAnyEntry(@NotNull Map<K, V> map,
                                                    @NotNull Predicate2<? super K, ? super V> entryChecker)
  {
    for (Map.Entry<K, V> entry : map.entrySet()) {
      if (entryChecker.test(entry.getKey(), entry.getValue())) {
        return entry;
      }
    }
    return null;
  }

  /**
   * Return all entries which which the given checker accepts.
   * @param map           map to filter
   * @param entryChecker  entry checker which marks the entries for the result
   * @param <K>           map key type
   * @param <V>           map value type
   * @return set of matching entries
   */
  @NotNull
  public static <K, V> Set<Map.Entry<K, V>> findAllEntries(@NotNull Map<K, V> map,
                                                           @NotNull Predicate2<? super K, ? super V> entryChecker)
  {
    return Types.filter(new LinkedHashSet<>(),
                        map.entrySet(),
                        e -> entryChecker.test(e.getKey(), e.getValue()));
  }

  /**
   * Check whether any of a bunch of elements is contained in a collection.
   * @param set               collection, often a {@code Set}
   * @param possibleElements  possible elements which may be contained in the {@code set}
   * @param <T> element type of the set
   * @return {@code true} if at least one of the possible elements is contained in {@code set}<br>
   *         {@code false} if none is contained, or {@code set} or {@code possibleElements} is empty
   */
  public static <T> boolean containsAny(@NotNull Collection<T> set, @NotNull Iterable<? extends T> possibleElements)
  {
    if (!set.isEmpty()) {
      for (T elem : possibleElements) {
        if (set.contains(elem)) {
          return true;
        }
      }
    }
    return false;
  }

  /**
   * Go over an iterable and sepparate its elements depending a condition.
   * @param elements     elements to be separated
   * @param trueCollect  collector for the elements where {@code condition} returned {@code true}
   * @param falseCollect collector for the elements where {@code condition} returned {@code false}
   * @param condition    condition applied to the elements
   * @param <T> element type
   */
  public static <T> void separate(@NotNull Iterable<T> elements,
                                  @NotNull Consumer<? super T> trueCollect,
                                  @NotNull Consumer<? super T> falseCollect,
                                  @NotNull Predicate<? super T> condition)
  {
    for (T elem : elements) {
      if (condition.test(elem)) {
        trueCollect.accept(elem);
      }
      else {
        falseCollect.accept(elem);
      }
    }
  }

  /**
   * Go over an iterable and sepparate its elements depending a condition.
   * @param elements     elements to be separated
   * @param trueCollect  collector for the elements where {@code condition} returned {@code true}
   * @param falseCollect collector for the elements where {@code condition} returned {@code false}
   * @param condition    condition applied to the elements
   * @param <T> element type
   */
  public static <T> void separate(@NotNull Iterable<T> elements,
                                  @NotNull Collection<? super T> trueCollect,
                                  @NotNull Collection<? super T> falseCollect,
                                  @NotNull Predicate<? super T> condition)
  {
    separate(elements, trueCollect::add, falseCollect::add, condition);
  }

  /**
   * Fo over an iterable and separate its elements depending on a condition.
   * This method allows to define the slot type into which the elements are separated.
   * @param elements elements to be separated
   * @param slotCreator slot creator called twice to create the collection into which the results are separated
   * @param condition condition which decides whether an element goes to the results' first collection ({@code true}),
   *                  or to its second collection ({@code false}
   * @param <T> element type
   * @param <C> return type
   * @return pair of two C collections, {@link Pair#first first} contains the positive results,
   *         {@link Pair#second second} the negative
   */
  @NotNull
  public static <T, C extends Collection<? super T>> Pair<C> separate(@NotNull Iterable<T> elements,
                                                                      @NotNull Supplier<C> slotCreator,
                                                                      @NotNull Predicate<? super T> condition)
  {
    final Pair<C> result = Pair.createPair(slotCreator.get(), slotCreator.get());
    separate(elements, result.first, result.second, condition);
    return result;
  }

  /**
   * Go over an iterable and separate its elements into two lists depending on a condition.
   * @param elements elements to be separated
   * @param condition condition which decides whether an element goes to the results' first list ({@code true}),
   *                  or to its second list ({@code false}
   * @param <T> element type
   * @return pair of two linked lists, {@link Pair#first first} contains the positive results,
   *         {@link Pair#second second} the negative
   */
  @NotNull
  public static <T> Pair<LinkedList<T>> separate(@NotNull Iterable<T> elements,
                                                 @NotNull Predicate<? super T> condition)
  {
    return separate(elements, LinkedList::new, condition);
  }

  /**
   * This creates a string consisting only of spaces which is sometimes useful.
   * @param count number of spaces, must not be negative
   * @return string with as many spaces as required
   * @see #multiple(String, int)
   */
  @NotNull
  public static String spaces(int count)
  {
    return multiple(" ", count);
  }

  /**
   * Create a string from multiple repetitions of a character.
   * @param ch    the character
   * @param count the number of repetitions, not negative
   * @return string created from concatenating {@code ch} {@code count} times
   * @see #spaces(int)
   * @see #multiple(String, int)
   */
  @NotNull
  public static String multiple(char ch, int count)
  {
    return multiple(Character.toString(ch), count);
  }

  /**
   * Create a string from a base string repeated multiple times.
   * @param str   base string
   * @param count the number of repetitions, not negative
   * @return string created from concatenating {@code str} {@code count} times
   * @see #spaces(int)
   */
  @NotNull
  public static String multiple(@NotNull String str, int count)
  {
    if (count < 0) {
      throw new IllegalArgumentException("count must not be negative!");
    }
    if (count == 0  ||  str.isEmpty()) {
      return Empty.STRING;
    }
    final StringBuilder sb = new StringBuilder(count * str.length());
    String run = str;
    final int stop = Integer.highestOneBit(count);
    int mask = 1;
    // the forever loop is necessary to avoid duplicating run too often
    while (true) {
      if ((mask & count) != 0) {
        sb.append(run);
        if (mask == stop) {
          break;
        }
      }
      run = run + run;
      mask <<= 1;
    }
    return sb.toString();
  }

  /**
   * Create a list in reverse order for a given list.
   * This is a List method on its own in newer Java versions, and should only be used for Java8 compatibility.
   * @param list list to be reverted
   * @return reverted list
   * @param <T> element type of both lists
   */
  @NotNull
  public static <T> List<T> reversedList(@NotNull List<T> list)
  {
    final ArrayList<T> reversed = new ArrayList<>(list.size());
    for (ListIterator<T> it = list.listIterator(list.size());  it.hasPrevious();  ) {
      reversed.add(it.previous());
    }
    return reversed;
  }
}
