// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2022-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

/**
 * A choice between two values.
 * One value is associated with {@code true}, the other with {@code false}.
 * This is basically the functional equivalent of the ternary operator
 * with the restriction that both values have to be {@code null}.
 * See {@link TwoWayChoiceN} with does not have this restriction.
 * <p>
 * The recommended way to create a two-way choice is {@link TwoWayChoice#of(Object, Object)}.
 *
 * @param <T> value type
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 27, 2022
 */
@FunctionalInterface
public interface TwoWayChoice<T>
{
  /**
   * Return the selected choice.
   * @param select selector
   * @return selected choice
   */
  @NotNull
  T choose(boolean select);

  /**
   * Create a two-way choice.
   * @param trueValue   value returned for {@code true} selector, required to be not {@code null}
   * @param falseValue  value returned for {@code false} selector, required to be not {@code null}
   * @return boolean choice which performs the same as {@code select ? trueValue : falseValue}
   * @param <V> value type of selection
   */
  @NotNull
  static <V> TwoWayChoice<V> of(@NotNull V trueValue,
                                @NotNull V falseValue)
  {
    return select -> select ? trueValue : falseValue;
  }
}
