// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.util.*;

/**
 * Synchronized list.
 * <p>
 * This does synchronization a bit different than the collection returned by standard Java
 * {@link Collections#synchronizedList(List)}.
 * <p>
 * Especially iterators are running over a copy of the original list, so the list has not to stay
 * synchronized via any iteration.
 * <p>
 * Please note that this does not support sublisting.
 * <p>
 * It is adamant that all accesses to the underlying list are done via this list,
 * otherwise the synchronization guarantee is void.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @see Types#synchronizedCollection(Collection)
 */
public class SynchronizedList<T>
        implements List<T>
{
  private final List<T> baseList;
  private List<T> copyList;

  /**
   * Constructor.
   * @param baseList the base list wrapped by this synchronizer
   */
  public SynchronizedList(@NotNull List<T> baseList)
  {
    this.baseList = baseList;
  }

  /**
   * Get the copy.
   * Access it always via this method.
   * @return copy
   */
  @NotNull
  private synchronized List<T> getCopy()
  {
    if (copyList == null) {
      copyList = Collections.unmodifiableList(new ArrayList<>(baseList));
    }
    return copyList;
  }

  /**
   * Remove the copy.
   * To be called when the underlying collection is changed.
   */
  private synchronized void clearCopy()
  {
    copyList = null;
  }

  /**
   * Returns the number of elements in this collection.  If this collection
   * contains more than <tt>Integer.MAX_VALUE</tt> elements, returns
   * <tt>Integer.MAX_VALUE</tt>.
   *
   * @return the number of elements in this collection
   */
  @Override
  public synchronized int size()
  {
    return baseList.size();
  }

  /**
   * Returns <tt>true</tt> if this collection contains no elements.
   *
   * @return <tt>true</tt> if this collection contains no elements
   */
  @Override
  public synchronized boolean isEmpty()
  {
    return baseList.isEmpty();
  }

  /**
   * Returns <tt>true</tt> if this collection contains the specified
   * element.  More formally, returns <tt>true</tt> if and only if this
   * collection contains at least one element <tt>e</tt> such that
   * <tt>(o==null ? e==null : o.equals(e))</tt>.
   *
   * @param o element whose presence in this collection is to be tested.
   * @return <tt>true</tt> if this collection contains the specified
   *         element
   * @throws ClassCastException   if the type of the specified element
   *                              is incompatible with this collection (optional).
   * @throws NullPointerException if the specified element is null and this
   *                              collection does not support null elements (optional).
   */
  @Override
  public boolean contains(Object o)
  {
    return getCopy().contains(o);
  }

  /**
   * Returns an iterator over the elements in this collection.  There are no
   * guarantees concerning the order in which the elements are returned
   * (unless this collection is an instance of some class that provides a
   * guarantee).
   *
   * @return an <tt>Iterator</tt> over the elements in this collection
   */
  @Override
  @NotNull
  public Iterator<T> iterator()
  {
    final Iterator<T> copyIterator = getCopy().iterator();
    return new Iterator<T>() {
      private T current;
      private boolean hasCurrent;

      @Override
      public boolean hasNext()
      {
        return copyIterator.hasNext();
      }

      /**
       * Returns the next element in the iteration.
       *
       * @return the next element in the iteration.
       * @throws NoSuchElementException iteration has no more elements.
       */
      @Override
      public T next()
      {
        current = copyIterator.next();
        hasCurrent = true;
        return current;
      }

      /**
       * Removes from the underlying collection the last element returned by the
       * iterator (optional operation).  This method can be called only once per
       * call to <tt>next</tt>.  The behavior of an iterator is unspecified if
       * the underlying collection is modified while the iteration is in
       * progress in any way other than by calling this method.
       *
       * @throws UnsupportedOperationException if the <tt>remove</tt>
       *                                       operation is not supported by this Iterator.
       * @throws IllegalStateException         if the <tt>next</tt> method has not
       *                                       yet been called, or the <tt>remove</tt> method has already
       *                                       been called after the last call to the <tt>next</tt>
       *                                       method.
       */
      @Override
      public void remove()
      {
        if (!hasCurrent) {
          throw new IllegalStateException("Cannot delete w/o iterating!");
        }
        SynchronizedList.this.remove(current);
      }
    };
  }

  /**
   * Returns a list iterator over the elements in this list (in proper
   * sequence).
   *
   * @return a list iterator over the elements in this list (in proper
   * sequence)
   */
  @NotNull
  @Override
  public ListIterator<T> listIterator()
  {
    return getCopy().listIterator();
  }

  /**
   * Returns a list iterator over the elements in this list (in proper
   * sequence), starting at the specified position in the list.
   * The specified index indicates the first element that would be
   * returned by an initial call to {@link ListIterator#next next}.
   * An initial call to {@link ListIterator#previous previous} would
   * return the element with the specified index minus one.
   *
   * @param index index of the first element to be returned from the
   *              list iterator (by a call to {@link ListIterator#next next})
   * @return a list iterator over the elements in this list (in proper
   * sequence), starting at the specified position in the list
   * @throws IndexOutOfBoundsException if the index is out of range
   *                                   ({@code index &lt; 0 || index &gt; size()})
   */
  @NotNull
  @Override
  public ListIterator<T> listIterator(int index)
  {
    return getCopy().listIterator(index);
  }

  /**
   * Returns an array containing all of the elements in this collection.  If
   * the collection makes any guarantees as to what order its elements are
   * returned by its iterator, this method must return the elements in the
   * same order.<p>
   *
   * The returned array will be "safe" in that no references to it are
   * maintained by this collection.  (In other words, this method must
   * allocate a new array even if this collection is backed by an array).
   * The caller is thus free to modify the returned array.<p>
   *
   * This method acts as bridge between array-based and collection-based
   * APIs.
   *
   * @return an array containing all of the elements in this collection
   */
  @Override
  @NotNull
  public Object[] toArray()
  {
    return getCopy().toArray();
  }

  /**
   * Returns an array containing all of the elements in this collection;
   * the runtime type of the returned array is that of the specified array.
   * If the collection fits in the specified array, it is returned therein.
   * Otherwise, a new array is allocated with the runtime type of the
   * specified array and the size of this collection.<p>
   *
   * If this collection fits in the specified array with room to spare
   * (i.e., the array has more elements than this collection), the element
   * in the array immediately following the end of the collection is set to
   * <tt>null</tt>.  This is useful in determining the length of this
   * collection <i>only</i> if the caller knows that this collection does
   * not contain any <tt>null</tt> elements.)<p>
   *
   * If this collection makes any guarantees as to what order its elements
   * are returned by its iterator, this method must return the elements in
   * the same order.<p>
   *
   * Like the <tt>toArray</tt> method, this method acts as bridge between
   * array-based and collection-based APIs.  Further, this method allows
   * precise control over the runtime type of the output array, and may,
   * under certain circumstances, be used to save allocation costs<p>
   *
   * Suppose <tt>l</tt> is a <tt>List</tt> known to contain only strings.
   * The following code can be used to dump the list into a newly allocated
   * array of <tt>String</tt>:
   *
   * <pre>
   *     String[] x = (String[]) v.toArray(new String[0]);
   * </pre><p>
   *
   * Note that <tt>toArray(new Object[0])</tt> is identical in function to
   * <tt>toArray()</tt>.
   *
   * @param a the array into which the elements of this collection are to be
   *          stored, if it is big enough; otherwise, a new array of the same
   *          runtime type is allocated for this purpose.
   * @return an array containing the elements of this collection
   * @throws ArrayStoreException  the runtime type of the specified array is
   *                              not a supertype of the runtime type of every element in this
   *                              collection.
   * @throws NullPointerException if the specified array is <tt>null</tt>.
   */
  @Override
  @NotNull
  public <U> U[] toArray(@NotNull U[] a)
  {
    return getCopy().toArray(a);
  }

  /**
   * Ensures that this collection contains the specified element (optional
   * operation).  Returns <tt>true</tt> if this collection changed as a
   * result of the call.  (Returns <tt>false</tt> if this collection does
   * not permit duplicates and already contains the specified element.)<p>
   *
   * Collections that support this operation may place limitations on what
   * elements may be added to this collection.  In particular, some
   * collections will refuse to add <tt>null</tt> elements, and others will
   * impose restrictions on the type of elements that may be added.
   * Collection classes should clearly specify in their documentation any
   * restrictions on what elements may be added.<p>
   *
   * If a collection refuses to add a particular element for any reason
   * other than that it already contains the element, it <i>must</i> throw
   * an exception (rather than returning <tt>false</tt>).  This preserves
   * the invariant that a collection always contains the specified element
   * after this call returns.
   *
   * @param o element whose presence in this collection is to be ensured.
   * @return <tt>true</tt> if this collection changed as a result of the
   *         call
   * @throws UnsupportedOperationException <tt>add</tt> is not supported by
   *                                       this collection.
   * @throws ClassCastException            class of the specified element prevents it
   *                                       from being added to this collection.
   * @throws NullPointerException          if the specified element is null and this
   *                                       collection does not support null elements.
   * @throws IllegalArgumentException      some aspect of this element prevents
   *                                       it from being added to this collection.
   */
  @Override
  public synchronized boolean add(T o)
  {
    if (baseList.add(o)) {
      clearCopy();
      return true;
    }
    return false;
  }

  /**
   * Removes a single instance of the specified element from this
   * collection, if it is present (optional operation).  More formally,
   * removes an element <tt>e</tt> such that <tt>(o==null ?  e==null :
   * o.equals(e))</tt>, if this collection contains one or more such
   * elements.  Returns true if this collection contained the specified
   * element (or equivalently, if this collection changed as a result of the
   * call).
   *
   * @param o element to be removed from this collection, if present.
   * @return <tt>true</tt> if this collection changed as a result of the
   *         call
   * @throws ClassCastException            if the type of the specified element
   *                                       is incompatible with this collection (optional).
   * @throws NullPointerException          if the specified element is null and this
   *                                       collection does not support null elements (optional).
   * @throws UnsupportedOperationException remove is not supported by this
   *                                       collection.
   */
  @Override
  public synchronized boolean remove(Object o)
  {
    if (baseList.remove(o)) {
      clearCopy();
      return true;
    }
    return false;
  }

  /**
   * Returns <tt>true</tt> if this collection contains all of the elements
   * in the specified collection.
   *
   * @param c collection to be checked for containment in this collection.
   * @return <tt>true</tt> if this collection contains all of the elements
   *         in the specified collection
   * @throws ClassCastException   if the types of one or more elements
   *                              in the specified collection are incompatible with this
   *                              collection (optional).
   * @throws NullPointerException if the specified collection contains one
   *                              or more null elements and this collection does not support null
   *                              elements (optional);
   *                              or if the specified collection is
   *                              <tt>null</tt>.
   * @see #contains(Object)
   */
  @Override
  public boolean containsAll(@NotNull Collection<?> c)
  {
    return getCopy().containsAll(c);
  }

  /**
   * Adds all of the elements in the specified collection to this collection
   * (optional operation).  The behavior of this operation is undefined if
   * the specified collection is modified while the operation is in progress.
   * (This implies that the behavior of this call is undefined if the
   * specified collection is this collection, and this collection is
   * nonempty.)
   *
   * @param c elements to be inserted into this collection.
   * @return <tt>true</tt> if this collection changed as a result of the
   *         call
   * @throws UnsupportedOperationException if this collection does not
   *                                       support the <tt>addAll</tt> method.
   * @throws ClassCastException            if the class of an element of the specified
   *                                       collection prevents it from being added to this collection.
   * @throws NullPointerException          if the specified collection contains one
   *                                       or more null elements and this collection does not support null
   *                                       elements, or if the specified collection is <tt>null</tt>.
   * @throws IllegalArgumentException      some aspect of an element of the
   *                                       specified collection prevents it from being added to this
   *                                       collection.
   * @see #add(Object)
   */
  @Override
  public synchronized boolean addAll(@NotNull Collection<? extends T> c)
  {
    if (baseList.addAll(c)) {
      clearCopy();
      return true;
    }
    return false;
  }

  /**
   * Removes all this collection's elements that are also contained in the
   * specified collection (optional operation).  After this call returns,
   * this collection will contain no elements in common with the specified
   * collection.
   *
   * @param c elements to be removed from this collection.
   * @return <tt>true</tt> if this collection changed as a result of the
   *         call
   * @throws UnsupportedOperationException if the <tt>removeAll</tt> method
   *                                       is not supported by this collection.
   * @throws ClassCastException            if the types of one or more elements
   *                                       in this collection are incompatible with the specified
   *                                       collection (optional).
   * @throws NullPointerException          if this collection contains one or more
   *                                       null elements and the specified collection does not support
   *                                       null elements (optional);
   *                                       or if the specified collection is
   *                                       <tt>null</tt>.
   * @see #remove(Object)
   * @see #contains(Object)
   */
  @Override
  public synchronized boolean removeAll(@NotNull Collection<?> c)
  {
    if (baseList.removeAll(c)) {
      clearCopy();
      return true;
    }
    return false;
  }

  /**
   * Retains only the elements in this collection that are contained in the
   * specified collection (optional operation).  In other words, removes from
   * this collection all of its elements that are not contained in the
   * specified collection.
   *
   * @param c elements to be retained in this collection.
   * @return <tt>true</tt> if this collection changed as a result of the
   *         call
   * @throws UnsupportedOperationException if the <tt>retainAll</tt> method
   *                                       is not supported by this Collection.
   * @throws ClassCastException            if the types of one or more elements
   *                                       in this collection are incompatible with the specified
   *                                       collection (optional).
   * @throws NullPointerException          if this collection contains one or more
   *                                       null elements and the specified collection does not support null
   *                                       elements (optional);
   *                                       or if the specified collection is
   *                                       <tt>null</tt>.
   * @see #remove(Object)
   * @see #contains(Object)
   */
  @Override
  public synchronized boolean retainAll(@NotNull Collection<?> c)
  {
    if (baseList.retainAll(c)) {
      clearCopy();
      return true;
    }
    return false;
  }

  /**
   * Removes all of the elements from this collection (optional operation).
   * This collection will be empty after this method returns unless it
   * throws an exception.
   *
   * @throws UnsupportedOperationException if the <tt>clear</tt> method is
   *                                       not supported by this collection.
   */
  @Override
  public synchronized void clear()
  {
    if (!baseList.isEmpty()) {
      baseList.clear();
      clearCopy();
    }
  }

  /**
   * Inserts all of the elements in the specified collection into this
   * list at the specified position (optional operation).  Shifts the
   * element currently at that position (if any) and any subsequent
   * elements to the right (increases their indices).  The new elements
   * will appear in this list in the order that they are returned by the
   * specified collection's iterator.  The behavior of this operation is
   * undefined if the specified collection is modified while the
   * operation is in progress.  (Note that this will occur if the specified
   * collection is this list, and it's nonempty.)
   *
   * @param index index at which to insert the first element from the
   *              specified collection
   * @param c     collection containing elements to be added to this list
   * @return <tt>true</tt> if this list changed as a result of the call
   * @throws UnsupportedOperationException if the <tt>addAll</tt> operation
   *                                       is not supported by this list
   * @throws ClassCastException            if the class of an element of the specified
   *                                       collection prevents it from being added to this list
   * @throws NullPointerException          if the specified collection contains one
   *                                       or more null elements and this list does not permit null
   *                                       elements, or if the specified collection is null
   * @throws IllegalArgumentException      if some property of an element of the
   *                                       specified collection prevents it from being added to this list
   * @throws IndexOutOfBoundsException     if the index is out of range
   *                                       (<tt>index &lt; 0 || index &gt; size()</tt>)
   */
  @Override
  public synchronized boolean addAll(int index, @NotNull Collection<? extends T> c)
  {
    clearCopy();
    return baseList.addAll(index, c);
  }

  /**
   * Returns the element at the specified position in this list.
   *
   * @param index index of the element to return
   * @return the element at the specified position in this list
   * @throws IndexOutOfBoundsException if the index is out of range
   *                                   (<tt>index &lt; 0 || index &gt;= size()</tt>)
   */
  @Override
  public T get(int index)
  {
    return getCopy().get(index);
  }

  /**
   * Replaces the element at the specified position in this list with the
   * specified element (optional operation).
   *
   * @param index   index of the element to replace
   * @param element element to be stored at the specified position
   * @return the element previously at the specified position
   * @throws UnsupportedOperationException if the <tt>set</tt> operation
   *                                       is not supported by this list
   * @throws ClassCastException            if the class of the specified element
   *                                       prevents it from being added to this list
   * @throws NullPointerException          if the specified element is null and
   *                                       this list does not permit null elements
   * @throws IllegalArgumentException      if some property of the specified
   *                                       element prevents it from being added to this list
   * @throws IndexOutOfBoundsException     if the index is out of range
   *                                       (<tt>index &lt; 0 || index &gt;= size()</tt>)
   */
  @Override
  public synchronized T set(int index, T element)
  {
    clearCopy();
    return baseList.set(index, element);
  }

  /**
   * Inserts the specified element at the specified position in this list
   * (optional operation).  Shifts the element currently at that position
   * (if any) and any subsequent elements to the right (adds one to their
   * indices).
   *
   * @param index   index at which the specified element is to be inserted
   * @param element element to be inserted
   * @throws UnsupportedOperationException if the <tt>add</tt> operation
   *                                       is not supported by this list
   * @throws ClassCastException            if the class of the specified element
   *                                       prevents it from being added to this list
   * @throws NullPointerException          if the specified element is null and
   *                                       this list does not permit null elements
   * @throws IllegalArgumentException      if some property of the specified
   *                                       element prevents it from being added to this list
   * @throws IndexOutOfBoundsException     if the index is out of range
   *                                       (<tt>index &lt; 0 || index &gt; size()</tt>)
   */
  @Override
  public synchronized void add(int index, T element)
  {
    clearCopy();
    baseList.add(index, element);
  }

  /**
   * Removes the element at the specified position in this list (optional
   * operation).  Shifts any subsequent elements to the left (subtracts one
   * from their indices).  Returns the element that was removed from the
   * list.
   *
   * @param index the index of the element to be removed
   * @return the element previously at the specified position
   * @throws UnsupportedOperationException if the <tt>remove</tt> operation
   *                                       is not supported by this list
   * @throws IndexOutOfBoundsException     if the index is out of range
   *                                       (<tt>index &lt; 0 || index &gt;= size()</tt>)
   */
  @Override
  public synchronized T remove(int index)
  {
    clearCopy();
    return baseList.remove(index);
  }

  /**
   * Returns the index of the first occurrence of the specified element
   * in this list, or -1 if this list does not contain the element.
   * More formally, returns the lowest index <tt>i</tt> such that
   * <tt>(o==null&nbsp;?&nbsp;get(i)==null&nbsp;:&nbsp;o.equals(get(i)))</tt>,
   * or -1 if there is no such index.
   *
   * @param o element to search for
   * @return the index of the first occurrence of the specified element in
   * this list, or -1 if this list does not contain the element
   * @throws ClassCastException   if the type of the specified element
   *                              is incompatible with this list
   *                              (<a href="Collection.html#optional-restrictions">optional</a>)
   * @throws NullPointerException if the specified element is null and this
   *                              list does not permit null elements
   *                              (<a href="Collection.html#optional-restrictions">optional</a>)
   */
  @Override
  public int indexOf(Object o)
  {
    return getCopy().indexOf(o);
  }

  /**
   * Returns the index of the last occurrence of the specified element
   * in this list, or -1 if this list does not contain the element.
   * More formally, returns the highest index <tt>i</tt> such that
   * <tt>(o==null&nbsp;?&nbsp;get(i)==null&nbsp;:&nbsp;o.equals(get(i)))</tt>,
   * or -1 if there is no such index.
   *
   * @param o element to search for
   * @return the index of the last occurrence of the specified element in
   * this list, or -1 if this list does not contain the element
   * @throws ClassCastException   if the type of the specified element
   *                              is incompatible with this list
   *                              (<a href="Collection.html#optional-restrictions">optional</a>)
   * @throws NullPointerException if the specified element is null and this
   *                              list does not permit null elements
   *                              (<a href="Collection.html#optional-restrictions">optional</a>)
   */
  @Override
  public int lastIndexOf(Object o)
  {
    return getCopy().lastIndexOf(o);
  }

  /**
   * Returns a view of the portion of this list between the specified
   * <tt>fromIndex</tt>, inclusive, and <tt>toIndex</tt>, exclusive.  (If
   * <tt>fromIndex</tt> and <tt>toIndex</tt> are equal, the returned list is
   * empty.)  The returned list is backed by this list, so non-structural
   * changes in the returned list are reflected in this list, and vice-versa.
   * The returned list supports all of the optional list operations supported
   * by this list.<p>
   *
   * This method eliminates the need for explicit range operations (of
   * the sort that commonly exist for arrays).  Any operation that expects
   * a list can be used as a range operation by passing a subList view
   * instead of a whole list.  For example, the following idiom
   * removes a range of elements from a list:
   * <pre>
   *      list.subList(from, to).clear();
   * </pre>
   * Similar idioms may be constructed for <tt>indexOf</tt> and
   * <tt>lastIndexOf</tt>, and all of the algorithms in the
   * <tt>Collections</tt> class can be applied to a subList.<p>
   *
   * The semantics of the list returned by this method become undefined if
   * the backing list (i.e., this list) is <i>structurally modified</i> in
   * any way other than via the returned list.  (Structural modifications are
   * those that change the size of this list, or otherwise perturb it in such
   * a fashion that iterations in progress may yield incorrect results.)
   *
   * @param fromIndex low endpoint (inclusive) of the subList
   * @param toIndex   high endpoint (exclusive) of the subList
   * @return a view of the specified range within this list
   * @throws UnsupportedOperationException sublisting is not supported
   */
  @NotNull
  @Override
  public List<T> subList(int fromIndex, int toIndex)
  {
    throw new UnsupportedOperationException("No sublisting for synchronized list.");
  }

  @Override
  public synchronized void sort(Comparator<? super T> c)
  {
    clearCopy();
    baseList.sort(c);
  }

  /**
   * Create a synchronized list from the given basic list.
   * @param basicList  basic list
   * @param <U>  type of list
   * @return synchronized list
   */
  @NotNull
  public static <U> SynchronizedList<U> create(@NotNull List<U> basicList)
  {
    return new SynchronizedList<>(basicList);
  }
}
