// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright ©2023-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

/**
 * A subjective view on objects.
 * <p>
 * This class allows to substitute objects with views which expose different
 * {@link Object#equals(Object)} and {@link Object#hashCode()} implementations.
 * <p>
 * It does not work on primitive types, but their boxed counterparts. If you
 * provide a {@link Subjective} for a boxed primitive class it is also
 * a good idea to provide one for its one-dimensional array type. E.g. if
 * you provide a subjective for {@code Double} you should also provide one
 * for {@code double[]}.
 * <p>
 * Hashing of arrays is usually done flat, but you can change this by
 * setting the {@code deep} flag in the constructors to {@code true}.
 * This will influence all arrays, not just the ones which are handled
 * by subjectives.
 * 
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since February 13, 2023
 */
public class Subjectivity
{
  private final boolean deep;
  @NotNull
  private final Map<Class<?>, Subjective<?>> subjectMapping = new HashMap<>();

  /**
   * Constructor.
   * @param deep make equals and hashcode work deep?
   * @param subjects subjective views used by this subjectivity
   */
  public Subjectivity(boolean deep, @NotNull Iterable<Subjective<?>> subjects)
  {
    subjects.forEach(sub -> subjectMapping.put(sub.type(), sub));
    this.deep = deep;
  }

  /**
   * Constructor.
   * @param deep make equals and hashcode work deep?
   * @param subjects subjective views used by this subjectivity
   */
  public Subjectivity(boolean deep, @NotNull Subjective<?> ... subjects)
  {
    this(deep, Indexable.viewArray(subjects));
  }

  /**
   * Wrap array in a deep hash and deep equals handler if {@link #deep} is set.
   *
   * @param obj obj which might be wrapped
   * @return wrapped or not wrapped {@code obj}
   */
  private Object possibleDeepWrap(@NotNull Object obj)
  {
    if (!deep) {
      return obj;
    }
    
    final Class<?> objClass = obj.getClass();
    if (!objClass.isArray()) {
      return obj;
    }

    final Class<?> elementType = objClass.getComponentType();

    if (elementType.isPrimitive()) {
      if (obj instanceof int[]) {
        return new IntArrayWrap((int[]) obj);
      }
      if (obj instanceof double[]) {
        return new DoubleArrayWrap((double[]) obj);
      }
      if (obj instanceof byte[]) {
        return new ByteArrayWrap((byte[]) obj);
      }
      if (obj instanceof long[]) {
        return new LongArrayWrap((long[]) obj);
      }
      if (obj instanceof short[]) {
        return new ShortArrayWrap((short[]) obj);
      }
      if (obj instanceof char[]) {
        return new CharArrayWrap((char[]) obj);
      }
      if (obj instanceof boolean[]) {
        return new BooleanArrayWrap((boolean[]) obj);
      }
    }
    return new ArrayWrap((Object[]) obj);
  }

  /**
   * Substitute the given object.
   * <p>
   * It is strictly recommended to use a substitute only for checking equality or hashing.
   *
   * @param obj object to be substituted, not {@code null}
   * @return {@code obj} if there is no substitution for the given object<br>
   *         or a substitute which stands in for the object
   * @see #substitute0(Object)
   */
  @NotNull
  public Object substitute(@NotNull Object obj)
  {
    final Class<?> aClass = obj.getClass();
    final Subjective<?> subjective = subjectMapping.get(aClass);
    if (subjective != null) {
      return possibleDeepWrap(subjective.subjectFrom(obj));
    }

    if (aClass.isArray()) {
      return possibleDeepWrap(substituteArray(aClass.getComponentType(), obj, 1));
    }

    return obj;
  }

  /**
   * Substitute the given object which may be {@code null}.
   * @param obj object to be substituted, possibly {@code null}
   * @return {@code obj} if there is no substitution for the given object, or it is {@code null}<br>
   *         or a substitute which stands in for the object
   */
  @Nullable
  public Object substitute0(@Nullable Object obj)
  {
    return obj == null
            ? null
            : substitute(obj);
  }

  @NotNull
  private Object substituteArray(@NotNull Class<?> componentType,
                                 @NotNull Object array,
                                 int level)
  {
    final Subjective<?> subjective = subjectMapping.get(componentType);
    if (subjective != null) {
      return repackArray(array, subjective, level);
    }

    if (deep && componentType.isArray()) {
      return substituteArray(componentType.getComponentType(), array, level + 1);
    }

    if (deep &&  !componentType.isPrimitive()) {
      final Object[] incoming = (Object[]) array;
      Object[] outgoing = null;
      for (int i = 0;  i < incoming.length; ++i) {
        final Object in = incoming[i];
        final Object subst = substitute0(in);
        if (outgoing != null) {
          outgoing[i] = subst;
        }
        else if (subst != in) {
          // first substitute, so have to create substitution for whole array
          outgoing = new Object[incoming.length];
          if (i > 0) {
            System.arraycopy(incoming, 0, outgoing, 0, i);
          }
          outgoing[i] = subst;
        }
      }
      if (outgoing != null) {
        return outgoing;
      }
    }

    return array;
  }

  @NotNull
  private Object[] repackArray(@NotNull Object array,
                               @NotNull Subjective<?> sub,
                               int level)
  {
    final Object[] incoming = (Object[]) array;
    final Object[] outgoing = new Object[incoming.length];
    if (level == 1) {
      for (int i = incoming.length - 1;  i >= 0;  --i) {
        outgoing[i] = sub.subjectFrom(incoming[i]);
      }
    }
    else {
      final int lessLevel = level - 1;
      for (int i = incoming.length - 1;  i >= 0;  --i) {
        outgoing[i] = repackArray(incoming[i], sub, lessLevel);
      }
    }
    return outgoing;
  }

  private static class ArrayWrap
          implements Cloneable
  {
    @NotNull
    private final Object[] array;

    public ArrayWrap(@NotNull Object[] array)
    {
      this.array = array;
    }

    @Override
    public int hashCode()
    {
      return Arrays.deepHashCode(array);
    }

    @Override
    public boolean equals(Object obj)
    {
      if (obj == this) {
        return true;
      }
      final Object[] otherArray;
      if (obj instanceof ArrayWrap) {
        otherArray = ((ArrayWrap)obj).array; 
      }
      else if (obj instanceof Object[]) {
        otherArray = (Object[]) obj;
      }
      else {
        return false;
      }
      return Arrays.deepEquals(array, otherArray);
    }

    @Override
    public String toString()
    {
      return String.format("WRAP{%s}", Arrays.toString(array));
    }

    @Override
    public ArrayWrap clone()
    {
      try {
        return (ArrayWrap)super.clone();
      } catch (CloneNotSupportedException e) {
        throw new AssertionError();
      }
    }
  }
  
  private static class IntArrayWrap
          implements Cloneable
  {
    @NotNull
    private final int[] array;

    public IntArrayWrap(@NotNull int[] array)
    {
      this.array = array;
    }

    @Override
    public int hashCode()
    {
      return Arrays.hashCode(array);
    }

    @Override
    public boolean equals(Object obj)
    {
      if (obj == this) {
        return true;
      }
      final int[] otherArray;
      if (obj instanceof IntArrayWrap) {
        otherArray = ((IntArrayWrap)obj).array; 
      }
      else if (obj instanceof int[]) {
        otherArray = (int[]) obj;
      }
      else {
        return false;
      }
      return Arrays.equals(array, otherArray);
    }

    @Override
    public String toString()
    {
      return String.format("WRAP{%s}", Arrays.toString(array));
    }

    @Override
    public IntArrayWrap clone()
    {
      try {
        return (IntArrayWrap)super.clone();
      } catch (CloneNotSupportedException e) {
        throw new AssertionError();
      }
    }
  }
  
  private static class DoubleArrayWrap
          implements Cloneable
  {
    @NotNull
    private final double[] array;

    public DoubleArrayWrap(@NotNull double[] array)
    {
      this.array = array;
    }

    @Override
    public int hashCode()
    {
      return Arrays.hashCode(array);
    }

    @Override
    public boolean equals(Object obj)
    {
      if (obj == this) {
        return true;
      }
      final double[] otherArray;
      if (obj instanceof DoubleArrayWrap) {
        otherArray = ((DoubleArrayWrap)obj).array; 
      }
      else if (obj instanceof double[]) {
        otherArray = (double[]) obj;
      }
      else {
        return false;
      }
      return Arrays.equals(array, otherArray);
    }

    @Override
    public String toString()
    {
      return String.format("WRAP{%s}", Arrays.toString(array));
    }

    @Override
    public DoubleArrayWrap clone()
    {
      try {
        return (DoubleArrayWrap)super.clone();
      } catch (CloneNotSupportedException e) {
        throw new AssertionError();
      }
    }
  }
  
  private static class FloatArrayWrap
          implements Cloneable
  {
    @NotNull
    private final float[] array;

    public FloatArrayWrap(@NotNull float[] array)
    {
      this.array = array;
    }

    @Override
    public int hashCode()
    {
      return Arrays.hashCode(array);
    }

    @Override
    public boolean equals(Object obj)
    {
      if (obj == this) {
        return true;
      }
      final float[] otherArray;
      if (obj instanceof FloatArrayWrap) {
        otherArray = ((FloatArrayWrap)obj).array; 
      }
      else if (obj instanceof float[]) {
        otherArray = (float[]) obj;
      }
      else {
        return false;
      }
      return Arrays.equals(array, otherArray);
    }

    @Override
    public String toString()
    {
      return String.format("WRAP{%s}", Arrays.toString(array));
    }

    @Override
    public FloatArrayWrap clone()
    {
      try {
        return (FloatArrayWrap)super.clone();
      } catch (CloneNotSupportedException e) {
        throw new AssertionError();
      }
    }
  }
  
  private static class LongArrayWrap
          implements Cloneable
  {
    @NotNull
    private final long[] array;

    public LongArrayWrap(@NotNull long[] array)
    {
      this.array = array;
    }

    @Override
    public int hashCode()
    {
      return Arrays.hashCode(array);
    }

    @Override
    public boolean equals(Object obj)
    {
      if (obj == this) {
        return true;
      }
      final long[] otherArray;
      if (obj instanceof LongArrayWrap) {
        otherArray = ((LongArrayWrap)obj).array; 
      }
      else if (obj instanceof long[]) {
        otherArray = (long[]) obj;
      }
      else {
        return false;
      }
      return Arrays.equals(array, otherArray);
    }

    @Override
    public String toString()
    {
      return String.format("WRAP{%s}", Arrays.toString(array));
    }

    @Override
    public LongArrayWrap clone()
    {
      try {
        return (LongArrayWrap)super.clone();
      } catch (CloneNotSupportedException e) {
        throw new AssertionError();
      }
    }
  }

  private static class ShortArrayWrap
          implements Cloneable
  {
    @NotNull
    private final short[] array;

    public ShortArrayWrap(@NotNull short[] array)
    {
      this.array = array;
    }

    @Override
    public int hashCode()
    {
      return Arrays.hashCode(array);
    }

    @Override
    public boolean equals(Object obj)
    {
      if (obj == this) {
        return true;
      }
      final short[] otherArray;
      if (obj instanceof ShortArrayWrap) {
        otherArray = ((ShortArrayWrap)obj).array;
      }
      else if (obj instanceof short[]) {
        otherArray = (short[]) obj;
      }
      else {
        return false;
      }
      return Arrays.equals(array, otherArray);
    }

    @Override
    public String toString()
    {
      return String.format("WRAP{%s}", Arrays.toString(array));
    }

    @Override
    public ShortArrayWrap clone()
    {
      try {
        return (ShortArrayWrap)super.clone();
      } catch (CloneNotSupportedException e) {
        throw new AssertionError();
      }
    }
  }

  private static class ByteArrayWrap
          implements Cloneable
  {
    @NotNull
    private final byte[] array;

    public ByteArrayWrap(@NotNull byte[] array)
    {
      this.array = array;
    }

    @Override
    public int hashCode()
    {
      return Arrays.hashCode(array);
    }

    @Override
    public boolean equals(Object obj)
    {
      if (obj == this) {
        return true;
      }
      final byte[] otherArray;
      if (obj instanceof ByteArrayWrap) {
        otherArray = ((ByteArrayWrap)obj).array;
      }
      else if (obj instanceof byte[]) {
        otherArray = (byte[]) obj;
      }
      else {
        return false;
      }
      return Arrays.equals(array, otherArray);
    }

    @Override
    public String toString()
    {
      return String.format("WRAP{%s}", Arrays.toString(array));
    }

    @Override
    public ByteArrayWrap clone()
    {
      try {
        return (ByteArrayWrap)super.clone();
      } catch (CloneNotSupportedException e) {
        throw new AssertionError();
      }
    }
  }

  private static class CharArrayWrap
          implements Cloneable
  {
    @NotNull
    private final char[] array;

    public CharArrayWrap(@NotNull char[] array)
    {
      this.array = array;
    }

    @Override
    public int hashCode()
    {
      return Arrays.hashCode(array);
    }

    @Override
    public boolean equals(Object obj)
    {
      if (obj == this) {
        return true;
      }
      final char[] otherArray;
      if (obj instanceof CharArrayWrap) {
        otherArray = ((CharArrayWrap)obj).array;
      }
      else if (obj instanceof char[]) {
        otherArray = (char[]) obj;
      }
      else {
        return false;
      }
      return Arrays.equals(array, otherArray);
    }

    @Override
    public String toString()
    {
      return String.format("WRAP{%s}", Arrays.toString(array));
    }

    @Override
    public CharArrayWrap clone()
    {
      try {
        return (CharArrayWrap)super.clone();
      } catch (CloneNotSupportedException e) {
        throw new AssertionError();
      }
    }
  }

  private static class BooleanArrayWrap
          implements Cloneable
  {
    @NotNull
    private final boolean[] array;

    public BooleanArrayWrap(@NotNull boolean[] array)
    {
      this.array = array;
    }

    @Override
    public int hashCode()
    {
      return Arrays.hashCode(array);
    }

    @Override
    public boolean equals(Object obj)
    {
      if (obj == this) {
        return true;
      }
      final boolean[] otherArray;
      if (obj instanceof BooleanArrayWrap) {
        otherArray = ((BooleanArrayWrap)obj).array;
      }
      else if (obj instanceof boolean[]) {
        otherArray = (boolean[]) obj;
      }
      else {
        return false;
      }
      return Arrays.equals(array, otherArray);
    }

    @Override
    public String toString()
    {
      return String.format("WRAP{%s}", Arrays.toString(array));
    }

    @Override
    public BooleanArrayWrap clone()
    {
      try {
        return (BooleanArrayWrap)super.clone();
      } catch (CloneNotSupportedException e) {
        throw new AssertionError();
      }
    }
  }
}
