// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright ©2023-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

/**
 * Type-based information how to create a {@link Subject} from an object.
 * This is used to override standard Java object behavior, especially for
 * {@link Object#equals(Object)} and  {@link Object#hashCode()}.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since February 12, 2023
 * @param <T> type which this subjective is handling
 */
public interface Subjective<T>
{
  /**
   * Get the subjective type.
   *
   * @return type of subjectivized object.
   */
  @NotNull
  Class<T> type();

  /**
   * Cast any object to the type of this subjective.
   * @param object object to be cast
   * @return object cast to {@linkplain #type()}
   * @throws ClassCastException if object has wrong type
   */
  @NotNull
  default T cast(@NotNull Object object)
  {
    return type().cast(object);
  }

  /**
   * Get the substitute object for the given object.
   * Any implementation is expected to handle the incoming {@code object}
   * as immutable, i.e. to never change it.
   * @param object object to be substituted
   * @return substitution object
   */
  @NotNull
  Subject<T> subject(@NotNull T object);

  /**
   * Helper method to create a subject from an arbitrary object.
   * @param object object expected to be of the {@linkplain #type()} of this subjective
   * @return subjectivized {@code object}
   * @throws ClassCastException if object has wrong type
   */
  @NotNull
  default Subject<T> subjectFrom(@NotNull Object object)
  {
    return subject(cast(object));
  }
}
