// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright ©2023-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

/**
 * This is mainly a marker interface for subjectivized objects.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since February 13, 2023
 * @param <T> original type
 */
public interface Subject<T>
{
  /**
   * Get the substituted object.
   * @return substituted object
   */
  @NotNull
  T substituted();

  abstract class Base<TT> implements Subject<TT>
  {
    @NotNull
    protected final TT object;
    protected final Class<TT> type;

    public Base(@NotNull TT object,
                @NotNull Class<TT> type)
    {
      this.object = object;
      this.type = type;
    }

    @NotNull
    @Override
    public TT substituted()
    {
      return object;
    }

    protected abstract boolean isEqualTo(@NotNull TT other);

    @Override
    public boolean equals(Object obj)
    {
      if (obj == null) {
        return false;
      }
      if (obj instanceof Subject) {
        obj = ((Subject<?>)obj).substituted();
      }
      if (!type.isInstance(obj)) {
        return false;
      }
      return isEqualTo(type.cast(obj));
    }

    @Override
    public String toString()
    {
      return String.format("SUBJECT{%s}", object);
    }
  }
}
