// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.lang.ref.SoftReference;

/**
 * Wrapper for a property change listener which reference the original change listener softly.
 * <p>
 * If the original listener is garbage collected, this property change listener is just doing nothing.
 * Note that although the referenced listener and everything what is attached to it might be released,
 * this listener is still attached to all classes it was added as a property change listener, because
 * it has no chance to know where it was added, so it cannot remove itself.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SoftPropertyChangeListener
        implements PropertyChangeListener
{
  /** The weakly wrapped listener. */
  private SoftReference<PropertyChangeListener> wrapped;

  /**
   * Constructor.
   * @param wrapped listener to wrap, which is only hold via a weak reference
   */
  public SoftPropertyChangeListener(@NotNull PropertyChangeListener wrapped)
  {
    this.wrapped = new SoftReference<>(wrapped);
  }

  /**
   * This method gets called when a bound property is changed.
   *
   * @param evt A PropertyChangeEvent object describing the event source
   *            and the property that has changed.
   */
  @Override
  public void propertyChange(PropertyChangeEvent evt)
  {
    final PropertyChangeListener listener = getListener();
    if (listener != null) {
      listener.propertyChange(evt);
    }
  }

  /**
   * Is this listener still valid?
   * @return {@code true} if the reference is still available<br>
   *         {@code false} if it is no longer available
   */
  public boolean isValid()
  {
    return getListener() != null;
  }

  /**
   * Get the listener.
   * This takes also care of cleanup if the reference is gone.
   * @return the listener or {@code null}
   */
  @Nullable
  private PropertyChangeListener getListener()
  {
    final PropertyChangeListener listener = wrapped.get();
    if (listener == null) {
      // can remove reference as well
      wrapped = null;
    }
    return listener;
  }
}
