// ============================================================================
// File:               Sequences
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2023-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            11/20/23 10:19 AM
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.Ordering;

import java.util.Iterator;
import java.util.Objects;
import java.util.function.BiPredicate;

/**
 * Equality and comparability for sequences.
 * <p>
 * A sequence is an {@link Iterable} which is expected to provide the same sequence of
 * items in the same order. E.g. a {@link java.util.HashSet} and a {@link java.util.SortedSet}
 * provide different sequences, even if they contain the same elements.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 20, 2023
 */
public class Sequences
{
  /**
   * Are two objects identical?
   * This pours {@code e1 == e2} into a bi-predicate.
   */
  public static final BiPredicate<Object, Object> IDENTICAL = (e1, e2) -> e1 == e2;

  /** Don't create. */
  private Sequences() {}

  /**
   * Are two sequences equal in the standard Java definition of equality?
   * This convenience method compares two sequences element by element using
   * {@link Objects#deepEquals(Object, Object)} as equality definition.
   * Two sequences with different lengths are always considered unequal.
   * <p>
   * This method deliberately allows comparison of sequences of different types.
   * @param seq1 first sequence
   * @param seq2 second sequence
   * @return {@code true} if both sequences are considered equal<br>
   *         {@code false} if they are considered unequal
   */
  public static boolean areEqual(@NotNull Iterable<?> seq1,
                                 @NotNull Iterable<?> seq2)
  {
    return seq1 == seq2  ||  areEqual(seq1, seq2, Objects::deepEquals);
  }

  /**
   * Do two iterators provide the same sequence, using the standard Java definition of equality?
   * This method compares two iterators element by element for equality using
   * {@link Objects#deepEquals(Object, Object)}.
   * Two sequences with different lengths are always considered unequal.
   * <p>
   * This method deliberately allows comparison of sequences of different types.
   * <p>
   * The iterators used as parameters will be both exhausted if this method returns {@code true}.
   * Otherwise, they will stand at the first inequality, which can mean that one is exhausted
   * while the other is not.
   * @param it1  iterator over the first sequence
   * @param it2  iterator over the second sequence
   * @return {@code true} if both sequences are considered equal<br>
   *         {@code false} if they are considered unequal
   */
  public static boolean areEqual(@NotNull Iterator<?> it1,
                                 @NotNull Iterator<?> it2)
  {
    return areEqual(it1, it2, Objects::deepEquals);
  }

  /**
   * Are two sequences equal in the standard Java definition of equality?
   * This convenience method compares two sequences element by element using
   * {@link Objects#equals(Object, Object)} as equality definition.
   * Two sequences with different lengths are always considered unequal.
   * <p>
   * This method deliberately allows comparison of sequences of different types.
   * @param seq1 first sequence
   * @param seq2 second sequence
   * @return {@code true} if both sequences are considered equal<br>
   *         {@code false} if they are considered unequal
   */
  public static boolean areIdentical(@NotNull Iterable<?> seq1,
                                     @NotNull Iterable<?> seq2)
  {
    return seq1 == seq2  ||  areEqual(seq1, seq2, IDENTICAL);
  }

  /**
   * Do two iterators provide the same sequence, using the standard Java definition of equality?
   * This method compares two iterators element by element for equality using
   * {@link Objects#equals(Object, Object)}.
   * Two sequences with different lengths are always considered unequal.
   * <p>
   * This method deliberately allows comparison of sequences of different types.
   * <p>
   * The iterators used as parameters will be both exhausted if this method returns {@code true}.
   * Otherwise, they will stand at the first inequality, which can mean that one is exhausted
   * while the other is not.
   * @param it1  iterator over the first sequence
   * @param it2  iterator over the second sequence
   * @return {@code true} if both sequences are considered equal<br>
   *         {@code false} if they are considered unequal
   */
  public static boolean areIdentical(@NotNull Iterator<?> it1,
                                     @NotNull Iterator<?> it2)
  {
    return areEqual(it1, it2, IDENTICAL);
  }

  /**
   * Are two sequences equal?
   * This method compares two sequences element by element for equality.
   * The user is free to decide what is considered "equal".
   * Two sequences with different lengths are always considered unequal.
   * <p>
   * This method deliberately allows comparison of sequences of different types.
   * @param seq1     first sequence
   * @param seq2     second sequence
   * @param equality element equality definition, has to return {@code true} for elements considered,
   *                 and {@code false} for elements considered unequal
   * @return {@code true} if both sequences are considered equal<br>
   *         {@code false} if they are considered unequal
   * @param <T1> element type of first sequence
   * @param <T2> element type of second sequence
   */
  public static <T1, T2> boolean areEqual(@NotNull Iterable<T1> seq1,
                                          @NotNull Iterable<T2> seq2,
                                          @NotNull BiPredicate<? super T1, ? super T2> equality)
  {
    return areEqual(seq1.iterator(), seq2.iterator(), equality);
  }

  /**
   * Are two arrays equal?
   * This method compares two arrays element by element for equality.
   * The user is free to decide what is considered "equal".
   * Two arrays with different lengths are always considered unequal.
   * <p>
   * This method deliberately allows comparison of arrays of different types.
   * @param arr1     first array
   * @param arr2     second array
   * @param equality element equality definition, has to return {@code true} for elements considered,
   *                 and {@code false} for elements considered unequal
   * @return {@code true} if both sequences are considered equal<br>
   *         {@code false} if they are considered unequal
   * @param <T1> element type of first array
   * @param <T2> element type of second array
   */
  public static <T1, T2> boolean areEqual(@NotNull T1[] arr1,
                                          @NotNull T2[] arr2,
                                          @NotNull BiPredicate<? super T1, ? super T2> equality)
  {
    return areEqual(arr1, 0, arr1.length,
                    arr2, 0, arr2.length,
                    equality);
  }

  /**
   * Are parts of two arrays equal?
   * This method compares intervals of two arrays element by element for equality.
   * The user is free to decide what is considered "equal".
   * Two intervals with different lengths are always considered unequal.
   * <p>
   * This method deliberately allows comparison of arrays of different types.
   * @param arr1     first array
   * @param start1   index of the first relevant element in the first array
   * @param len1     number of relevant elements in the first array
   * @param arr2     second array
   * @param start2   index of the first relevant element in the second array
   * @param len2     number of relevant elements in the second array
   * @param equality element equality definition, has to return {@code true} for elements considered,
   *                 and {@code false} for elements considered unequal
   * @return {@code true} if both intervals are considered equal<br>
   *         {@code false} if they are considered unequal
   * @param <T1> element type of first array
   * @param <T2> element type of second array
   */
  public static <T1, T2> boolean areEqual(@NotNull T1[] arr1, int start1, int len1,
                                          @NotNull T2[] arr2, int start2, int len2,
                                          @NotNull BiPredicate<? super T1, ? super T2> equality)
  {
    if (len1 != len2) {
      return false;
    }
    for (int i = 0;  i < len1;  ++i) {
      if (!equality.test(arr1[start1 + i], arr2[start2 + i])) {
        return false;
      }
    }
    return true;
  }

  /**
   * Do two iterators provide the same sequence?
   * This method compares two iterators element by element for equality.
   * The user is free to decide what is considered "equal".
   * Two sequences with different lengths are always considered unequal.
   * <p>
   * This method deliberately allows comparison of sequences of different types.
   * <p>
   * The iterators used as parameters will be both exhausted if this method returns {@code true}.
   * Otherwise, they will stand at the first inequality, which can mean that one is exhausted
   * while the other is not.
   * @param it1      iterator over the first sequence
   * @param it2      iterator over the second sequence
   * @param equality element equality definition, has to return {@code true} for elements considered,
   *                 and {@code false} for elements considered unequal
   * @return {@code true} if both sequences are considered equal<br>
   *         {@code false} if they are considered unequal
   * @param <T1> element type of first sequence
   * @param <T2> element type of second sequence
   */
  public static <T1, T2> boolean areEqual(@NotNull Iterator<T1> it1,
                                          @NotNull Iterator<T2> it2,
                                          @NotNull BiPredicate<? super T1, ? super T2> equality)
  {
    while (it1.hasNext()  &&  it2.hasNext()) {
      final T1 elem1 = it1.next();
      final T2 elem2 = it2.next();
      if (!equality.test(elem1, elem2)) {
        return false;
      }
    }
    return it1.hasNext() == it2.hasNext(); // only true if both are exhausted
  }

  /**
   * Compare two sequences of comparable elements and return the result.
   * This method will assume lexical ordering, i.e., if the sequences have the same
   * elements, but one is longer, the longer sequence will be sorted after the shorter sequence.
   * @param seq1     first sequence
   * @param seq2     second sequence
   * @return {@link Order} enum value describing the order of both sequences
   * @param <T> common (super) element type of both sequences
   */
  @NotNull
  public static <T extends Comparable<T>> Order compare(@NotNull Iterable<? extends T> seq1,
                                                        @NotNull Iterable<? extends T> seq2)
  {
    return compare(seq1.iterator(), seq2.iterator(), Ordering.natural());
  }

  /**
   * Compare two sequences of comparable elements and return the result.
   * This method will assume lexical ordering, i.e., if the sequences have the same
   * elements, but one is longer, the longer sequence will be sorted after the shorter sequence.
   * @param seq1     first sequence
   * @param seq2     second sequence
   * @param longerOrder order which is returned when both sequences contain the same elements,
   *                    but the second sequence is longer. If in this case the first sequence is
   *                    longer, the {@link Order#inverse()} of {@code longerOrder} will be returned.
   * @return {@link Order} enum value describing the order of both sequences
   * @param <T> common (super) element type of both sequences
   */
  @NotNull
  public static <T extends Comparable<T>> Order compare(@NotNull Iterable<? extends T> seq1,
                                                        @NotNull Iterable<? extends T> seq2,
                                                        @NotNull Order longerOrder)
  {
    return compare(seq1.iterator(), seq2.iterator(), longerOrder, Ordering.natural());
  }

  /**
   * Compare two sequences and return the result.
   * This method will assume lexical ordering, i.e., if the sequences have the same
   * elements, but one is longer, the longer sequence will be sorted after the shorter sequence.
   * @param seq1     first sequence
   * @param seq2     second sequence
   * @param ordering ordering relation
   * @return {@link Order} enum value describing the order of both sequences
   * @param <T> common (super) element type of both sequences
   */
  @NotNull
  public static <T> Order compare(@NotNull Iterable<? extends T> seq1,
                                  @NotNull Iterable<? extends T> seq2,
                                  @NotNull Ordering<? super T> ordering)
  {
    return compare(seq1.iterator(), seq2.iterator(), ordering);
  }

  /**
   * Compare two sequences and return the result.
   * @param seq1     first sequence
   * @param seq2     second sequence
   * @param longerOrder order which is returned when both sequences contain the same elements,
   *                    but the second sequence is longer. If in this case the first sequence is
   *                    longer, the {@link Order#inverse()} of {@code longerOrder} will be returned.
   * @param ordering ordering relation
   * @return {@link Order} enum value describing the order of both sequences
   * @param <T> common (super) element type of both sequences
   */
  @NotNull
  public static <T> Order compare(@NotNull Iterable<? extends T> seq1,
                                  @NotNull Iterable<? extends T> seq2,
                                  @NotNull Order longerOrder,
                                  @NotNull Ordering<? super T> ordering)
  {
    return compare(seq1.iterator(), seq2.iterator(), longerOrder, ordering);
  }

  /**
   * Compare the iterators of two sequences and return the result.
   * This method will assume lexical ordering, i.e., if the sequences have the same
   * elements, but one is longer, the longer sequence will be sorted after the shorter sequence.
   * @param it1      iterator of the first sequence (changed)
   * @param it2      iterator of the second sequence (changed)
   * @param ordering ordering relation
   * @return {@link Order} enum value describing the order of both sequences
   * @param <T> common (super) element type of both sequences
   */
  @NotNull
  public static <T> Order compare(@NotNull Iterator<? extends T> it1,
                                  @NotNull Iterator<? extends T> it2,
                                  @NotNull Ordering<? super T> ordering)
  {
    return compare(it1, it2, Order.Ascending, ordering);
  }

  /**
   * Compare the iterators of two sequences and return the result.
   * This method will assume lexical ordering, i.e., if the sequences have the same
   * elements, but one is longer, the longer sequence will be sorted after the shorter sequence.
   * @param it1      iterator of the first sequence (changed)
   * @param it2      iterator of the second sequence (changed)
   * @return {@link Order} enum value describing the order of both sequences
   * @param <T> common (super) element type of both sequences
   */
  @NotNull
  public static <T extends Comparable<T>> Order compare(@NotNull Iterator<? extends T> it1,
                                                        @NotNull Iterator<? extends T> it2)
  {
    return compare(it1, it2, Order.Ascending, Ordering.natural());
  }

  /**
   * Compare the iterators of two sequences of comparable elements and return the result.
   * @param it1      iterator of the first sequence (changed)
   * @param it2      iterator of the second sequence (changed)
   * @param longerOrder order which is returned when both sequences contain the same elements,
   *                    but the second sequence is longer. If in this case the first sequence is
   *                    longer, the {@link Order#inverse()} of {@code longerOrder} will be returned.
   * @return {@link Order} enum value describing the order of both sequences
   * @param <T> common (super) element type of both sequences
   */
  @NotNull
  public static <T extends Comparable<T>> Order compare(@NotNull Iterator<? extends T> it1,
                                                        @NotNull Iterator<? extends T> it2,
                                                        @NotNull Order longerOrder)
  {
    return compare(it1, it2, longerOrder, Ordering.natural());
  }

  /**
   * Compare the iterators of two sequences and return the result.
   * @param it1      iterator of the first sequence (changed)
   * @param it2      iterator of the second sequence (changed)
   * @param longerOrder order which is returned when both sequences contain the same elements,
   *                    but the second sequence is longer. If in this case the first sequence is
   *                    longer, the {@link Order#inverse()} of {@code longerOrder} will be returned.
   * @param ordering ordering relation
   * @return {@link Order} enum value describing the order of both sequences
   * @param <T> common (super) element type of both sequences
   */
  @NotNull
  public static <T> Order compare(@NotNull Iterator<? extends T> it1,
                                  @NotNull Iterator<? extends T> it2,
                                  @NotNull Order longerOrder,
                                  @NotNull Ordering<? super T> ordering)
  {
    while (it1.hasNext()  &&  it2.hasNext()) {
      final T elem1 = it1.next();
      final T elem2 = it2.next();
      final Order order = ordering.check(elem1, elem2);
      if (order != Order.Same) {
        return order;
      }
    }
    final boolean firstHasNext = it1.hasNext();
    if (firstHasNext == it2.hasNext()) {
      // when coming here, both iterators are exhausted
      return Order.Same;
    }
    return firstHasNext
            ? longerOrder.inverse()
            : longerOrder;
  }
}
