// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.lang.ref.Reference;
import java.util.*;

/**
 * A list which holds its elements as references.
 * The list will be automatically cleaned up when an iterator is constructed.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class RefWrapperList<V>
        implements List<V>
{
  private final ReferenceCreator<? extends Reference<V>, V> referenceCreator;

  /** The basic list wrapped by this wrapper. */
  private final List<Reference<V>> wrappedList;

  /**
   * Default constructor.
   * This will use a wrapped linked list.
   * @param creator reference creator used to create the internally stored references
   */
  public RefWrapperList(@NotNull ReferenceCreator<? extends Reference<V>, V> creator)
  {
    referenceCreator = creator;
    wrappedList = new LinkedList<>();
  }

  /**
   * Create a reference wrapper list from a collection.
   * This will use a wrapped linked list.
   *
   * @param creator reference creator used to create the internally stored references
   * @param c collection the collection elements will be wrapped in the given reference type and added to this list
   */
  public RefWrapperList(@NotNull ReferenceCreator<? extends Reference<V>, V> creator,
                        Collection<? extends V> c)
  {
    this(creator);
    addAll(c);
  }

  /**
   * Constructor.
   * This allows to define a dedicated underlying list.
   * As this wrapper has to have complete control of this list it should not be used elsewhere.
   * @param creator reference creator used to create the internally stored references
   * @param wrappedList wrapped list
   */
  protected RefWrapperList(@NotNull ReferenceCreator<? extends Reference<V>, V> creator,
                           @NotNull List<Reference<V>> wrappedList)
  {
    referenceCreator = creator;
    this.wrappedList = wrappedList;
  }

  /**
   * Returns the number of elements in this list.  If this list contains
   * more than <tt>Integer.MAX_VALUE</tt> elements, returns
   * <tt>Integer.MAX_VALUE</tt>.
   *
   * @return the number of elements in this list.
   */
  @Override
  public int size()
  {
    return wrappedList.size();
  }

  /**
   * Returns <tt>true</tt> if this list contains no elements.
   *
   * @return <tt>true</tt> if this list contains no elements.
   */
  @Override
  public boolean isEmpty()
  {
    return wrappedList.isEmpty();
  }

  /**
   * Returns <tt>true</tt> if this list contains the specified element.
   * More formally, returns <tt>true</tt> if and only if this list contains
   * at least one element <tt>e</tt> such that
   * <tt>(o==null&nbsp;?&nbsp;e==null&nbsp;:&nbsp;o.equals(e))</tt>.
   *
   * @param o element whose presence in this list is to be tested.
   * @return <tt>true</tt> if this list contains the specified element.
   * @throws ClassCastException   if the type of the specified element
   *                              is incompatible with this list (optional).
   * @throws NullPointerException if the specified element is null and this
   *                              list does not support null elements (optional).
   */
  @Override
  public boolean contains(Object o)
  {
    for (Reference<V> ref : wrappedList) {
      if (ref == null) {
        // Null value
        if (o == null) {
          return true;
        }
      }
      else {
        V value = ref.get();
        if (value != null) {
          if (value.equals(o)) {
            return true;
          }
        }
      }
    }
    return false;
  }

  /**
   * Returns an iterator over the elements in this list in proper sequence.
   *
   * @return an iterator over the elements in this list in proper sequence.
   */
  @Override
  @NotNull
  public Iterator<V> iterator()
  {
    return toDirectList().iterator();
  }

  /**
   * Returns an array containing all of the elements in this list in proper
   * sequence.  Obeys the general contract of the
   * <tt>Collection.toArray</tt> method.
   *
   * @return an array containing all of the elements in this list in proper
   *         sequence.
   * @see java.util.Arrays#asList(Object[])
   */
  @Override
  @NotNull
  public Object[] toArray()
  {
    return toDirectList().toArray();
  }

  /**
   * Returns an array containing all of the elements in this list in proper
   * sequence; the runtime type of the returned array is that of the
   * specified array.  Obeys the general contract of the
   * <tt>Collection.toArray(Object[])</tt> method.
   *
   * @param a the array into which the elements of this list are to
   *          be stored, if it is big enough; otherwise, a new array of the
   *          same runtime type is allocated for this purpose.
   * @return an array containing the elements of this list.
   * @throws ArrayStoreException  if the runtime type of the specified array
   *                              is not a supertype of the runtime type of every element in
   *                              this list.
   * @throws NullPointerException if the specified array is <tt>null</tt>.
   */
  @Override
  @NotNull
  public <T> T[] toArray(@NotNull T[] a)
  {
    return toDirectList().toArray(a);
  }

  /**
   * Appends the specified element to the end of this list (optional
   * operation). <p>
   *
   * Lists that support this operation may place limitations on what
   * elements may be added to this list.  In particular, some
   * lists will refuse to add null elements, and others will impose
   * restrictions on the type of elements that may be added.  List
   * classes should clearly specify in their documentation any restrictions
   * on what elements may be added.
   *
   * @param o element to be appended to this list.
   * @return <tt>true</tt> (as per the general contract of the
   *         <tt>Collection.add</tt> method).
   * @throws UnsupportedOperationException if the <tt>add</tt> method is not
   *                                       supported by this list.
   * @throws ClassCastException            if the class of the specified element
   *                                       prevents it from being added to this list.
   * @throws NullPointerException          if the specified element is null and this
   *                                       list does not support null elements.
   * @throws IllegalArgumentException      if some aspect of this element
   *                                       prevents it from being added to this list.
   */
  @Override
  public boolean add(V o)
  {
    Reference<V> ref = createReference(o);
    return wrappedList.add(ref);
  }

  /**
   * Removes the first occurrence in this list of the specified element
   * (optional operation).  If this list does not contain the element, it is
   * unchanged.  More formally, removes the element with the lowest index i
   * such that <tt>(o==null ? get(i)==null : o.equals(get(i)))</tt> (if
   * such an element exists).
   *
   * @param o element to be removed from this list, if present.
   * @return <tt>true</tt> if this list contained the specified element.
   * @throws ClassCastException            if the type of the specified element
   *                                       is incompatible with this list (optional).
   * @throws NullPointerException          if the specified element is null and this
   *                                       list does not support null elements (optional).
   * @throws UnsupportedOperationException if the <tt>remove</tt> method is
   *                                       not supported by this list.
   */
  @Override
  public boolean remove(Object o)
  {
    for (ListIterator<Reference<V>> it = wrappedList.listIterator();  it.hasNext(); ) {
      Reference<V> ref = it.next();
      if (ref == null) {
        if (o == null) {
          it.remove();
          return true;
        }
      }
      else {
        if (ref.equals(o)) {
          it.remove();
          return true;
        }
      }
    }
    return false;
  }

  /**
   * Returns <tt>true</tt> if this list contains all of the elements of the
   * specified collection.
   *
   * @param c collection to be checked for containment in this list.
   * @return <tt>true</tt> if this list contains all of the elements of the
   *         specified collection.
   * @throws ClassCastException   if the types of one or more elements
   *                              in the specified collection are incompatible with this
   *                              list (optional).
   * @throws NullPointerException if the specified collection contains one
   *                              or more null elements and this list does not support null
   *                              elements (optional);
   *                              or if the specified collection is
   *                              <tt>null</tt>.
   * @see #contains(Object)
   */
  @Override
  public boolean containsAll(@NotNull Collection<?> c)
  {
    // this is O(n^2), but whatever
    for (Object o : c) {
      if (!contains(o)) {
        return false;
      }
    }
    return true;
  }

  /**
   * Appends all of the elements in the specified collection to the end of
   * this list, in the order that they are returned by the specified
   * collection's iterator (optional operation).  The behavior of this
   * operation is unspecified if the specified collection is modified while
   * the operation is in progress.  (Note that this will occur if the
   * specified collection is this list, and it's nonempty.)
   *
   * @param c collection whose elements are to be added to this list.
   * @return <tt>true</tt> if this list changed as a result of the call.
   * @throws UnsupportedOperationException if the <tt>addAll</tt> method is
   *                                       not supported by this list.
   * @throws ClassCastException            if the class of an element in the specified
   *                                       collection prevents it from being added to this list.
   * @throws NullPointerException          if the specified collection contains one
   *                                       or more null elements and this list does not support null
   *                                       elements, or if the specified collection is <tt>null</tt>.
   * @throws IllegalArgumentException      if some aspect of an element in the
   *                                       specified collection prevents it from being added to this
   *                                       list.
   * @see #add(Object)
   */
  @Override
  public boolean addAll(@NotNull Collection<? extends V> c)
  {
    boolean result = false;
    for (V v : c) {
      if(add(v)) {
        result = true;
      }
    }
    return result;
  }

  /**
   * Inserts all of the elements in the specified collection into this
   * list at the specified position (optional operation).  Shifts the
   * element currently at that position (if any) and any subsequent
   * elements to the right (increases their indices).  The new elements
   * will appear in this list in the order that they are returned by the
   * specified collection's iterator.  The behavior of this operation is
   * unspecified if the specified collection is modified while the
   * operation is in progress.  (Note that this will occur if the specified
   * collection is this list, and it's nonempty.)
   *
   * @param index index at which to insert first element from the specified
   *              collection.
   * @param c     elements to be inserted into this list.
   * @return <tt>true</tt> if this list changed as a result of the call.
   * @throws UnsupportedOperationException if the <tt>addAll</tt> method is
   *                                       not supported by this list.
   * @throws ClassCastException            if the class of one of elements of the
   *                                       specified collection prevents it from being added to this
   *                                       list.
   * @throws NullPointerException          if the specified collection contains one
   *                                       or more null elements and this list does not support null
   *                                       elements, or if the specified collection is <tt>null</tt>.
   * @throws IllegalArgumentException      if some aspect of one of elements of
   *                                       the specified collection prevents it from being added to
   *                                       this list.
   * @throws IndexOutOfBoundsException     if the index is out of range (index
   *                                       &lt; 0 || index &gt; size()).
   */
  @Override
  public boolean addAll(int index, @NotNull Collection<? extends V> c)
  {
    List<Reference<V>> refList = new ArrayList<>(c.size());
    for (V v : c) {
      refList.add(createReference(v));
    }
    return wrappedList.addAll(index, refList);
  }

  /**
   * Removes from this list all the elements that are contained in the
   * specified collection (optional operation).
   *
   * @param c collection that defines which elements will be removed from
   *          this list.
   * @return <tt>true</tt> if this list changed as a result of the call.
   * @throws UnsupportedOperationException if the <tt>removeAll</tt> method
   *                                       is not supported by this list.
   * @throws ClassCastException            if the types of one or more elements
   *                                       in this list are incompatible with the specified
   *                                       collection (optional).
   * @throws NullPointerException          if this list contains one or more
   *                                       null elements and the specified collection does not support
   *                                       null elements (optional);
   *                                       or if the specified collection is
   *                                       <tt>null</tt>.
   * @see #remove(Object)
   * @see #contains(Object)
   */
  @Override
  public boolean removeAll(@NotNull Collection<?> c)
  {
    boolean result = false;
    for (ListIterator<Reference<V>> it = wrappedList.listIterator();  it.hasNext(); ) {
      Reference<V> ref = it.next();
      V value = ref == null ? null : ref.get();
      if (c.contains(value)) {
        it.remove();
        result = true;
      }
    }
    return result;
  }

  /**
   * Retains only the elements in this list that are contained in the
   * specified collection (optional operation).  In other words, removes
   * from this list all the elements that are not contained in the specified
   * collection.
   *
   * @param c collection that defines which elements this set will retain.
   * @return <tt>true</tt> if this list changed as a result of the call.
   * @throws UnsupportedOperationException if the <tt>retainAll</tt> method
   *                                       is not supported by this list.
   * @throws ClassCastException            if the types of one or more elements
   *                                       in this list are incompatible with the specified
   *                                       collection (optional).
   * @throws NullPointerException          if this list contains one or more
   *                                       null elements and the specified collection does not support
   *                                       null elements (optional);
   *                                       or if the specified collection is
   *                                       <tt>null</tt>.
   * @see #remove(Object)
   * @see #contains(Object)
   */
  @Override
  public boolean retainAll(@NotNull Collection<?> c)
  {
    boolean result = false;
    for (ListIterator<Reference<V>> it = wrappedList.listIterator();  it.hasNext(); ) {
      Reference<V> ref = it.next();
      V value = ref == null ? null : ref.get();
      if (!c.contains(value)) {
        it.remove();
        result = true;
      }
    }
    return result;
  }

  /**
   * Removes all of the elements from this list (optional operation).  This
   * list will be empty after this call returns (unless it throws an
   * exception).
   *
   * @throws UnsupportedOperationException if the <tt>clear</tt> method is
   *                                       not supported by this list.
   */
  @Override
  public void clear()
  {
    wrappedList.clear();
  }

  /**
   * Returns the element at the specified position in this list.
   *
   * @param index index of element to return.
   * @return the element at the specified position in this list.
   * @throws IndexOutOfBoundsException if the index is out of range (index
   *                                   &lt; 0 || index &gt;= size()).
   */
  @Override
  public V get(int index)
  {
    Reference<V> ref = wrappedList.get(index);
    return ref == null ? null : ref.get();
  }

  /**
   * Replaces the element at the specified position in this list with the
   * specified element (optional operation).
   *
   * @param index   index of element to replace.
   * @param element element to be stored at the specified position.
   * @return the element previously at the specified position.
   * @throws UnsupportedOperationException if the <tt>set</tt> method is not
   *                                       supported by this list.
   * @throws ClassCastException            if the class of the specified element
   *                                       prevents it from being added to this list.
   * @throws NullPointerException          if the specified element is null and
   *                                       this list does not support null elements.
   * @throws IllegalArgumentException      if some aspect of the specified
   *                                       element prevents it from being added to this list.
   * @throws IndexOutOfBoundsException     if the index is out of range
   *                                       (index &lt; 0 || index &gt;= size()).
   */
  @Override
  public V set(int index, V element)
  {
    Reference<V> ref = wrappedList.set(index, createReference(element));
    return ref == null ? null : ref.get();
  }

  /**
   * Inserts the specified element at the specified position in this list
   * (optional operation).  Shifts the element currently at that position
   * (if any) and any subsequent elements to the right (adds one to their
   * indices).
   *
   * @param index   index at which the specified element is to be inserted.
   * @param element element to be inserted.
   * @throws UnsupportedOperationException if the <tt>add</tt> method is not
   *                                       supported by this list.
   * @throws ClassCastException            if the class of the specified element
   *                                       prevents it from being added to this list.
   * @throws NullPointerException          if the specified element is null and
   *                                       this list does not support null elements.
   * @throws IllegalArgumentException      if some aspect of the specified
   *                                       element prevents it from being added to this list.
   * @throws IndexOutOfBoundsException     if the index is out of range
   *                                       (index &lt; 0 || index &gt; size()).
   */
  @Override
  public void add(int index, V element)
  {
    wrappedList.add(index, createReference(element));
  }

  /**
   * Removes the element at the specified position in this list (optional
   * operation).  Shifts any subsequent elements to the left (subtracts one
   * from their indices).  Returns the element that was removed from the
   * list.
   *
   * @param index the index of the element to removed.
   * @return the element previously at the specified position.
   * @throws UnsupportedOperationException if the <tt>remove</tt> method is
   *                                       not supported by this list.
   * @throws IndexOutOfBoundsException     if the index is out of range (index
   *                                       &lt; 0 || index &gt;= size()).
   */
  @Override
  public V remove(int index)
  {
    Reference<V> ref = wrappedList.remove(index);
    return ref == null ? null : ref.get();
  }

  /**
   * Returns the index in this list of the first occurrence of the specified
   * element, or -1 if this list does not contain this element.
   * More formally, returns the lowest index <tt>i</tt> such that
   * <tt>(o==null ? get(i)==null : o.equals(get(i)))</tt>,
   * or -1 if there is no such index.
   *
   * @param o element to search for.
   * @return the index in this list of the first occurrence of the specified
   *         element, or -1 if this list does not contain this element.
   * @throws ClassCastException   if the type of the specified element
   *                              is incompatible with this list (optional).
   * @throws NullPointerException if the specified element is null and this
   *                              list does not support null elements (optional).
   */
  @Override
  public int indexOf(Object o)
  {
    int index = 0;
    for (Reference<V> ref : wrappedList) {
      if (ref == null) {
        if (o == null) {
          return index;
        }
      }
      else if (ref.equals(o)) {
        return index;
      }
      ++index;
    }
    return -1;
  }

  /**
   * Returns the index in this list of the last occurrence of the specified
   * element, or -1 if this list does not contain this element.
   * More formally, returns the highest index <tt>i</tt> such that
   * <tt>(o==null ? get(i)==null : o.equals(get(i)))</tt>,
   * or -1 if there is no such index.
   *
   * @param o element to search for.
   * @return the index in this list of the last occurrence of the specified
   *         element, or -1 if this list does not contain this element.
   * @throws ClassCastException   if the type of the specified element
   *                              is incompatible with this list (optional).
   * @throws NullPointerException if the specified element is null and this
   *                              list does not support null elements (optional).
   */
  @Override
  public int lastIndexOf(Object o)
  {
    int index = 0;
    int found = -1;
    for (Reference<V> ref : wrappedList) {
      if (ref == null) {
        if (o == null) {
          found = index;
        }
      }
      else if (ref.equals(o)) {
        found = index;
      }
      ++index;
    }
    return found;
  }

  /**
   * Returns a list iterator of the elements in this list (in proper
   * sequence).
   *
   * @return a list iterator of the elements in this list (in proper
   *         sequence).
   */
  @Override
  @NotNull
  public ListIterator<V> listIterator()
  {
    return toDirectList().listIterator();
  }

  /**
   * Returns a list iterator of the elements in this list (in proper
   * sequence), starting at the specified position in this list.  The
   * specified index indicates the first element that would be returned by
   * an initial call to the <tt>next</tt> method.  An initial call to
   * the <tt>previous</tt> method would return the element with the
   * specified index minus one.
   *
   * @param index index of first element to be returned from the
   *              list iterator (by a call to the <tt>next</tt> method).
   * @return a list iterator of the elements in this list (in proper
   *         sequence), starting at the specified position in this list.
   * @throws IndexOutOfBoundsException if the index is out of range (index
   *                                   &lt; 0 || index &gt; size()).
   */
  @Override
  @NotNull
  public ListIterator<V> listIterator(int index)
  {
    return toDirectList().listIterator(index);
  }

  /**
   * Returns a view of the portion of this list between the specified
   * <tt>fromIndex</tt>, inclusive, and <tt>toIndex</tt>, exclusive.  (If
   * <tt>fromIndex</tt> and <tt>toIndex</tt> are equal, the returned list is
   * empty.)  The returned list is backed by this list, so non-structural
   * changes in the returned list are reflected in this list, and vice-versa.
   * The returned list supports all of the optional list operations supported
   * by this list.<p>
   *
   * This method eliminates the need for explicit range operations (of
   * the sort that commonly exist for arrays).   Any operation that expects
   * a list can be used as a range operation by passing a subList view
   * instead of a whole list.  For example, the following idiom
   * removes a range of elements from a list:
   * <pre>
   * 	    list.subList(from, to).clear();
   * </pre>
   * Similar idioms may be constructed for <tt>indexOf</tt> and
   * <tt>lastIndexOf</tt>, and all of the algorithms in the
   * <tt>Collections</tt> class can be applied to a subList.<p>
   *
   * The semantics of the list returned by this method become undefined if
   * the backing list (i.e., this list) is <i>structurally modified</i> in
   * any way other than via the returned list.  (Structural modifications are
   * those that change the size of this list, or otherwise perturb it in such
   * a fashion that iterations in progress may yield incorrect results.)
   *
   * @param fromIndex low endpoint (inclusive) of the subList.
   * @param toIndex   high endpoint (exclusive) of the subList.
   * @return a view of the specified range within this list.
   * @throws IndexOutOfBoundsException for an illegal endpoint index value
   *                                   (fromIndex &lt; 0 || toIndex &gt; size || fromIndex &gt; toIndex).
   */
  @Override
  @NotNull
  public List<V> subList(int fromIndex, int toIndex)
  {
    return new RefWrapperList<>(referenceCreator, wrappedList.subList(fromIndex, toIndex));
  }

  /**
   * Copy the referenced elements into a directly referenced list.
   * By doing this the elements in this duplicate list will be excluded from garbage collection.
   * This does also do cleanup lost references from the list.
   * @return directly referenced list
   */
  private List<V> toDirectList()
  {
    ArrayList<V> list = new ArrayList<>(wrappedList.size());
    for (ListIterator<Reference<V>> it = wrappedList.listIterator();
            it.hasNext();
            ) {
      Reference<V> ref = it.next();
      if (ref == null) {
        // null value
        list.add(null);
      }
      else {
        V value = ref.get();
        if (value == null) {
          // garbage collected, remove
          it.remove();
        }
        else {
          list.add(value);
        }
      }
    }
    return list;
  }

  private Reference<V> createReference(V value)
  {
    return value == null ? null : referenceCreator.createReference(value);
  }

  /**
   * Factory method which allows to define a given underlying list.
   * This allows to define a dedicated underlying list.
   * As this wrapper has to have complete control of this list it should not be used elsewhere.
   * The recommended way of invoking this constructor is (e.g. for ArrayList)
   * <blockquote><pre>
   *   refWrapperList =
   *      RefWrapperList.create(ReferenceUtil.getSoftReferenceCreator
   *                            new ArrayList&lt;Reference&lt;XY&gt;&gt;());
   * }
   * </pre></blockquote>
   * @param creator reference creator
   * @param baseList basic list
   * @param <W> value type
   * @return reference wrapper list based on baseList
   */
  public static <W> RefWrapperList<W> create(@NotNull ReferenceCreator<? extends Reference<W>, W> creator,
                                             @NotNull List<Reference<W>> baseList)
  {
    return new RefWrapperList<>(creator, baseList);
  }

}
