// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

import java.lang.ref.Reference;
import java.util.*;

/**
 * A hash st which holds its elements as references.
 * The list will be automatically cleaned up when an iterator is constructed.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class RefWrapperHashSet<V>
        implements Set<V>
{
  private final ReferenceCreator<? extends Reference<V>, V> referenceCreator;

  private static final class HashReferenceWrapper<W>
  {
    private final Reference<W> reference;

    private HashReferenceWrapper(Reference<W> reference)
    {
      this.reference = reference;
    }

    public Reference<W> getReference()
    {
      return reference;
    }

    @Override
    public boolean equals(Object o)
    {
      if (this == o) {
        return true;
      }
      if (o == null || getClass() != o.getClass()) {
        return false;
      }

      final HashReferenceWrapper<?> that = (HashReferenceWrapper<?>)o;

      if (reference == null || that.reference == null) {
        return reference == that.reference;
      }

      Object value = reference.get();
      Object thatValue = that.reference.get();

      return value == null ? thatValue == null : value.equals(thatValue);
    }

    @Override
    public int hashCode()
    {
      if (reference == null) {
        return 0;
      }
      final Object value = reference.get();
      return value != null ? value.hashCode() : 0;
    }
  }

  /** The basic set wrapped by this wrapper. */
  private final HashSet<HashReferenceWrapper<V>> wrappedSet;

  /**
   * Default constructor.
   * This will use a wrapped hash set
   * @param creator reference creator used to create the internally stored references
   */
  public RefWrapperHashSet(@NotNull ReferenceCreator<? extends Reference<V>, V> creator)
  {
    referenceCreator = creator;
    wrappedSet = new HashSet<>();
  }

  /**
   * Create a reference wrapper list from a collection.
   * This will use a wrapped linked list and no reference queue.
   *
   * @param creator reference creator used to create the internally stored references
   * @param c collection the collection elements will be wrapped in the given reference type and added to this list
   */
  public RefWrapperHashSet(@NotNull ReferenceCreator<? extends Reference<V>, V> creator,
                           @NotNull Collection<? extends V> c)
  {
    this(creator);
    addAll(c);
  }

  /**
   * Returns the number of elements in this list.  If this list contains
   * more than <tt>Integer.MAX_VALUE</tt> elements, returns
   * <tt>Integer.MAX_VALUE</tt>.
   *
   * @return the number of elements in this list.
   */
  @Override
  public int size()
  {
    return wrappedSet.size();
  }

  /**
   * Returns <tt>true</tt> if this list contains no elements.
   *
   * @return <tt>true</tt> if this list contains no elements.
   */
  @Override
  public boolean isEmpty()
  {
    return wrappedSet.isEmpty();
  }

  /**
   * Returns <tt>true</tt> if this list contains the specified element.
   * More formally, returns <tt>true</tt> if and only if this list contains
   * at least one element <tt>e</tt> such that
   * <tt>(o==null&nbsp;?&nbsp;e==null&nbsp;:&nbsp;o.equals(e))</tt>.
   *
   * @param o element whose presence in this list is to be tested.
   * @return <tt>true</tt> if this list contains the specified element.
   * @throws ClassCastException   if the type of the specified element
   *                              is incompatible with this list (optional).
   * @throws NullPointerException if the specified element is null and this
   *                              list does not support null elements (optional).
   */
  @Override
  @SuppressWarnings("unchecked")
  public boolean contains(Object o)
  {
    return wrappedSet.contains(createReference((V)o));
  }

  /**
   * Returns an iterator over the elements in this list in proper sequence.
   *
   * @return an iterator over the elements in this list in proper sequence.
   */
  @Override
  @NotNull
  public Iterator<V> iterator()
  {
    return toDirectSet().iterator();
  }

  /**
   * Returns an array containing all of the elements in this list in proper
   * sequence.  Obeys the general contract of the
   * <tt>Collection.toArray</tt> method.
   *
   * @return an array containing all of the elements in this list in proper
   *         sequence.
   * @see java.util.Arrays#asList(Object[])
   */
  @Override
  @NotNull
  public Object[] toArray()
  {
    return toDirectSet().toArray();
  }

  /**
   * Returns an array containing all of the elements in this list in proper
   * sequence; the runtime type of the returned array is that of the
   * specified array.  Obeys the general contract of the
   * <tt>Collection.toArray(Object[])</tt> method.
   *
   * @param a the array into which the elements of this list are to
   *          be stored, if it is big enough; otherwise, a new array of the
   *          same runtime type is allocated for this purpose.
   * @return an array containing the elements of this list.
   * @throws ArrayStoreException  if the runtime type of the specified array
   *                              is not a supertype of the runtime type of every element in
   *                              this list.
   * @throws NullPointerException if the specified array is <tt>null</tt>.
   */
  @Override
  @NotNull
  public <U> U[] toArray(@NotNull U[] a)
  {
    return toDirectSet().toArray(a);
  }

  /**
   * Appends the specified element to the end of this list (optional
   * operation). <p>
   *
   * Lists that support this operation may place limitations on what
   * elements may be added to this list.  In particular, some
   * lists will refuse to add null elements, and others will impose
   * restrictions on the type of elements that may be added.  List
   * classes should clearly specify in their documentation any restrictions
   * on what elements may be added.
   *
   * @param o element to be appended to this list.
   * @return <tt>true</tt> (as per the general contract of the
   *         <tt>Collection.add</tt> method).
   * @throws UnsupportedOperationException if the <tt>add</tt> method is not
   *                                       supported by this list.
   * @throws ClassCastException            if the class of the specified element
   *                                       prevents it from being added to this list.
   * @throws NullPointerException          if the specified element is null and this
   *                                       list does not support null elements.
   * @throws IllegalArgumentException      if some aspect of this element
   *                                       prevents it from being added to this list.
   */
  @Override
  public boolean add(V o)
  {
    HashReferenceWrapper<V> ref = createReference(o);
    return wrappedSet.add(ref);
  }

  /**
   * Removes the first occurrence in this list of the specified element
   * (optional operation).  If this list does not contain the element, it is
   * unchanged.  More formally, removes the element with the lowest index i
   * such that <tt>(o==null ? get(i)==null : o.equals(get(i)))</tt> (if
   * such an element exists).
   *
   * @param o element to be removed from this list, if present.
   * @return <tt>true</tt> if this list contained the specified element.
   * @throws ClassCastException            if the type of the specified element
   *                                       is incompatible with this list (optional).
   * @throws NullPointerException          if the specified element is null and this
   *                                       list does not support null elements (optional).
   * @throws UnsupportedOperationException if the <tt>remove</tt> method is
   *                                       not supported by this list.
   */
  @Override
  @SuppressWarnings("unchecked")
  public boolean remove(Object o)
  {
    return wrappedSet.remove(createReference((V)o));
  }

  /**
   * Returns <tt>true</tt> if this list contains all of the elements of the
   * specified collection.
   *
   * @param c collection to be checked for containment in this list.
   * @return <tt>true</tt> if this list contains all of the elements of the
   *         specified collection.
   * @throws ClassCastException   if the types of one or more elements
   *                              in the specified collection are incompatible with this
   *                              list (optional).
   * @throws NullPointerException if the specified collection contains one
   *                              or more null elements and this list does not support null
   *                              elements (optional);
   *                              or if the specified collection is
   *                              <tt>null</tt>.
   * @see #contains(Object)
   */
  @Override
  public boolean containsAll(@NotNull Collection<?> c)
  {
    for (Object o : c) {
      if (!contains(o)) {
        return false;
      }
    }
    return true;
  }

  /**
   * Appends all of the elements in the specified collection to the end of
   * this list, in the order that they are returned by the specified
   * collection's iterator (optional operation).  The behavior of this
   * operation is unspecified if the specified collection is modified while
   * the operation is in progress.  (Note that this will occur if the
   * specified collection is this list, and it's nonempty.)
   *
   * @param c collection whose elements are to be added to this list.
   * @return <tt>true</tt> if this list changed as a result of the call.
   * @throws UnsupportedOperationException if the <tt>addAll</tt> method is
   *                                       not supported by this list.
   * @throws ClassCastException            if the class of an element in the specified
   *                                       collection prevents it from being added to this list.
   * @throws NullPointerException          if the specified collection contains one
   *                                       or more null elements and this list does not support null
   *                                       elements, or if the specified collection is <tt>null</tt>.
   * @throws IllegalArgumentException      if some aspect of an element in the
   *                                       specified collection prevents it from being added to this
   *                                       list.
   * @see #add(Object)
   */
  @Override
  public boolean addAll(@NotNull Collection<? extends V> c)
  {
    boolean result = false;
    for (V v : c) {
      if(add(v)) {
        result = true;
      }
    }
    return result;
  }

  /**
   * Removes from this list all the elements that are contained in the
   * specified collection (optional operation).
   *
   * @param c collection that defines which elements will be removed from
   *          this list.
   * @return <tt>true</tt> if this list changed as a result of the call.
   * @throws UnsupportedOperationException if the <tt>removeAll</tt> method
   *                                       is not supported by this list.
   * @throws ClassCastException            if the types of one or more elements
   *                                       in this list are incompatible with the specified
   *                                       collection (optional).
   * @throws NullPointerException          if this list contains one or more
   *                                       null elements and the specified collection does not support
   *                                       null elements (optional);
   *                                       or if the specified collection is
   *                                       <tt>null</tt>.
   * @see #remove(Object)
   * @see #contains(Object)
   */
  @Override
  public boolean removeAll(@NotNull Collection<?> c)
  {
    boolean result = false;
    for (Object o : c) {
      if (remove(o)) {
        result = true;
      }
    }
    return result;
  }

  /**
   * Retains only the elements in this list that are contained in the
   * specified collection (optional operation).  In other words, removes
   * from this list all the elements that are not contained in the specified
   * collection.
   *
   * @param c collection that defines which elements this set will retain.
   * @return <tt>true</tt> if this list changed as a result of the call.
   * @throws UnsupportedOperationException if the <tt>retainAll</tt> method
   *                                       is not supported by this list.
   * @throws ClassCastException            if the types of one or more elements
   *                                       in this list are incompatible with the specified
   *                                       collection (optional).
   * @throws NullPointerException          if this list contains one or more
   *                                       null elements and the specified collection does not support
   *                                       null elements (optional);
   *                                       or if the specified collection is
   *                                       <tt>null</tt>.
   * @see #remove(Object)
   * @see #contains(Object)
   */
  @Override
  @SuppressWarnings("unchecked")
  public boolean retainAll(@NotNull Collection<?> c)
  {
    List<HashReferenceWrapper<V>> list = new ArrayList<>(c.size());
    for (Object o : c) {
      list.add(createReference((V)o));
    }
    return wrappedSet.retainAll(list);
  }

  /**
   * Removes all of the elements from this list (optional operation).  This
   * list will be empty after this call returns (unless it throws an
   * exception).
   *
   * @throws UnsupportedOperationException if the <tt>clear</tt> method is
   *                                       not supported by this list.
   */
  @Override
  public void clear()
  {
    wrappedSet.clear();
  }

  /**
   * Copy the referenced elements into a directly referenced list.
   * By doing this the elements in this duplicate list will be excluded from garbage collection.
   * This does also do cleanup lost references from the list.
   * @return directly referenced list
   */
  private Set<V> toDirectSet()
  {
    HashSet<V> set = new HashSet<>(wrappedSet.size(), 0.75f);
    LinkedList<HashReferenceWrapper<V>> discarded = new LinkedList<>();
    for (HashReferenceWrapper<V> w : wrappedSet) {
      Reference<V> ref = w.getReference();
      if (ref == null) {
        set.add(null);
      }
      else {
        V value = ref.get();
        if (value == null) {
          discarded.add(w);
        }
        else {
          set.add(value);
        }
      }
    }
    // do cleanup
    for (HashReferenceWrapper<V> w : discarded) {
      wrappedSet.remove(w);
    }
    return set;
  }

  private HashReferenceWrapper<V> createReference(V value)
  {
    return new HashReferenceWrapper<>(value == null ? null : referenceCreator.createReference(value));
  }

}
