// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2021-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.function.FragileFunction1;
import de.caff.generics.function.Procedure2;
import de.caff.generics.util.Counter;
import de.caff.generics.util.ThreadSafeCounter;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.function.Function;

/**
 * A least recently used cache which also allows to predefine a way to recreate lost entries.
 * <p>
 * A good way to use it is construction with a function which is able to recreate
 * the value for a given key and calling only the {@link #get(Object)} method.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since May 10, 2021
 */
public class RecreatingLeastRecentlyUsedCache<K, V>
        extends LeastRecentlyUsedCache<K, V>
{
  @NotNull
  private final Function<K, V> recreator;
  @NotNull
  private final Map<K, Future<V>> pendingRecreations = new HashMap<>();
  @NotNull
  private final Counter numRecreations = new ThreadSafeCounter();

  /**
   * Constructor.
   *
   * @param mostRecentLimit maximal number of cached values, non-negative
   * @param recreator       creator for values which are currently not present
   */
  public RecreatingLeastRecentlyUsedCache(int mostRecentLimit, @NotNull Function<K, V> recreator)
  {
    super(mostRecentLimit);
    this.recreator = recreator;
  }

  /**
   * Constructor.
   * <p>
   * This allows to use an recreator which might throw an exception.
   * In case of exceptions obviously nothing is created, but the {@code errorHandler}
   * will be called with the key an exception.
   *
   * @param mostRecentLimit maximal number of cached values, non-negative
   * @param recreator       creator for values which are currently not present
   * @param errorHandler    error handler for exceptions during the recreation
   */
  public RecreatingLeastRecentlyUsedCache(int mostRecentLimit,
                                          @NotNull FragileFunction1<V, ? extends Exception, K> recreator,
                                          @NotNull Procedure2<K, Exception> errorHandler)
  {
    super(mostRecentLimit);
    this.recreator = k -> {
      try {
        return recreator.apply(k);
      } catch (Exception x) {
        errorHandler.apply(k, x);
      }
      return null;
    };
  }

  @Nullable
  @Override
  public V get(@NotNull K key)
  {
    final FutureTask<V> task;
    synchronized (pendingRecreations) {
      final Future<V> future = pendingRecreations.get(key);
      if (future != null) {
        try {
          return future.get();
        } catch (InterruptedException | ExecutionException e) {
          e.printStackTrace();
        }
      }

      final V value = super.get(key);
      if (value != null) {
        return value;
      }

      task = new FutureTask<>(() -> {
        final V result = recreator.apply(key);
        numRecreations.add1();
        if (result != null) {
          put(key, result);
        }
        return result;
      });
      pendingRecreations.put(key, task);
    }
    task.run();
    synchronized (pendingRecreations) {
      pendingRecreations.remove(key, task);
    }
    try {
      return task.get();
    } catch (InterruptedException | ExecutionException e) {
      e.printStackTrace();
      return null;
    }
  }

  @Override
  public void cleanup()
  {
    synchronized (pendingRecreations) {
      super.cleanup();
      pendingRecreations.clear();
    }
  }

  @Nullable
  @Override
  public V remove(K key)
  {
    synchronized (pendingRecreations) {
      pendingRecreations.remove(key);
    }
    return super.remove(key);
  }

  /**
   * Get the number of times the recreation function was called.
   * @return number of recreations
   */
  public int getNumRecreations()
  {
    return numRecreations.getValue();
  }
}