// ============================================================================
// File:               Pythonesque
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2021-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            05.12.21 15:57
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;

/**
 * Helper class for pythionesque indexing.
 * Python (the computer language) allows arrays and similar to be indexed with negative values
 * with the meaning &quot;count from the back&quot;. So the index {@code -1} will refer to the last element,
 * {@code -2} to the one before it and so on.
 * <p>
 * Note taht {@link #UNMAPPABLE} (same value as {@link Integer#MIN_VALUE} is handled special.
 * All methods here will throw an {@link IndexOutOfBoundsException} if the index to map
 * has has is {@link #UNMAPPABLE}. This allows to use this value as a special invalid index indicator even
 * when Pythonesque indexing is used.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 05, 2021
 */
public final class Pythonesque
{
  /**
   * Unmappable index, same as {@link Integer#MAX_VALUE}.
   * This index can be used as an impossible result even if
   * Pythonesque indexing is used, as it is not mappable.
   * The mapping methods here will all throw an {@link IndexOutOfBoundsException} if called with
   * an index of {@code UNMAPPABLE}.
   */
  public static final int UNMAPPABLE = Integer.MIN_VALUE;

  /** Format used for index out of bounds exception. */
  public static final String ERROR_INDEX_OUT_OF_BOUNDS_FORMAT = "Requested index [%d] for size %d!";

  /** Don't create. */
  private Pythonesque() {}

  /**
   * Map a possibly negative index from an indexable of a given size to
   * the standard Java index.
   * This method will not check the parameters for sanity.
   * @param index index to be mapped, must not be {@link #UNMAPPABLE}
   * @param size  size of the indexable element for which the index is meant
   * @return canonized index, possibly invalid if {@code index} is outside the
   *         range {@code [-size, size[}
   * @throws IndexOutOfBoundsException if index is {@link #UNMAPPABLE}
   */
  public static int map(int index, int size)
  {
    if (index >= 0) {
      return index;
    }
    if (index == UNMAPPABLE) {
      throw new IndexOutOfBoundsException(String.format(ERROR_INDEX_OUT_OF_BOUNDS_FORMAT, index, size));
    }
    return size + index;
  }

  /**
   * Map a possibly negative index from an indexable of a given size to
   * the standard Java index.
   * This method will not check the parameters for sanity.
   * @param index index to be mapped, must not be {@link #UNMAPPABLE}
   * @param indexable indexable which is inde
   * @return canonized index, possibly invalid if {@code index} is outside the
   *         range {@code [-size, size[}
   * @throws IndexOutOfBoundsException if index is {@link #UNMAPPABLE}
   */
  public static int map(int index, @NotNull Sizeable indexable)
  {
    if (index >= 0) {
      return index;
    }
    if (index == UNMAPPABLE) {
      throw new IndexOutOfBoundsException(String.format(ERROR_INDEX_OUT_OF_BOUNDS_FORMAT, index, indexable.size()));
    }
    return index + indexable.size();
  }

  /**
   * Map a possibly negative index from an indexable of a given size to
   * the standard Java index, but keep a lower bound.
   * This method will take care that {@code 0} is returned if a negative
   * index has a magnitude larger than the size.
   * @param index index to be mapped, must not be {@link #UNMAPPABLE}
   * @param size  size of the indexable element for which the index is meant
   * @return canonized index, possibly invalid if {@code index} is too large
   * @throws IndexOutOfBoundsException if index is {@link #UNMAPPABLE}
   */
  public static int mapLB(int index, int size)
  {
    if (index >= 0) {
      return index;
    }
    if (index < -size) {
      if (index == UNMAPPABLE) {
        throw new IndexOutOfBoundsException(String.format(ERROR_INDEX_OUT_OF_BOUNDS_FORMAT, index, size));
      }
      return 0;
    }
    return index + size;
  }

  /**
   * Map a possibly negative index from an indexable of a given size to
   * the standard Java index, but keep a lower bound.
   * This method will take care that {@code 0} is returned if a negative
   * index has a magnitude larger than the size.
   * @param index index to be mapped, must not be {@link #UNMAPPABLE}
   * @param indexable indexable which is inde
   * @return canonized index, possibly invalid if {@code index} is too large
   * @throws IndexOutOfBoundsException if index is {@link #UNMAPPABLE}
   */
  public static int mapLB(int index, @NotNull Sizeable indexable)
  {
    if (index >= 0) {
      return index;
    }
    if (index == UNMAPPABLE) {
      throw new IndexOutOfBoundsException(String.format(ERROR_INDEX_OUT_OF_BOUNDS_FORMAT, index, indexable.size()));
    }
    final int size = indexable.size();
    if (index < -size) {
      return 0;
    }
    return index + size;
  }

  /**
   * Map a possibly negative index from an indexable of a given size to
   * the standard Java index, and keep an upper bound.
   * This method will take care that {@code size - 1} is returned if a positive
   * index is larger or queal to size
   * @param index index to be mapped
   * @param size  size of the indexable element for which the index is meant
   * @return canonized index, possibly invalid if {@code index} is too large
   * @throws IndexOutOfBoundsException if index is {@link #UNMAPPABLE}
   */
  public static int mapUB(int index, int size)
  {
    if (index >= 0) {
      return index >= size
              ? size - 1
              : index;
    }
    if (index == UNMAPPABLE) {
      throw new IndexOutOfBoundsException(String.format(ERROR_INDEX_OUT_OF_BOUNDS_FORMAT, index, size));
    }
    return index + size;
  }

  /**
   * Map a possibly negative index from an indexable of a given size to
   * the standard Java index, but keep a lower bound.
   * This method will take care that {@code 0} is returned if a negative
   * index has a magnitude larger than the size.
   * @param index index to be mapped, must not be {@link #UNMAPPABLE}
   * @param indexable indexable which is inde
   * @return canonized index, possibly invalid if {@code index} is too large
   * @throws IndexOutOfBoundsException if index is {@link #UNMAPPABLE}
   */
  public static int mapUB(int index, @NotNull Sizeable indexable)
  {
    return mapUB(index, indexable.size());
  }

  /**
   * Map a possibly negative index from an indexable of a given size to
   * the standard Java index.
   * This method will throw an {@code IndexOutOfBoundsException} if {@code index} is negative
   * and out of range. It will not check positive indices, as it is expected that the returned
   * index is used in a way that it is checked anyway
   * @param index index to be mapped
   * @param size  size of the indexable element for which the index is meant
   * @return canonized index in the range {@code [0, size[}
   * @throws IndexOutOfBoundsException if index is outside the range {@code [-size, size[}.
   *                                   Note that this is always true for {@link #UNMAPPABLE}
   */
  public static int mapX(int index, int size)
  {
    if (index >= 0)  {
      return index;
    }
    if (index < -size) {
      throw new IndexOutOfBoundsException(String.format(ERROR_INDEX_OUT_OF_BOUNDS_FORMAT, index, size));
    }
    return index + size;
  }

  /**
   * Map a possibly negative index from an indexable of a given size to
   * the standard Java index.
   * This method will throw an {@code IndexOutOfBoundsException} if {@code index} is negative
   * and out of range. It will not check positive indices, as it is expected that the returned
   * index is used in a way that it is checked anyway
   * @param index index to be mapped
   * @param indexable indexable which is inde
   * @return canonized index in the range {@code [0, size[}
   * @throws IndexOutOfBoundsException if index is negative and smaller than {@code -size}.
   *                                   Note that this is always true for {@link #UNMAPPABLE}
   */
  public static int mapX(int index, @NotNull Sizeable indexable)
  {
    if (index >= 0) {
      return index;
    }
    final int size = indexable.size();
    if (index < -size) {
      throw new IndexOutOfBoundsException(String.format(ERROR_INDEX_OUT_OF_BOUNDS_FORMAT, index, size));
    }
    return index + size;
  }
}
