// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.function.Function1;

/**
 * Primitives helper tool.
 * <p>
 * This tool class provides several helper methods and helpful constants for handling  primitive Java values.
 * <p>
 * One large part is fixing Java's broken handling of {@code double} and {@code float} equality
 * and hashing. In Java the following assertion is valid and this can result in subtle errors:
 * <pre>{@code
 *     final double nz = -0.0;  // IEEE 784 negative zero
 *     final double pz = +0.0;  // IEEE 784 positive zero, same as 0.0
 *
 *     final Double abNZ = nz; // automatically boxed negative zero (i.e. Double.valueOf())
 *     final Double abPZ = pz; // automatically boxed positive zero (i.e. Double.valueOf())
 *
 *     assert abNZ == pz  &&  !abNZ.equals(pz);  // same, but not equal
 * }</pre>
 * Thanks to automatic boxing and unboxing this problem is hidden so deep in the Java language
 * that it is hard to overcome. This class provides some methods of checking floating
 * point values for equality or hashing them with a more natural handling of {@code 0.0}.
 * There is another problem with IEEE 784 having a lot more NaN values which Java
 * usually collapse into one value, but not for equality, hashing, and comparison. But in
 * normal usage this should pose no real problem. See
 * <a href="https://caff.de/posts/double_comparison/">Double Comparison</a> where I elaborate
 * (probably too much) about this mess.
 * <ul>
 *   <li>{@link #areEqual(double, double)} checks two {@code double} values for equality with correct zero handling and collapsed NaNs.</li>
 *   <li>{@link #areEqual(double[], double[])} checks two {@code double[]} arrays for equality with correct zero handling and collapsed NaNs.</li>
 *   <li>{@link #areEqual(float, float)} checks two {@code float} values for equality with correct zero handling and collapsed NaNs.</li>
 *   <li>{@link #areEqual(float[], float[])} checks two {@code float[]} arrays for equality with correct zero handling and collapsed NaNs.</li>
 *   <li>{@link #areEqual(Object, Object)} deep-checks possible arrays. See the documentation for what it cannot do.</li>
 *   <li>{@link #hash(double)} calculates hash values from {@code double} values which agree with above equality.</li>
 *   <li>{@link #hash(double[])} calculates hash values from {@code double[]} arrays which agree with above equality.</li>
 *   <li>{@link #hash(float)} calculates hash values from {@code float} values which agree with above equality.</li>
 *   <li>{@link #hash(float[])} calculates hash values from {@code float[]} arrays which agree with above equality.</li>
 *   <li>
 *     {@link #hashAny(Object)} calculates hash values for general objects which may be boxed floating point values,
 *     or array of these or their primitive counterparts. See the documentation for what it cannot do.
 *   </li>
 *   <li>{@link #boxed(double)} boxes a {@code double} value with zero collapse.</li>
 *   <li>{@link #boxedN(double)} does the same with also collapsing all possible NaN values.</li>
 *   <li>{@link #boxed(float)} boxes a {@code float} value with zero collapse.</li>
 *   <li>{@link #boxedN(float)} does the same with also collapsing all possible NaN values.</li>
 * </ul>
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public final class Primitives
{
  /** Constant for boxed value of (positive) float 0.0f. */
  private static final Float FLOAT_BOXED_0 = 0.0f;
  /** Constant for boxed value of (positive) double 0.0. */
  private static final Double DOUBLE_BOXED_0 = 0.0;

  /** Constant for boxed value of {@code Float.NaN}. */
  private static final Float FLOAT_BOXED_NAN = Float.NaN;

  /** Constant for boxed value of {@code Double.NaN}. */
  private static final Double DOUBLE_BOXED_NAN = Double.NaN;

  /** Hash value of all double NaNs. */
  private static final int DOUBLE_NAN_HASH = Double.hashCode(Double.NaN);
  /** Hash value of all float NaNs. */
  private static final int FLOAT_NAN_HASH = Float.hashCode(Float.NaN);

  /** Don't construct. */
  private Primitives()
  {
  }

  /**
   * Function which clones byte arrays.
   * {@code null} is accepted.
   */
  public static final Function1<byte[], byte[]> BYTE_ARRAY_CLONER =
          arg -> arg != null
                  ? arg.clone()
                  : null;
  /**
   * Function which clones short arrays.
   * {@code null} is accepted.
   */
  public static final Function1<short[], short[]> SHORT_ARRAY_CLONER =
          arg -> arg != null
                  ? arg.clone()
                  : null;
  /**
   * Function which clones int arrays.
   * {@code null} is accepted.
   */
  public static final Function1<int[], int[]> INT_ARRAY_CLONER =
          arg -> arg != null
                  ? arg.clone()
                  : null;
  /**
   * Function which clones long arrays.
   * {@code null} is accepted.
   */
  public static final Function1<long[], long[]> LONG_ARRAY_CLONER =
          arg -> arg != null
                  ? arg.clone()
                  : null;
  /**
   * Function which clones char arrays.
   * {@code null} is accepted.
   */
  public static final Function1<char[], char[]> CHAR_ARRAY_CLONER =
          arg -> arg != null
                  ? arg.clone()
                  : null;
  /**
   * Function which clones float arrays.
   * {@code null} is accepted.
   */
  public static final Function1<float[], float[]> FLOAT_ARRAY_CLONER =
          arg -> arg != null
                  ? arg.clone()
                  : null;
  /**
   * Function which clones double arrays.
   * {@code null} is accepted.
   */
  public static final Function1<double[], double[]> DOUBLE_ARRAY_CLONER =
          arg -> arg != null
                  ? arg.clone()
                  : null;

  /**
   * Get the unsigned value of a byte.
   * @param b byte value
   * @return unsigned value between {@code 0x00} and {@code 0xFF}
   */
  public static int unsigned(byte b)
  {
    return b & 0xff;
  }

  /**
   * Get the unsigned value of a short int.
   * @param s short value
   * @return unsigned value between {@code 0x0000} and {@code 0xFFFF}
   */
  public static int unsigned(short s)
  {
    return s & 0xffff;
  }

  /**
   * Get the unsigned value of an integer.
   * @param i int value
   * @return unsigned value between {@code 0x00000000} and {@code 0xFFFFFFFF}
   */
  public static long unsigned(int i)
  {
    return i & 0xffffffffL;
  }

  /**
   * Compare two bytes as if they are unsigned.
   * @param b1 first byte, interpreted as unsigned 8 bit value
   * @param b2 second byte interpreted as unsigned 8 bit value
   * @return less than {@code 0} if  {@code unsigned(b1) < unsigned(b2)},
   *         greater than {@code 0} if {@code unsigned(b1) > unsigned(b2)},
   *         or {@code 0} if {@code unsigned(b1) == unsigned(b2)}
   */
  public static int compareUnsigned(byte b1, byte b2)
  {
    return Integer.compare(unsigned(b1), unsigned(b2));
  }

  /**
   * Compare two short integers as if they are unsigned.
   * @param s1 first short integer, interpreted as unsigned 16 bit value
   * @param s2 second short integer interpreted as unsigned 16 bit value
   * @return less than {@code 0} if  {@code unsigned(s1) < unsigned(s2)},
   *         greater than {@code 0} if {@code unsigned(s1) > unsigned(s2)},
   *         or {@code 0} if {@code unsigned(s1) == unsigned(s2)}
   */
  public static int compareUnsigned(short s1, short s2)
  {
    return Integer.compare(unsigned(s1), unsigned(s2));
  }

  private static int positionOfSingleBit(int value)
  {
    switch (value) {
    case 0:
      return -1;
    case 1:
      return 0;
    }

    int pos;
    int run = value >>> 16;
    if (run == 0) {
      pos = 0;
    }
    else {
      pos = 16;
      value = run;
    }

    run = value >> 8;
    if (run != 0) {
      pos += 8;
      value = run;
    }

    run = value >> 4;
    if (run != 0) {
      pos += 4;
      value = run;
    }

    run = value >> 2;
    if (run != 0) {
      pos += 2;
      value = run;
    }

    run = value >> 1;
    if (run != 0) {
      pos += 1;
    }

    return pos;
  }

  private static int positionOfSingleBit(long value)
  {
    if (value == 0L) {
      return -1;
    }
    if (value == 1L) {
      return 0;
    }

    final long run = value >>> 32;
    if (run != 0L) {
      return positionOfSingleBit((int)run) + 32;
    }
    return positionOfSingleBit((int)value);
  }

  /**
   * Get the position of the highest one bit.
   * @param value value to check
   * @return {@code -1} if not bit is set, or a value between {@code 0} (only the 1 bit is set)
   *         and {@code 31} (the highest bit is set)
   */
  public static int positionOfHighestOneBit(int value)
  {
    final int highestBit = Integer.highestOneBit(value);
    return positionOfSingleBit(highestBit);
  }

  /**
   * Get the position of the lowest one bit.
   * @param value value to check
   * @return {@code -1} if not bit is set, or a value between {@code 0} (the lowest bit is set)
   *         and {@code 31} (only the highest bit is set)
   */
  public static int positionOfLowestOneBit(int value)
  {
    final int lowestBit = Integer.lowestOneBit(value);
    return positionOfSingleBit(lowestBit);
  }

  /**
   * Get the position of the highest one bit.
   * @param value value to check
   * @return {@code -1} if not bit is set, or a value between {@code 0} (only the lowest bit is set)
   *         and {@code 63} (the highest bit is set)
   */
  public static int positionOfHighestOneBit(long value)
  {
    final long highestBit = Long.highestOneBit(value);
    return positionOfSingleBit(highestBit);
  }

  /**
   * Get the position of the lowest one bit.
   * @param value value to check
   * @return {@code -1} if not bit is set, or a value between {@code 0} (the lowest bit is set)
   *         and {@code 63} (only the highest bit is set)
   */
  public static int positionOfLowestOneBit(long value)
  {
    final long lowestBit = Long.lowestOneBit(value);
    return positionOfSingleBit(lowestBit);
  }

  /**
   * Are two primitive {@code double} values equal?
   * <p>
   * Thanks to the underlying standard ANSI / IEEE Std 754-1985 defining floating point arithmetic
   * {@code Double.NaN} is not equal to anything, not even to itself. This is often not what you want.
   * {@link java.lang.Double#compare(double, double)} falls back to the binary representation in
   * some cases, which again is not always a good idea as it differs between {@code -0.0} and
   * {@code +0.0} which are considered not equal, although they represent the same mathematical number.
   * <p>
   * This method considers two values equal if the both represent the same number.
   * So two {@code NaN} values are equal, and both {@code 0.0} values are equal in any combination, too.
   * This seems the most natural way to compare doubles.
   * @param v1 first value to compare
   * @param v2 second value to compare
   * @return {@code true}: if both values are considered equal according to the above definition<br>
   *         {@code false}: otherwise
   * @see #hash(double)
   */
  public static boolean areEqual(double v1, double v2)
  {
    if (v1 == v2) { // this handles -0.0 == 0.0 correctly
      return true;
    }
    return Double.isNaN(v1)  &&  Double.isNaN(v2);  // make any 2 NaN values equals
  }

  /**
   * Are two primitive {@code float} values equal?
   * <p>
   * Thanks to the underlying standard ANSI / IEEE Std 754-1985 defining floating point arithmetic
   * {@code Float.NaN} is not equal to anything, not even to itself. This is often not what you want.
   * {@link java.lang.Float#compare(float, float)} falls back to the binary representation in
   * some cases, which again is not always a good idea as it differs between {@code -0.0f} and
   * {@code +0.0f} which are considered not equal, although they represent the same mathematical number.
   * <p>
   * This method considers two values equal if the both represent the same number.
   * So two {@code NaN} values are equal, and both {@code 0.0f} values are equal in any combination, too.
   * This seems the most natural way to compare floats.
   * @param v1 first value to compare
   * @param v2 second value to compare
   * @return {@code true}: if both values are considered equal according to the above definition<br>
   *         {@code false}: otherwise
   */
  public static boolean areEqual(float v1, float v2)
  {
    if (v1 == v2) { // this handles -0.0f == 0.0f correctly
      return true;
    }
    return Float.isNaN(v1)  &&  Float.isNaN(v2); // make any 2 NaN values equal
  }

  /**
   * Are two primitive {@code double} values nearly equal?
   * @param v1 first value to compare
   * @param v2 second value to compare
   * @param eps allowed deviation for both values to be still considered equal, a small non-negative number
   * @return {@code true}: if both values are considered equal within the given allowance<br>
   *         {@code false}: if they differ too much
   */
  public static boolean areEqual(double v1, double v2, double eps)
  {
    assert eps >= 0.0;
    return Math.abs(v1 - v2) <= eps;
  }

  /**
   * Are two primitive {@code float} values nearly equal?
   * @param v1 first value to compare
   * @param v2 second value to compare
   * @param eps allowed deviation for both values to be still considered equal, a small non-negative number
   * @return {@code true}: if both values are considered equal within the given allowance<br>
   *         {@code false}: if they differ too much
   */
  public static boolean areEqual(float v1, float v2, float eps)
  {
    assert eps >= 0.0f;
    return Math.abs(v1 - v2) <= eps;
  }

  /**
   * Are two primitive {@code double} arrays equal?
   * <p>
   * This method uses {@link #areEqual(double, double)} for comparing elements.
   * @param v1 first array to compare
   * @param v2 second array to compare
   * @return {@code true}: if both values are considered equal according to the above definition<br>
   *         {@code false}: otherwise
   * @see #hash(double[])
   */
  public static boolean areEqual(@NotNull double[] v1, @NotNull double[] v2)
  {
    if (v1 == v2) {
      return true;
    }
    if (v1.length != v2.length) {
      return false;
    }
    for (int i = v1.length - 1;  i >= 0;  --i) {
      if (!areEqual(v1[i], v2[i])) {
        return false;
      }
    }
    return true;
  }

  /**
   * Are two primitive {@code float} arrays equal?
   * <p>
   * This method uses {@link #areEqual(float, float)} for comparing elements.
   * @param v1 first array to compare
   * @param v2 second array to compare
   * @return {@code true}: if both values are considered equal according to the above definition<br>
   *         {@code false}: otherwise
   * @see #hash(float[])
   */
  public static boolean areEqual(@NotNull float[] v1, @NotNull float[] v2)
  {
    if (v1 == v2) {
      return true;
    }
    if (v1.length != v2.length) {
      return false;
    }
    for (int i = v1.length - 1;  i >= 0;  --i) {
      if (!areEqual(v1[i], v2[i])) {
        return false;
      }
    }
    return true;
  }

  /**
   * Hashcode calculation for primitive double values giving more natural results.
   * <p>
   * This method agrees with the equality definition provided by
   * {@linkplain #areEqual(double, double)}. Standard Java's
   * {@link Double#hashCode(double)} will return different values
   * for +0.0 and negative -0.0, and for each NaN value.
   *
   * @param v double value to hash
   * @return hash value
   * @see #areEqual(double, double)
   */
  public static int hash(double v)
  {
    if (v == 0.0) {  // this takes care of both -0.0 and +0.0
      return 0;
    }
    if (Double.isNaN(v)) { // this unites all possible NaN values
      return DOUBLE_NAN_HASH;
    }
    return Double.hashCode(v);
  }

  /**
   * Hashcode calculation for primitive float values giving more natural results.
   * <p>
   * This method agrees with the equality definition provided by
   * {@linkplain #areEqual(float, float)}. Standard Java's
   * {@link Double#hashCode(double)} will return different values
   * for +0.0f and negative -0.0f, and for each NaN value.
   *
   * @param v float value to hash
   * @return hash value
   * @see #areEqual(float, float)
   */
  public static int hash(float v)
  {
    if (v == 0.0f) {  // this takes care of both -0.0f and +0.0f
      return 0;
    }
    if (Float.isNaN(v)) { // this unites all possible NaN values
      return FLOAT_NAN_HASH;
    }
    return Double.hashCode(v);
  }

  /**
   * Hashcode calculation for a primitive double array giving more natural results.
   * <p>
   * See {@linkplain #hash(double)} for information.
   *
   * @param arr double array to hash
   * @return hash value
   * @see #areEqual(double, double)
   */
  public static int hash(@Nullable double[] arr)
  {
    if (arr == null) {
      return 0;
    }
    int result = 1;
    for (double v : arr) {
      result = 31 * result + hash(v);
    }
    return result;
  }

  /**
   * Hashcode calculation for a primitive float array giving more natural results.
   * <p>
   * See {@linkplain #hash(float)} for information.
   *
   * @param arr double array to hash
   * @return hash value
   * @see #areEqual(double, double)
   */
  public static int hash(@Nullable float[] arr)
  {
    if (arr == null) {
      return 0;
    }
    int result = 1;
    for (float v : arr) {
      result = 31 * result + hash(v);
    }
    return result;
  }

  /**
   * Hash any object while handling simple floating point values and arrays of primitive
   * and boxed floating point values in a more natural way.
   * <p>
   * It will always hash arrays deeply.
   * <p> 
   * ATTENTION: this method only handles {@code double} and {@code float} values and
   * their boxed counterparts as well as arrays of these in any depth to make them work
   * more natural, but cannot provide this for floating point items inside arbitrary objects.
   * 
   * @param obj object to be hashed
   * @return hash value
   */
  public static int hashAny(@Nullable Object obj)
  {
    if (obj == null) {
      return 0;
    }
    return DEEP_NATURAL_FP.substitute(obj).hashCode();
  }

  /**
   * Are the too arbitrary objects equal?
   * This method handles primitive and boxed floating point values as well as arrays
   * of them in a more natural way compared to standard Java.
   * <p>
   * It will always compare arrays deeply.
   * <p> 
   * ATTENTION: this method only handles {@code double} and {@code float} values and
   * their boxed counterparts as well as arrays of these in any depth to make them work
   * more natural, but cannot provide this for floating point items inside arbitrary objects.
   *
   * @param obj1  first object
   * @param obj2  second object
   * @return {@code true} if both objects are considered the same<br>
   *         {@code false} if not
   */
  public static boolean areEqual(@Nullable Object obj1,
                                 @Nullable Object obj2)
  {
    if (obj1 == obj2) {
      return true;
    }
    if (obj1 != null  &&  obj2 != null) {
      return DEEP_NATURAL_FP.substitute(obj1).equals(DEEP_NATURAL_FP.substitute(obj2));
    }
    return false;
  }

  /** Subjective wrapper for {@code double} and {@code Double} which makes them work more natural. */
  private static final Subjective<Double> DOUBLE_SUBJECTIVE =
          new Subjective<Double>()
          {
            @NotNull
            @Override
            public Class<Double> type()
            {
              return Double.class;
            }

            @NotNull
            @Override
            public Subject<Double> subject(@NotNull Double object)
            {
              return new Subject.Base<Double>(object, Double.class)
              {
                @NotNull
                @Override
                public Double substituted()
                {
                  return object;
                }

                @Override
                public int hashCode()
                {
                  return hash(object);
                }

                @Override
                protected boolean isEqualTo(@NotNull Double other)
                {
                  return areEqual(object.doubleValue(), other.doubleValue());
                }
              };
            }
          };
  /** Subjective wrapper for {@code float} and {@code Float} which makes them work more natural. */
  private static final Subjective<Float> FLOAT_SUBJECTIVE =
          new Subjective<Float>()
          {
            @NotNull
            @Override
            public Class<Float> type()
            {
              return Float.class;
            }

            @NotNull
            @Override
            public Subject<Float> subject(@NotNull Float object)
            {
              return new Subject.Base<Float>(object, Float.class)
              {
                @NotNull
                @Override
                public Float substituted()
                {
                  return object;
                }

                @Override
                public int hashCode()
                {
                  return hash(object);
                }

                @Override
                protected boolean isEqualTo(@NotNull Float other)
                {
                  return areEqual(object.floatValue(), other.floatValue());
                }
              };
            }
          };
  /** Subjective wrapper for {@code double[]} arrays which makes them work more natural. */
  private static final Subjective<double[]> DOUBLE_ARRAY_SUBJECTIVE =
          new Subjective<double[]>()
          {
            @NotNull
            @Override
            public Class<double[]> type()
            {
              return double[].class;
            }

            @NotNull
            @Override
            public Subject<double[]> subject(@NotNull double[] object)
            {
              return new Subject.Base<double[]>(object, double[].class)
              {
                @NotNull
                @Override
                public double[] substituted()
                {
                  return object;
                }

                @Override
                public int hashCode()
                {
                  return hash(object);
                }

                @Override
                protected boolean isEqualTo(@NotNull double[] other)
                {
                  return areEqual(object, other);
                }
              };
            }
          };
  /** Subjective wrapper for {@code float[]} arrays which makes them work more natural. */
  private static final Subjective<float[]> FLOAT_ARRAY_SUBJECTIVE =
          new Subjective<float[]>()
          {
            @NotNull
            @Override
            public Class<float[]> type()
            {
              return float[].class;
            }

            @NotNull
            @Override
            public Subject<float[]> subject(@NotNull float[] object)
            {
              return new Subject.Base<float[]>(object, float[].class)
              {
                @NotNull
                @Override
                public float[] substituted()
                {
                  return object;
                }

                @Override
                public int hashCode()
                {
                  return hash(object);
                }

                @Override
                protected boolean isEqualTo(@NotNull float[] other)
                {
                  return areEqual(object, other);
                }
              };
            }
          };

  /**
   * Subjective wrapper factory to repair floating point values in a way that they behave more natural.
   * More natural here means that they no longer differ between the two possible zero values, and neither
   * between all possible NaN values.
   * <p>
   * This will take care of standard methods {@link Object#equals(Object)} and {@link Object#hashCode()}
   * of {@code Double}, {@code double[]}, {@code Float}, and {@code float[]}
   * when handled directly or inside arrays. It will also do deep evaluation for both methods
   * when called with arbitrary arrays.
   */
  public static final Subjectivity DEEP_NATURAL_FP =
          new Subjectivity(true, DOUBLE_SUBJECTIVE, FLOAT_SUBJECTIVE, DOUBLE_ARRAY_SUBJECTIVE, FLOAT_ARRAY_SUBJECTIVE);

  /**
   * Get a boxed float value which takes care of handling positive and negative zero in a
   * more natural way.
   * <p>
   * This method does not collapse all possible NaN values. This should be a problem only in very
   * rare cases where the values are read from outside Java, or are deliberately created,
   * so the required general overhead seems too much. See {@linkplain #boxedN(float)} if you also need
   * NaN collapse.
   * @param value float value to be boxed
   * @return boxed value
   */
  @NotNull
  public static Float boxed(float value)
  {
    return value == 0.0f
            ? FLOAT_BOXED_0
            : Float.valueOf(value);
  }

  /**
   * Get a boxed float value which takes care of handling positive and negative zero in a
   * more natural way.
   * <p>
   * This method also collapses all possible 16,777,214 NaN values into one.
   * @param value float value to be boxed
   * @return boxed value
   */
  @NotNull
  public static Float boxedN(float value)
  {
    return value == 0.0f
            ? FLOAT_BOXED_0
            : (Float.isNaN(value)
                       ? FLOAT_BOXED_NAN
                       : Float.valueOf(value));
  }

  /**
   * Get a boxed double value which takes care of handling positive and negative zero in a
   * more natural way.
   * <p>
   * This method does not collapse all possible NaN values. This should be a problem only in very
   * rare cases where the values are read from outside Java, or are deliberately created,
   * so the required general overhead seems too much. See {@linkplain #boxedN(double)} if you also need
   * NaN collapse.
   * @param value double value to be boxed
   * @return boxed value
   */
  @NotNull
  public static Double boxed(double value)
  {
    return value == 0.0
            ? DOUBLE_BOXED_0
            : Double.valueOf(value);
  }

  /**
   * Get a boxed double value which takes care of handling positive and negative zero in a
   * more natural way.
   * <p>
   * This method also collapses all possible 9,007,199,254,740,990 NaN values into one.
   * @param value double value to be boxed
   * @return boxed value
   */
  @NotNull
  public static Double boxedN(double value)
  {
    return value == 0.0
            ? DOUBLE_BOXED_0
            : (Double.isNaN(value)
                       ? DOUBLE_BOXED_NAN
                       : Double.valueOf(value));
  }

  /**
   * Compare two double values without the quirks of {@link Double#compare(double, double)}.
   * This will sort all NaN values for which {@link Double#isNaN} as if they are the same value.
   * It will handle {@code -0.0} and {@code +0.0} as being the same value. 
   * That makes this method work the same as {@linkplain #areEqual(double, double)} and
   * {@linkplain #hash(double)}. If you use one of them it is best to combine them all.
   * @param v1 first value
   * @param v2 second value
   * @return less than zero if {@code v1} is less than{@code v2},
   *         more than zero if {@code v1} is greater than {@code v2},
   *         and {@code 0} if they considered are the same
   */
  public static int compare(double v1, double v2)
  {
    if (v1 == v2) { // this already takes care of +/- zero
      return 0;
    }
    if (v1 < v2) {
      return -1;
    }
    if (v1 > v2) {
      return 1;
    }
    // here well have some combination of NaN
    return Double.isNaN(v1)
            ? (Double.isNaN(v2)
                       ? 0
                       : 1)
            : -1;
  }

  /**
   * Compare two float values without the quirks of {@link Float#compare(float, float)}.
   * This will sort all NaN values for which {@link Float#isNaN} as if they are the same value.
   * It will handle {@code -0.0} and {@code +0.0} as being the same value. 
   * That makes this method work the same as {@linkplain #areEqual(float, float)} and
   * {@linkplain #hash(float)}. If you use one of them it is best to combine them all.
   * @param v1 first value
   * @param v2 second value
   * @return less than zero if {@code v1} is less than{@code v2},
   *         more than zero if {@code v1} is greater than {@code v2},
   *         and {@code 0} if they considered are the same
   */
  public static int compare(float v1, float v2)
  {
    if (v1 == v2) { // this already takes care of +/- zero
      return 0;
    }
    if (v1 < v2) {
      return -1;
    }
    if (v1 > v2) {
      return 1;
    }
    // here well have some combination of NaN
    return Float.isNaN(v1)
            ? (Float.isNaN(v2)
                       ? 0
                       : 1)
            : -1;
  }

  /**
   * Compare two double values without the quirks of {@link Double#compare(double, double)}.
   * This will sort all NaN values for which {@link Double#isNaN} as if they are the same value.
   * It will handle {@code -0.0} and {@code +0.0} as being the same value. 
   * That makes this method work the same as {@linkplain #areEqual(double, double)} and
   * {@linkplain #hash(double)}. If you use one of them it is best to combine them all.
   * @param v1 first value
   * @param v2 second value
   * @return the ordering of {@code v1} and {@code v2} when appearing in that order
   */
  @NotNull
  public static Order order(double v1, double v2)
  {
    if (v1 == v2) { // this already takes care of +/- zero
      return Order.Same;
    }
    if (v1 < v2) {
      return Order.Ascending;
    }
    if (v1 > v2) {
      return Order.Descending;
    }
    // here we have some combination of NaN
    return Double.isNaN(v1)
            ? (Double.isNaN(v2)
                       ? Order.Same
                       : Order.Descending)
            : Order.Ascending;
  }

  /**
   * Compare two float values without the quirks of {@link Float#compare(float, float)}.
   * This will sort all NaN values for which {@link Float#isNaN} as if they are the same value.
   * It will handle {@code -0.0} and {@code +0.0} as being the same value. 
   * That makes this method work the same as {@linkplain #areEqual(float, float)} and
   * {@linkplain #hash(float)}. If you use one of them it is best to combine them all.
   * @param v1 first value
   * @param v2 second value
   * @return the ordering of {@code v1} and {@code v2} when appearing in that order
   */
  @NotNull
  public static Order order(float v1, float v2)
  {
    if (v1 == v2) { // this already takes care of +/- zero
      return Order.Same;
    }
    if (v1 < v2) {
      return Order.Ascending;
    }
    if (v1 > v2) {
      return Order.Descending;
    }
    // here we have some combination of NaN
    return Float.isNaN(v1)
            ? (Float.isNaN(v2)
                       ? Order.Same
                       : Order.Descending)
            : Order.Ascending;
  }

}
