// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.ShortConsumer;

import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.PrimitiveIterator;
import java.util.function.Consumer;

/**
 * An iterator specialized for {@code short} values.
 * <p>
 * This is missing in the standard library: an iterator
 * for raw {@code short} values which does everything what
 * {@link java.util.PrimitiveIterator.OfInt} does for
 * [@code int}s.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since September 27, 2020
 */
public interface PrimitiveShortIterator
        extends PrimitiveIterator<Short, ShortConsumer>
{
  /** Empty iterator of primitive short values. */
  PrimitiveShortIterator EMPTY = new PrimitiveShortIterator()
  {
    @Override
    public short nextShort()
    {
      throw new NoSuchElementException("Empty short iterator has no elements!");
    }

    @Override
    public boolean hasNext()
    {
      return false;
    }

    @Override
    public void forEachRemaining(@NotNull ShortConsumer action)
    {
    }

    @Override
    public void forEachRemaining(@NotNull Consumer<? super Short> action)
    {
    }
  };

  /**
   * Returns the next {@code short} element in the iteration.
   *
   * @return the next {@code short} element in the iteration
   * @throws NoSuchElementException if the iteration has no more elements
   */
  short nextShort();

  /**
   * Performs the given action for each remaining element until all elements
   * have been processed or the action throws an exception.  Actions are
   * performed in the order of iteration, if that order is specified.
   * Exceptions thrown by the action are relayed to the caller.
   *
   * <p>The default implementation behaves as if:
   * <pre>{@code
   *     while (hasNext())
   *         action.accept(nextShort());
   * }</pre>
   *
   * @param action The action to be performed for each element
   * @throws NullPointerException if the specified action is null
   */
  default void forEachRemaining(@NotNull ShortConsumer action) {
    Objects.requireNonNull(action);
    while (hasNext()) {
      action.accept(nextShort());
    }
  }

  /**
   * {@inheritDoc}
   * The default implementation boxes the result of calling
   * {@link #nextShort()}, and returns that boxed result.
   */
  @Override
  default Short next() {
    return nextShort();
  }

  /**
   * {@inheritDoc}
   * If the action is an instance of {@code ShortConsumer} then it is cast
   * to {@code ShortConsumer} and passed to {@link #forEachRemaining};
   * otherwise the action is adapted to an instance of
   * {@code ShortConsumer}, by boxing the argument of {@code ShortConsumer},
   * and then passed to {@link #forEachRemaining}.
   */
  @Override
  default void forEachRemaining(@NotNull Consumer<? super Short> action) {
    if (action instanceof ShortConsumer) {
      forEachRemaining((ShortConsumer) action);
    }
    else {
      // The method reference action::accept is never null
      Objects.requireNonNull(action);
      forEachRemaining((ShortConsumer) action::accept);
    }
  }

  /**
   * View this short iterator as an int iterator.
   * Please note that you should use either this iterator
   * of the returned one, but don't mix using both.
   * @return int iterator view of this iterator
   */
  @NotNull
  default PrimitiveIterator.OfInt asIntIterator()
  {
    return new OfInt()
    {
      @Override
      public int nextInt()
      {
        return PrimitiveShortIterator.this.nextShort();
      }

      @Override
      public boolean hasNext()
      {
        return PrimitiveShortIterator.this.hasNext();
      }
    };
  }

  /**
   * View this short iterator as an int iterator using the values of
   * this iterator as unsigned shorts.
   * Please note that you should use either this iterator
   * of the returned one, but don't mix using both.
   * @return int iterator view of this iterator
   */
  @NotNull
  default PrimitiveIterator.OfInt asUnsignedIntIterator()
  {
    return new OfInt()
    {
      @Override
      public int nextInt()
      {
        return PrimitiveShortIterator.this.nextShort() & 0xffff;
      }

      @Override
      public boolean hasNext()
      {
        return PrimitiveShortIterator.this.hasNext();
      }
    };
  }

  /**
   * View this short iterator as a long iterator.
   * Please note that you should use either this iterator
   * of the returned one, but don't mix using both.
   * @return long iterator view of this iterator
   */
  @NotNull
  default PrimitiveIterator.OfLong asLongIterator()
  {
    return new OfLong()
    {
      @Override
      public long nextLong()
      {
        return PrimitiveShortIterator.this.nextShort();
      }

      @Override
      public boolean hasNext()
      {
        return PrimitiveShortIterator.this.hasNext();
      }
    };
  }

  /**
   * View this byte iterator as a long iterator using the values of
   * this iterator as unsigned bytes.
   * Please note that you should use either this iterator
   * of the returned one, but don't mix using both.
   * @return long iterator view of this iterator
   */
  @NotNull
  default PrimitiveIterator.OfLong asUnsignedLongIterator()
  {
    return new OfLong()
    {
      @Override
      public long nextLong()
      {
        return PrimitiveShortIterator.this.nextShort() & 0xffffL;
      }

      @Override
      public boolean hasNext()
      {
        return PrimitiveShortIterator.this.hasNext();
      }
    };
  }
}
