// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.FragileLongConsumer;

import java.util.*;
import java.util.function.Consumer;
import java.util.function.LongConsumer;
import java.util.stream.LongStream;
import java.util.stream.StreamSupport;

/**
 * Iterable which iterates over a primitive long.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 13, 2021
 * @see LongIndexable for creating one on the fly
 */
public interface PrimitiveLongIterable
        extends Iterable<Long>
{
  /** Empty iterator of primitive long values. */
  PrimitiveIterator.OfLong EMPTY_LONG_ITERATOR = new PrimitiveIterator.OfLong()
  {
    @Override
    public long nextLong()
    {
      throw new NoSuchElementException("No elements in empty iterator!");
    }

    @Override
    public boolean hasNext()
    {
      return false;
    }

    @Override
    public void forEachRemaining(LongConsumer action)
    {
    }

    @Override
    public void forEachRemaining(Consumer<? super Long> action)
    {
    }
  };
  /** Empty iterable of primitive long values. */
  PrimitiveLongIterable EMPTY = () -> EMPTY_LONG_ITERATOR;

  /**
   * Get a primitive long iterator.
   * @return long iterator
   */
  @NotNull
  PrimitiveIterator.OfLong longIterator();

  @Override
  @NotNull
  default Iterator<Long> iterator()
  {
    return longIterator();
  }

  /**
   * Invoke the given action with each long value in this iterable.
   * @param action action to be call for each long value in sequemce.
   */
  default void forEachLong(@NotNull LongConsumer action)
  {
    for (PrimitiveIterator.OfLong it = longIterator();  it.hasNext(); ) {
      action.accept(it.nextLong());
    }
  }

  /**
   * Invoke the given fragile action with each value in this iterable.
   * @param action action to be called for each value in sequence
   * @param <E> checked exception thrown by action
   * @throws E if {@code action} throws it
   */
  default <E extends Exception> void forEachLongFragile(@NotNull FragileLongConsumer<E> action)
          throws E
  {
    for (PrimitiveIterator.OfLong it = longIterator();  it.hasNext(); ) {
      action.accept(it.nextLong());
    }
  }

  /**
   * Does this long iterable contain the given value?
   * This is an operation which takes time linear to the size of this iterable
   * and should only be used rarely.
   * @param value value to check for
   * @return {@code true} if the value is contained in this iterable, {@code false} if not
   */
  default boolean containsLong(long value)
  {
    for (PrimitiveIterator.OfLong iterator = longIterator();
         iterator.hasNext();  ) {
      if (iterator.nextLong() == value) {
        return true;
      }
    }
    return false;
  }

  /**
   * Calculate the sum of the elements of this iterable as a long value.
   * Please note that this might silently overflow.
   * @return sum of all elements of this iterable
   */
  default long sum()
  {
    long s = 0;
    for (PrimitiveIterator.OfLong it = longIterator();  it.hasNext();  ) {
      s += it.nextLong();
    }
    return s;
  }

  /**
   * Calculate the sum of the elements of this iterable as a long value.
   * This will throw an exception if summing up overflows the range of
   * {@code long}.
   * @return sum of all elements of this iterable
   */
  default long sumX()
  {
    long s = 0;
    for (PrimitiveIterator.OfLong it = longIterator();  it.hasNext();  ) {
      s = Math.addExact(s, it.nextLong());
    }
    return s;
  }

  /**
   * Calculate the average of all elements in this iterable.
   * @return average, possibly {@link Double#NaN} if this iterable is empts
   */
  @NotNull
  default OptionalDouble average()
  {
    int count = 0;
    double average = 0.0;
    for (PrimitiveIterator.OfLong it = longIterator();  it.hasNext();  ) {
      average = (average * count + it.nextLong()) / ++count;
    }
    return count == 0 ? OptionalDouble.empty() : OptionalDouble.of(average);
  }

  @Override
  default Spliterator.OfLong spliterator()
  {
    return Spliterators.spliteratorUnknownSize(longIterator(), Spliterator.ORDERED);
  }

  /**
   * Get a stream of primitive long values from this iterable.
   * @return {@code long} stream
   */
  @NotNull
  default LongStream stream()
  {
    return StreamSupport.longStream(spliterator(), false);
  }
}
