// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.FragileIntConsumer;

import java.util.*;
import java.util.function.Consumer;
import java.util.function.IntConsumer;
import java.util.stream.IntStream;
import java.util.stream.StreamSupport;

/**
 * Iterable which iterates over a primitive int.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 13, 2021
 * @see IntIndexable for creating one on the fly
 */
public interface PrimitiveIntIterable
        extends Iterable<Integer>
{
  /** Empty iterator of primitive int values. */
  PrimitiveIterator.OfInt EMPTY_INT_ITERATOR = new PrimitiveIterator.OfInt()
  {
    @Override
    public int nextInt()
    {
      throw new NoSuchElementException("No elements in empty iterator!");
    }

    @Override
    public boolean hasNext()
    {
      return false;
    }

    @Override
    public void forEachRemaining(IntConsumer action)
    {
    }

    @Override
    public void forEachRemaining(Consumer<? super Integer> action)
    {
    }
  };
  /** Empty iterable of primitive int values. */
  PrimitiveIntIterable EMPTY = () -> EMPTY_INT_ITERATOR;

  /**
   * Get a primitive int iterator.
   * @return int iterator
   */
  @NotNull
  PrimitiveIterator.OfInt intIterator();

  @Override
  @NotNull
  default Iterator<Integer> iterator()
  {
    return intIterator();
  }

  /**
   * Invoke the given action with each int value in this iterable.
   * @param action action to be called for each int value in sequence.
   */
  default void forEachInt(@NotNull IntConsumer action)
  {
    for (PrimitiveIterator.OfInt it = intIterator();  it.hasNext(); ) {
      action.accept(it.nextInt());
    }
  }

  /**
   * Invoke the given fragile action with each value in this iterable.
   * @param action action to be called for each value in sequence
   * @param <E> checked exception thrown by action
   * @throws E if {@code action} throws it
   */
  default <E extends Exception> void forEachIntFragile(@NotNull FragileIntConsumer<E> action)
          throws E
  {
    for (PrimitiveIterator.OfInt it = intIterator();  it.hasNext(); ) {
      action.accept(it.nextInt());
    }
  }

  /**
   * Does this integer iterable contain the given value?
   * This is an operation which takes time linear to the size of this iterable
   * and should only be used rarely.
   * @param value value to check for
   * @return {@code true} if the value is contained in this iterable, {@code false} if not
   */
  default boolean containsInt(int value)
  {
    for (PrimitiveIterator.OfInt iterator = intIterator();
         iterator.hasNext();  ) {
      if (iterator.nextInt() == value) {
        return true;
      }
    }
    return false;
  }

  /**
   * Calculate the sum of the elements of this iterable as an integer value.
   * <p>
   * Please note that this might silently overflow.
   * See {@link #sumX()} if you want to avoid this
   * with the price of running much slower, or {@link #longSum()}
   * if you want to make an overflow unlikely.
   * @return sum of all elements of this iterable
   */
  default int sum()
  {
    int s = 0;
    for (PrimitiveIterator.OfInt it = intIterator();  it.hasNext();  ) {
      s += it.nextInt();
    }
    return s;
  }

  /**
   * Calculate the sum of the elements of this iterable as an integer value.
   * <p>
   * This will throw an exception if summing up overflows.
   * @return sum of all elements of this iterable
   * @throws ArithmeticException if summing up overflows the range of {@code int}
   */
  default int sumX()
  {
    int s = 0;
    for (PrimitiveIterator.OfInt it = intIterator();  it.hasNext();  ) {
      s = Math.addExact(s, it.nextInt());
    }
    return s;
  }

  /**
   * Calculate the sum of the elements of this iterable as a long value.
   * Please note that this might silently overflow when summing on very large
   * (i.e. more than 2^32 iterations) iterables.
   * @return sum of all elements of this iterable
   */
  default long longSum()
  {
    long s = 0;
    for (PrimitiveIterator.OfInt it = intIterator();  it.hasNext();  ) {
      s += it.nextInt();
    }
    return s;
  }

  /**
   * Calculate the sum of the elements of this iterable as a long value.
   * This will throw an exception if summing up overflows the range of
   * {@code long}.
   * @return sum of all elements of this iterable
   * @throws ArithmeticException if summing up overflows the range of {@code long}
   */
  default long longSumX()
  {
    long s = 0;
    for (PrimitiveIterator.OfInt it = intIterator();  it.hasNext();  ) {
      s = Math.addExact(s, it.nextInt());
    }
    return s;
  }

  /**
   * Calculate the average of all elements in this iterable.
   * @return average, possibly {@link Double#NaN} if this iterable is empts
   */
  @NotNull
  default OptionalDouble average()
  {
    int count = 0;
    double average = 0.0;
    for (PrimitiveIterator.OfInt it = intIterator();  it.hasNext();  ) {
      average = (average * count + it.nextInt()) / ++count;
    }
    return count == 0 ? OptionalDouble.empty() : OptionalDouble.of(average);
  }

  @Override
  default Spliterator.OfInt spliterator()
  {
    return Spliterators.spliteratorUnknownSize(intIterator(), Spliterator.ORDERED);
  }

  /**
   * Get a stream of primitive integer values from this iterable.
   * @return {@code int} stream
   */
  @NotNull
  default IntStream stream()
  {
    return StreamSupport.intStream(spliterator(), false);
  }
}
