// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.FloatConsumer;

import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.PrimitiveIterator;
import java.util.function.Consumer;

/**
 * An iterator specialized for {@code float} values.
 * <p>
 * This is missing in the standard library: an iterator
 * for raw {@code float} values which does everything what
 * {@link OfInt} does for
 * [@code int}s.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since September 27, 2020
 */
public interface PrimitiveFloatIterator
        extends PrimitiveIterator<Float, FloatConsumer>
{
  /** Empty iterator of primitive float values. */
  PrimitiveFloatIterator EMPTY = new PrimitiveFloatIterator()
  {
    @Override
    public float nextFloat()
    {
      throw new NoSuchElementException("Empty float iterator has no elements!");
    }

    @Override
    public boolean hasNext()
    {
      return false;
    }

    @Override
    public void forEachRemaining(@NotNull FloatConsumer action)
    {
    }

    @Override
    public void forEachRemaining(@NotNull Consumer<? super Float> action)
    {
    }
  };

  /**
   * Returns the next {@code float} element in the iteration.
   *
   * @return the next {@code float} element in the iteration
   * @throws NoSuchElementException if the iteration has no more elements
   */
  float nextFloat();

  /**
   * Performs the given action for each remaining element until all elements
   * have been processed or the action throws an exception.  Actions are
   * performed in the order of iteration, if that order is specified.
   * Exceptions thrown by the action are relayed to the caller.
   *
   * <p>The default implementation behaves as if:
   * <pre>{@code
   *     while (hasNext())
   *         action.accept(nextFloat());
   * }</pre>
   *
   * @param action The action to be performed for each element
   * @throws NullPointerException if the specified action is null
   */
  default void forEachRemaining(@NotNull FloatConsumer action) {
    Objects.requireNonNull(action);
    while (hasNext()) {
      action.accept(nextFloat());
    }
  }

  /**
   * {@inheritDoc}
   * The default implementation boxes the result of calling
   * {@link #nextFloat()}, and returns that boxed result.
   */
  @Override
  default Float next() {
    return nextFloat();
  }

  /**
   * {@inheritDoc}
   * If the action is an instance of {@code FloatConsumer} then it is cast
   * to {@code FloatConsumer} and passed to {@link #forEachRemaining};
   * otherwise the action is adapted to an instance of
   * {@code FloatConsumer}, by boxing the argument of {@code FloatConsumer},
   * and then passed to {@link #forEachRemaining}.
   */
  @Override
  default void forEachRemaining(@NotNull Consumer<? super Float> action) {
    if (action instanceof FloatConsumer) {
      forEachRemaining((FloatConsumer) action);
    }
    else {
      // The method reference action::accept is never null
      Objects.requireNonNull(action);
      forEachRemaining((FloatConsumer) action::accept);
    }
  }

  /**
   * View this float iterator as a double iterator.
   * Please note that you should use either this iterator
   * of the returned one, but don't mix using both.
   * @return a view of this iterator as a double iterator
   */
  @NotNull
  default PrimitiveIterator.OfDouble asDoubleIterator()
  {
    return new OfDouble()
    {
      @Override
      public double nextDouble()
      {
        return PrimitiveFloatIterator.this.nextFloat();
      }

      @Override
      public boolean hasNext()
      {
        return PrimitiveFloatIterator.this.hasNext();
      }
    };
  }
}
