// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.FloatConsumer;
import de.caff.generics.function.FragileFloatConsumer;

import java.util.Iterator;

/**
 * Iterable which iterates over a primitive float.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 13, 2021
 */
public interface PrimitiveFloatIterable
        extends Iterable<Float>
{
  /**
   * Empty iterator of primitive float values.
   * @see PrimitiveFloatIterator#EMPTY
   */
  PrimitiveFloatIterator EMPTY_FLOAT_ITERATOR = PrimitiveFloatIterator.EMPTY;
  /** Empty iterable of primitive float values. */
  PrimitiveFloatIterable EMPTY = () -> EMPTY_FLOAT_ITERATOR;

  /**
   * Get a primitive float iterator.
   * @return float iterator
   */
  @NotNull
  PrimitiveFloatIterator floatIterator();

  @Override
  @NotNull
  default Iterator<Float> iterator()
  {
    return floatIterator();
  }

  /**
   * Invoke the given action with each float value in this iterable.
   * @param action action to be call for each float value in sequemce.
   */
  default void forEachFloat(@NotNull FloatConsumer action)
  {
    for (PrimitiveFloatIterator it = floatIterator();  it.hasNext(); ) {
      action.accept(it.nextFloat());
    }
  }

  /**
   * Invoke the given fragile action with each value in this iterable.
   * @param action action to be called for each value in sequence
   * @param <E> checked exception thrown by action
   * @throws E if {@code action} throws it
   */
  default <E extends Exception> void forEachFloatFragile(@NotNull FragileFloatConsumer<E> action)
          throws E
  {
    for (PrimitiveFloatIterator it = floatIterator(); it.hasNext(); ) {
      action.accept(it.nextFloat());
    }
  }

  /**
   * Does this float iterable contain the given value?
   * This is an operation which is linear to the size of this iterable
   * and should only be used rarely.
   * @param value value to check for
   * @return {@code true} if the value is contained in this iterable, {@code false} if not
   */
  default boolean containsFloat(float value)
  {
    for (PrimitiveFloatIterator iterator = floatIterator();
         iterator.hasNext();  ) {
      if (iterator.nextFloat() == value) {
        return true;
      }
    }
    return false;
  }

  /**
   * Does this float iterable contain the given value?
   * Compared to {@link #containsFloat(float)} this allows for a small deviation.
   * @param value value to check for
   * @param eps   epsilon defining the possible deviation from {@code value} which is still accepted,
   *              using {@code 0.0f} will behave like {@link #containsFloat(float)}
   * @return {@code true} if this iterable contains the value within the given epsilon,
   *         {@code false} if not
   */
  default boolean containsFloat(float value, float eps)
  {
    for (PrimitiveFloatIterator iterator = floatIterator();
         iterator.hasNext();  ) {
      if (Math.abs(iterator.nextFloat() - value) <= eps) {
        return true;
      }
    }
    return false;
  }

  /**
   * View this float iterable as a double iterable.
   * @return double iterable view of this iterable
   */
  @NotNull
  default PrimitiveDoubleIterable asDoubleIterable()
  {
    return () -> floatIterator().asDoubleIterator();
  }
}
