// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.FragileDoubleConsumer;

import java.util.*;
import java.util.function.Consumer;
import java.util.function.DoubleConsumer;
import java.util.stream.DoubleStream;
import java.util.stream.StreamSupport;

/**
 * Iterable which iterates over a primitive double.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 13, 2021
 */
public interface PrimitiveDoubleIterable
        extends Iterable<Double>
{
  /** Empty iterator of primitive double values. */
  PrimitiveIterator.OfDouble EMPTY_DOUBLE_ITERATOR = new PrimitiveIterator.OfDouble()
  {
    @Override
    public double nextDouble()
    {
      throw new NoSuchElementException("No elements in empty iterator!");
    }

    @Override
    public boolean hasNext()
    {
      return false;
    }

    @Override
    public void forEachRemaining(DoubleConsumer action)
    {
    }

    @Override
    public void forEachRemaining(Consumer<? super Double> action)
    {
    }
  };
  /** Empty iterable of primitive double values. */
  PrimitiveDoubleIterable EMPTY = () -> EMPTY_DOUBLE_ITERATOR;

  /**
   * Get a primitive double iterator.
   * @return double iterator
   */
  @NotNull
  PrimitiveIterator.OfDouble doubleIterator();

  @Override
  @NotNull
  default Iterator<Double> iterator()
  {
    return doubleIterator();
  }

  /**
   * Invoke the given action with each double value in this iterable.
   * @param action action to be call for each double value in sequemce.
   */
  default void forEachDouble(@NotNull DoubleConsumer action)
  {
    for (PrimitiveIterator.OfDouble it = doubleIterator();  it.hasNext(); ) {
      action.accept(it.nextDouble());
    }
  }

  /**
   * Invoke the given fragile action with each value in this iterable.
   * @param action action to be called for each value in sequence
   * @param <E> checked exception thrown by action
   * @throws E if {@code action} throws it
   */
  default <E extends Exception> void forEachDoubleFragile(@NotNull FragileDoubleConsumer<E> action)
          throws E
  {
    for (PrimitiveIterator.OfDouble it = doubleIterator();  it.hasNext(); ) {
      action.accept(it.nextDouble());
    }
  }

  /**
   * Does this double iterable contain the given value?
   * This is an operation which is linear to the size of this iterable
   * and should only be used rarely.
   * @param value value to check for
   * @return {@code true} if the value is contained in this iterable, {@code false} if not
   */
  default boolean containsDouble(double value)
  {
    for (PrimitiveIterator.OfDouble iterator = doubleIterator();
         iterator.hasNext();  ) {
      if (iterator.nextDouble() == value) {
        return true;
      }
    }
    return false;
  }

  /**
   * Does this float iterable contain the given value?
   * Compared to {@link #containsDouble(double)} this allows for a small deviation.
   * @param value value to check for
   * @param eps   epsilon defining the possible deviation from {@code value} which is still accepted
   * @return {@code true} if this indexable contains the value within the given epsilon,
   *         {@code false} if not
   */
  default boolean containsDouble(double value, double eps)
  {
    for (PrimitiveIterator.OfDouble iterator = doubleIterator();
         iterator.hasNext();  ) {
      if (Math.abs(iterator.nextDouble() - value) <= eps) {
        return true;
      }
    }
    return false;
  }

  /**
   * Calculate the sum of the elements of this iterable.
   * @return sum of all elements of this iterable
   */
  default double sum()
  {
    double s = 0;
    for (PrimitiveIterator.OfDouble it = doubleIterator();  it.hasNext();  ) {
      s += it.nextDouble();
    }
    return s;
  }

  /**
   * Calculate the average of all elements in this iterable.
   * @return average
   */
  @NotNull
  default OptionalDouble average()
  {
    int count = 0;
    double average = 0.0;
    for (PrimitiveIterator.OfDouble it = doubleIterator();  it.hasNext();  ) {
      average = (average * count + it.nextDouble()) / ++count;
    }
    return count == 0 ? OptionalDouble.empty() : OptionalDouble.of(average);
  }

  @Override
  default Spliterator.OfDouble spliterator()
  {
    return Spliterators.spliteratorUnknownSize(doubleIterator(), Spliterator.ORDERED);
  }

  /**
   * Get a stream of primitive long values from this iterable.
   * @return {@code double} stream
   */
  @NotNull
  default DoubleStream stream()
  {
    return StreamSupport.doubleStream(spliterator(), false);
  }
}
