// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.CharConsumer;
import de.caff.generics.function.FragileCharConsumer;

import java.util.Iterator;

/**
 * Iterable which iterates over a primitive char.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 13, 2021
 * @see CharIndexable for creating one on the fly
 */
public interface PrimitiveCharIterable
        extends Iterable<Character>
{
  /**
   * Empty iterator of primitive char values.
   * @see PrimitiveCharIterator#EMPTY
   */
  PrimitiveCharIterator EMPTY_CHAR_ITERATOR = PrimitiveCharIterator.EMPTY;
  /** Empty iterable of primitive char values. */
  PrimitiveCharIterable EMPTY = () -> EMPTY_CHAR_ITERATOR;

  /**
   * Get a primitive char iterator.
   * @return char iterator
   */
  @NotNull
  PrimitiveCharIterator charIterator();

  @Override
  @NotNull
  default Iterator<Character> iterator()
  {
    return charIterator();
  }

  /**
   * Invoke the given action with each char value in this iterable.
   * @param action action to be call for each char value in sequemce.
   */
  default void forEachChar(@NotNull CharConsumer action)
  {
    for (PrimitiveCharIterator it = charIterator();  it.hasNext(); ) {
      action.accept(it.nextChar());
    }
  }

  /**
   * Invoke the given fragile action with each value in this iterable.
   * @param action action to be called for each value in sequence
   * @param <E> checked exception thrown by action
   * @throws E if {@code action} throws it
   */
  default <E extends Exception> void forEachCharFragile(@NotNull FragileCharConsumer<E> action)
          throws E
  {
    for (PrimitiveCharIterator it = charIterator(); it.hasNext(); ) {
      action.accept(it.nextChar());
    }
  }

  /**
   * Does this char iterable contain the given value?
   * This is an operation which is linear to the size of this iterable
   * and should only be used rarely.
   * @param value value to check for
   * @return {@code true} if the value is contained in this iterable, {@code false} if not
   */
  default boolean containsChar(char value)
  {
    for (PrimitiveCharIterator iterator = charIterator();
         iterator.hasNext();  ) {
      if (iterator.nextChar() == value) {
        return true;
      }
    }
    return false;
  }
  
  /**
   * View this byte iterable as an int iterable.
   * @return int iterable view of this byte iterable
   */
  @NotNull
  default PrimitiveIntIterable asIntIterable()
  {
    return () -> charIterator().asIntIterator();
  }

  /**
   * View this byte iterable as a long iterable.
   * @return long iterable view of this byte iterable
   */
  @NotNull
  default PrimitiveLongIterable asLongIterable()
  {
    return () -> charIterator().asLongIterator();
  }
}
