// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.ByteConsumer;
import de.caff.generics.function.FragileByteConsumer;

import java.util.Iterator;

/**
 * Iterable which iterates over a primitive byte.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 13, 2021
 * @see ByteIndexable for creating one on the fly
 */
public interface PrimitiveByteIterable
        extends Iterable<Byte>
{
  /**
   * Empty iterator of primitive byte values.
   * @see PrimitiveByteIterator#EMPTY
   */
  PrimitiveByteIterator EMPTY_BYTE_ITERATOR = PrimitiveByteIterator.EMPTY;
  /** Empty iterable of primitive byte values. */
  PrimitiveByteIterable EMPTY = () -> EMPTY_BYTE_ITERATOR;

  /**
   * Get a primitive byte iterator.
   * @return byte iterator
   */
  @NotNull
  PrimitiveByteIterator byteIterator();

  @Override
  @NotNull
  default Iterator<Byte> iterator()
  {
    return byteIterator();
  }

  /**
   * Invoke the given action with each byte value in this iterable.
   * @param action action to be call for each byte value in sequemce.
   */
  default void forEachByte(@NotNull ByteConsumer action)
  {
    for (PrimitiveByteIterator it = byteIterator();  it.hasNext(); ) {
      action.accept(it.nextByte());
    }
  }

  /**
   * Invoke the given fragile action with each value in this iterable.
   * @param action action to be called for each value in sequence
   * @param <E> checked exception thrown by action
   * @throws E if {@code action} throws it
   */
  default <E extends Exception> void forEachByteFragile(@NotNull FragileByteConsumer<E> action)
          throws E
  {
    for (PrimitiveByteIterator it = byteIterator(); it.hasNext(); ) {
      action.accept(it.nextByte());
    }
  }

  /**
   * Does this byte iterable contain the given value?
   * This is an operation which takes time linear to the size of this iterable
   * and should only be used rarely.
   * @param value value to check for
   * @return {@code true} if the value is contained in this iterable, {@code false} if not
   */
  default boolean containsByte(byte value)
  {
    for (PrimitiveByteIterator iterator = byteIterator();
         iterator.hasNext();  ) {
      if (iterator.nextByte() == value) {
        return true;
      }
    }
    return false;
  }

  /**
   * View this byte iterable as a short iterable.
   * @return short iterable view of this byte iterable
   */
  @NotNull
  default PrimitiveShortIterable asShortIterable()
  {
    return () -> byteIterator().asShortIterator();
  }

  /**
   * View this byte iterable as an int iterable.
   * @return int iterable view of this byte iterable
   */
  @NotNull
  default PrimitiveIntIterable asIntIterable()
  {
    return () -> byteIterator().asIntIterator();
  }

  /**
   * View this byte iterable as an int iterable which uses the unsigned byte values
   * of this iterable.
   * @return unsigned int iterable view of this byte iterable
   */
  @NotNull
  default PrimitiveIntIterable asUnsignedIntIterable()
  {
    return () -> byteIterator().asUnsignedIntIterator();
  }

  /**
   * View this byte iterable as a long iterable.
   * @return long iterable view of this byte iterable
   */
  @NotNull
  default PrimitiveLongIterable asLongIterable()
  {
    return () -> byteIterator().asLongIterator();
  }

  /**
   * View this byte iterable as a long iterable which uses the unsigned byte values
   * of this iterable.
   * @return unsigned long iterable view of this byte iterable
   */
  @NotNull
  default PrimitiveLongIterable asUnsignedLongIterable()
  {
    return () -> byteIterator().asUnsignedLongIterator();
  }
}
